"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabContainerRunner = exports.InstanceInterruptionBehavior = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const ssm = require("aws-cdk-lib/aws-ssm");
const cr = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
/**
 * InstanceInterruptionBehavior enum.
 */
var InstanceInterruptionBehavior;
(function (InstanceInterruptionBehavior) {
    /**
     * hibernate
     */
    InstanceInterruptionBehavior["HIBERNATE"] = "hibernate";
    /**
     * stop
     */
    InstanceInterruptionBehavior["STOP"] = "stop";
    /**
     * terminate
     */
    InstanceInterruptionBehavior["TERMINATE"] = "terminate";
})(InstanceInterruptionBehavior = exports.InstanceInterruptionBehavior || (exports.InstanceInterruptionBehavior = {}));
/**
 * GitlabContainerRunner Construct for create a Gitlab Runner.
 */
class GitlabContainerRunner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const spotFleetId = id;
        const defaultProps = {
            gitlabRunnerImage: 'public.ecr.aws/gitlab/gitlab-runner:alpine',
            gitlaburl: 'https://gitlab.com/',
            ec2type: 't3.micro',
            tags: ['gitlab', 'awscdk', 'runner'],
        };
        const runnerProps = { ...defaultProps, ...props };
        const tokenParameterStore = new ssm.StringParameter(this, 'GitlabTokenParameter', {
            stringValue: 'GITLAB_TOKEN',
        });
        const shell = aws_ec2_1.UserData.forLinux();
        shell.addCommands(...this.createUserData(runnerProps, tokenParameterStore.parameterName));
        this.runnerRole =
            runnerProps.ec2iamrole ??
                new aws_iam_1.Role(this, 'runner-role', {
                    assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                    description: 'For Gitlab EC2 Runner Role',
                });
        this.validUntil = runnerProps.validUntil;
        const instanceProfile = new aws_iam_1.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.runnerRole.roleName],
        });
        tokenParameterStore.grantWrite(this.runnerRole);
        tokenParameterStore.grantRead(this.runnerRole);
        this.vpc =
            runnerProps.selfvpc ??
                new aws_ec2_1.Vpc(this, 'VPC', {
                    cidr: '10.0.0.0/16',
                    maxAzs: 2,
                    subnetConfiguration: [
                        {
                            cidrMask: 26,
                            name: 'RunnerVPC',
                            subnetType: aws_ec2_1.SubnetType.PUBLIC,
                        },
                    ],
                    natGateways: 0,
                });
        this.defaultRunnerSG = new aws_ec2_1.SecurityGroup(this, 'SpotFleetSg', {
            vpc: this.vpc,
        });
        this.defaultRunnerSG.connections.allowFromAnyIpv4(aws_ec2_1.Port.tcp(22));
        const spotOrOnDemand = runnerProps.spotFleet ?? false;
        if (spotOrOnDemand) {
            //throw new Error('yes new spotfleet');
            const imageId = aws_ec2_1.MachineImage.latestAmazonLinux({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }).getImage(this).imageId;
            const lt = new aws_ec2_1.CfnLaunchTemplate(this, 'LaunchTemplate', {
                launchTemplateData: {
                    imageId,
                    instanceType: runnerProps.ec2type,
                    blockDeviceMappings: [
                        {
                            deviceName: '/dev/xvda',
                            ebs: {
                                volumeSize: runnerProps.ebsSize ?? 60,
                            },
                        },
                    ],
                    userData: aws_cdk_lib_1.Fn.base64(shell.render()),
                    keyName: runnerProps.keyName,
                    tagSpecifications: [
                        {
                            resourceType: 'instance',
                            tags: [
                                {
                                    key: 'Name',
                                    value: `${aws_cdk_lib_1.Stack.of(this).stackName}/spotFleetGitlabRunner/${spotFleetId}`,
                                },
                            ],
                        },
                    ],
                    instanceMarketOptions: {
                        marketType: 'spot',
                        spotOptions: {
                            instanceInterruptionBehavior: runnerProps.instanceInterruptionBehavior ??
                                InstanceInterruptionBehavior.TERMINATE,
                        },
                    },
                    securityGroupIds: this.defaultRunnerSG.connections.securityGroups.map((m) => m.securityGroupId),
                    iamInstanceProfile: {
                        arn: instanceProfile.attrArn,
                    },
                },
            });
            const spotFleetRole = new aws_iam_1.Role(this, 'FleetRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
                ],
            });
            const vpcSubnetSelection = runnerProps.vpcSubnet ?? {
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            };
            const subnetConfig = this.vpc
                .selectSubnets(vpcSubnetSelection)
                .subnets.map((s) => ({
                subnetId: s.subnetId,
            }));
            this.cfnSpotFleet = new aws_ec2_1.CfnSpotFleet(this, id, {
                spotFleetRequestConfigData: {
                    launchTemplateConfigs: [
                        {
                            launchTemplateSpecification: {
                                launchTemplateId: lt.ref,
                                version: lt.attrLatestVersionNumber,
                            },
                            overrides: subnetConfig,
                        },
                    ],
                    iamFleetRole: spotFleetRole.roleArn,
                    targetCapacity: 1,
                    validUntil: aws_cdk_lib_1.Lazy.string({ produce: () => this.validUntil }),
                    terminateInstancesWithExpiration: true,
                },
            });
            const onEvent = new lambda.Function(this, 'OnEvent', {
                code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
                handler: 'index.on_event',
                runtime: lambda.Runtime.PYTHON_3_8,
                timeout: aws_cdk_lib_1.Duration.seconds(60),
            });
            const isComplete = new lambda.Function(this, 'IsComplete', {
                code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
                handler: 'index.is_complete',
                runtime: lambda.Runtime.PYTHON_3_8,
                timeout: aws_cdk_lib_1.Duration.seconds(60),
                role: onEvent.role,
            });
            const myProvider = new cr.Provider(this, 'MyProvider', {
                onEventHandler: onEvent,
                isCompleteHandler: isComplete,
                logRetention: logs.RetentionDays.ONE_DAY,
            });
            onEvent.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ['ec2:DescribeSpotFleetInstances'],
                resources: ['*'],
            }));
            const fleetInstancesId = new aws_cdk_lib_1.CustomResource(this, 'GetInstanceId', {
                serviceToken: myProvider.serviceToken,
                properties: {
                    SpotFleetRequestId: this.cfnSpotFleet.ref,
                },
            });
            fleetInstancesId.node.addDependency(this.cfnSpotFleet);
            this.spotFleetInstanceId = aws_cdk_lib_1.Token.asString(fleetInstancesId.getAtt('InstanceId'));
            this.spotFleetRequestId = aws_cdk_lib_1.Token.asString(fleetInstancesId.getAtt('SpotInstanceRequestId'));
            new aws_cdk_lib_1.CfnOutput(this, 'InstanceId', { value: this.spotFleetInstanceId });
            new aws_cdk_lib_1.CfnOutput(this, 'SpotFleetId', { value: this.cfnSpotFleet.ref });
        }
        else {
            this.runnerEc2 = new aws_ec2_1.Instance(this, 'GitlabRunner', {
                instanceType: new aws_ec2_1.InstanceType(runnerProps.ec2type),
                instanceName: 'Gitlab-Runner',
                vpc: this.vpc,
                vpcSubnets: runnerProps.vpcSubnet ?? {
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                },
                machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({
                    generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
                }),
                role: this.runnerRole,
                userData: shell,
                securityGroup: this.defaultRunnerSG,
                blockDevices: [
                    {
                        deviceName: '/dev/xvda',
                        volume: aws_ec2_1.BlockDeviceVolume.ebs(runnerProps.ebsSize ?? 60),
                    },
                ],
            });
            new aws_cdk_lib_1.CfnOutput(this, 'Runner-Instance-ID', {
                value: this.runnerEc2.instanceId,
            });
        }
        const unregisterRunnerOnEvent = new lambda.Function(this, 'unregisterRunnerOnEvent', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'unregister_runner.on_event',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: aws_cdk_lib_1.Duration.seconds(60),
        });
        const unregisterRunnerProvider = new cr.Provider(this, 'unregisterRunnerProvider', {
            onEventHandler: unregisterRunnerOnEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        const unregisterRunnerCR = new aws_cdk_lib_1.CustomResource(this, 'unregisterRunnerCR', {
            resourceType: 'Custom::unregisterRunnerProvider',
            serviceToken: unregisterRunnerProvider.serviceToken,
            properties: {
                TokenParameterStoreName: tokenParameterStore.parameterName,
                GitlabUrl: runnerProps.gitlaburl,
            },
        });
        tokenParameterStore.grantRead(unregisterRunnerOnEvent);
        unregisterRunnerCR.node.addDependency(tokenParameterStore);
        this.runnerRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        new aws_cdk_lib_1.CfnOutput(this, 'Runner-Role-Arn', {
            value: this.runnerRole.roleArn,
        });
    }
    /**
     * Add expire time function for spotfleet runner !!! .
     *
     * @param duration - Block duration.
     */
    expireAfter(duration) {
        const date = new Date();
        date.setSeconds(date.getSeconds() + duration.toSeconds());
        this.validUntil = date.toISOString();
    }
    dockerVolumesList(dockerVolume) {
        let tempString = '--docker-volumes "/var/run/docker.sock:/var/run/docker.sock"';
        if (dockerVolume) {
            let tempList = [];
            dockerVolume.forEach(e => {
                tempList.push(`"${e.hostPath}:${e.containerPath}"`);
            });
            tempList.forEach(e => {
                tempString = `${tempString} --docker-volumes ${e}`;
            });
        }
        return tempString;
    }
    /**
     * @param props
     * @param tokenParameterStoreName - the tokenParameterStoreName to put gitlab runner token.
     * @returns Array.
     */
    createUserData(props, tokenParameterStoreName) {
        return [
            'yum update -y ',
            'sleep 15 && amazon-linux-extras install docker && yum install -y amazon-cloudwatch-agent && systemctl start docker && usermod -aG docker ec2-user && chmod 777 /var/run/docker.sock',
            'systemctl restart docker && systemctl enable docker',
            `docker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock \
      --name gitlab-runner-register ${props.gitlabRunnerImage} register --non-interactive --url ${props.gitlaburl} --registration-token ${props.gitlabtoken} \
      --docker-pull-policy if-not-present ${this.dockerVolumesList(props?.dockerVolumes)} \
      --executor docker --docker-image "alpine:latest" --description "Docker Runner" \
      --tag-list "${props.tags?.join(',')}" --docker-privileged`,
            `sleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner ${props.gitlabRunnerImage}`,
            'TOKEN=$(cat /home/ec2-user/.gitlab-runner/config.toml | grep token | awk \'{print $3}\'| tr -d \'"\')',
            `aws ssm put-parameter --name ${tokenParameterStoreName} --value $TOKEN --overwrite --region ${aws_cdk_lib_1.Stack.of(this).region}`,
        ];
    }
}
exports.GitlabContainerRunner = GitlabContainerRunner;
_a = JSII_RTTI_SYMBOL_1;
GitlabContainerRunner[_a] = { fqn: "cdk-gitlab-runner.GitlabContainerRunner", version: "2.0.271" };
//# sourceMappingURL=data:application/json;base64,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