"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sns = require("@aws-cdk/aws-sns");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
test('Publish literal message to SNS topic', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const topic = new sns.Topic(stack, 'Topic');
    // WHEN
    const pub = new sfn.Task(stack, 'Publish', { task: new tasks.PublishToTopic(topic, {
            message: sfn.TaskInput.fromText('Publish this message'),
        }) });
    // THEN
    expect(stack.resolve(pub.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sns:publish',
                ],
            ],
        },
        End: true,
        Parameters: {
            TopicArn: { Ref: 'TopicBFC7AF6E' },
            Message: 'Publish this message',
        },
    });
});
test('Publish JSON to SNS topic with task token', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const topic = new sns.Topic(stack, 'Topic');
    // WHEN
    const pub = new sfn.Task(stack, 'Publish', { task: new tasks.PublishToTopic(topic, {
            integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            message: sfn.TaskInput.fromObject({
                Input: 'Publish this message',
                Token: sfn.Context.taskToken,
            }),
        }) });
    // THEN
    expect(stack.resolve(pub.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sns:publish.waitForTaskToken',
                ],
            ],
        },
        End: true,
        Parameters: {
            TopicArn: { Ref: 'TopicBFC7AF6E' },
            Message: {
                'Input': 'Publish this message',
                'Token.$': '$$.Task.Token',
            },
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied but task token is not included in message', () => {
    expect(() => {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new sfn.Task(stack, 'Publish', { task: new tasks.PublishToTopic(topic, {
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
                message: sfn.TaskInput.fromText('Publish this message'),
            }) });
        // THEN
    }).toThrow(/Task Token is missing in message/i);
});
test('Publish to topic with ARN from payload', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const topic = sns.Topic.fromTopicArn(stack, 'Topic', sfn.Data.stringAt('$.topicArn'));
    // WHEN
    const pub = new sfn.Task(stack, 'Publish', { task: new tasks.PublishToTopic(topic, {
            message: sfn.TaskInput.fromText('Publish this message'),
        }) });
    // THEN
    expect(stack.resolve(pub.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sns:publish',
                ],
            ],
        },
        End: true,
        Parameters: {
            'TopicArn.$': '$.topicArn',
            'Message': 'Publish this message',
        },
    });
});
test('Task throws if SYNC is supplied as service integration pattern', () => {
    expect(() => {
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        new sfn.Task(stack, 'Publish', { task: new tasks.PublishToTopic(topic, {
                integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
                message: sfn.TaskInput.fromText('Publish this message'),
            }) });
    }).toThrow(/Invalid Service Integration Pattern: SYNC is not supported to call SNS./i);
});
//# sourceMappingURL=data:application/json;base64,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