"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const path = require("path");
const tasks = require("../../lib");
const app = new cdk.App();
const stack = new cdk.Stack(app, 'aws-stepfunctions-integ');
const handler = new aws_lambda_1.Function(stack, 'Handler', {
    code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
    handler: 'index.main',
    runtime: aws_lambda_1.Runtime.PYTHON_3_6,
});
const submitJob = new sfn.Task(stack, 'Invoke Handler', {
    task: new tasks.InvokeFunction(handler),
});
const callbackHandler = new aws_lambda_1.Function(stack, 'CallbackHandler', {
    code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
    handler: 'index.main',
    runtime: aws_lambda_1.Runtime.PYTHON_3_6,
});
const taskTokenHandler = new sfn.Task(stack, 'Invoke Handler with task token', {
    task: new tasks.RunLambdaTask(callbackHandler, {
        integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
        payload: {
            token: sfn.Context.taskToken,
        },
    }),
    inputPath: '$.guid',
    resultPath: '$.status',
});
const isComplete = new sfn.Choice(stack, 'Job Complete?');
const jobFailed = new sfn.Fail(stack, 'Job Failed', {
    cause: 'AWS Batch Job Failed',
    error: 'DescribeJob returned FAILED',
});
const finalStatus = new sfn.Pass(stack, 'Final step');
const chain = sfn.Chain
    .start(submitJob)
    .next(taskTokenHandler)
    .next(isComplete
    .when(sfn.Condition.stringEquals('$.status', 'FAILED'), jobFailed)
    .when(sfn.Condition.stringEquals('$.status', 'SUCCEEDED'), finalStatus));
new sfn.StateMachine(stack, 'StateMachine', {
    definition: chain,
    timeout: cdk.Duration.seconds(30),
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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