"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const sfn = require("@aws-cdk/aws-stepfunctions");
const path = require("path");
/**
 * A Step Functions Task to evaluate an expression
 *
 * OUTPUT: the output of this task is the evaluated expression.
 *
 * @experimental
 */
class EvaluateExpression {
    constructor(props) {
        this.props = props;
    }
    bind(task) {
        const matches = this.props.expression.match(/\$[.\[][.a-zA-Z[\]0-9]+/g);
        if (!matches) {
            throw new Error('No paths found in expression');
        }
        const expressionAttributeValues = matches.reduce((acc, m) => ({
            ...acc,
            [m]: sfn.Data.stringAt(m),
        }), {});
        const evalFn = createEvalFn(this.props.runtime || lambda.Runtime.NODEJS_10_X, task);
        return {
            resourceArn: evalFn.functionArn,
            policyStatements: [new iam.PolicyStatement({
                    resources: [evalFn.functionArn],
                    actions: ['lambda:InvokeFunction'],
                })],
            parameters: {
                expression: this.props.expression,
                expressionAttributeValues,
            },
        };
    }
}
exports.EvaluateExpression = EvaluateExpression;
function createEvalFn(runtime, scope) {
    const code = lambda.Code.asset(path.join(__dirname, `eval-${runtime.name}-handler`));
    const lambdaPurpose = 'Eval';
    switch (runtime) {
        case lambda.Runtime.NODEJS_10_X:
            return new lambda.SingletonFunction(scope, 'EvalFunction', {
                runtime,
                handler: 'index.handler',
                uuid: 'a0d2ce44-871b-4e74-87a1-f5e63d7c3bdc',
                lambdaPurpose,
                code,
            });
        // TODO: implement other runtimes
        default:
            throw new Error(`The runtime ${runtime.name} is currently not supported.`);
    }
}
//# sourceMappingURL=data:application/json;base64,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