"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SSMParameter = exports.SSMParameterType = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const custom = require("@aws-cdk/custom-resources");
/**
 * The SSM Parameter type.
 *
 * SecureString is atm not supported
 *
 * @stability stable
 */
var SSMParameterType;
(function (SSMParameterType) {
    SSMParameterType["STRING"] = "String";
    SSMParameterType["STRING_LIST"] = "StringList";
})(SSMParameterType = exports.SSMParameterType || (exports.SSMParameterType = {}));
/**
 * @stability stable
 */
class SSMParameter extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(parent, name, props) {
        super(parent, name);
        if (!props.parameterName) {
            throw new Error('parameterName cannot be an empty string');
        }
        if (props.parameterName.length > 2048) {
            throw new Error('Name cannot be longer than 2048 characters.');
        }
        this.parameterName = props.parameterName;
        const putParameter = new custom.AwsCustomResource(this, 'PutParameter', {
            onUpdate: {
                service: 'SSM',
                action: 'putParameter',
                parameters: {
                    Name: props.parameterName,
                    Value: props.defaultValue || '',
                    Type: props.type || SSMParameterType.STRING,
                },
                // ignore if ParameterAlreadyExists as we don't override anyway
                ignoreErrorCodesMatching: '.*',
                physicalResourceId: custom.PhysicalResourceId.of(Date.now().toString()),
            },
            role: new iam.Role(this, 'putParameterRole', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
                    iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMFullAccess'),
                ],
            }),
            policy: custom.AwsCustomResourcePolicy.fromSdkCalls({ resources: custom.AwsCustomResourcePolicy.ANY_RESOURCE }),
        });
        const getParameter = new custom.AwsCustomResource(this, 'GetParameter', {
            onUpdate: {
                service: 'SSM',
                action: 'getParameter',
                parameters: {
                    Name: props.parameterName,
                },
                physicalResourceId: custom.PhysicalResourceId.of(Date.now().toString()),
            },
            role: new iam.Role(this, 'getParameterRole', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
                    iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMReadOnlyAccess'),
                ],
            }),
            policy: custom.AwsCustomResourcePolicy.fromSdkCalls({ resources: custom.AwsCustomResourcePolicy.ANY_RESOURCE }),
        });
        this.parameterValue = getParameter.getResponseField('Parameter.Value');
        // Run the get ssm parameter after put parameter
        getParameter.node.addDependency(putParameter);
        if (props.delete) {
            new custom.AwsCustomResource(this, 'DeleteParameter', {
                onDelete: {
                    service: 'SSM',
                    action: 'deleteParameter',
                    parameters: {
                        Name: props.parameterName,
                    },
                    physicalResourceId: custom.PhysicalResourceId.of(Date.now().toString()),
                },
                role: new iam.Role(this, 'deleteParameterRole', {
                    assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
                    managedPolicies: [
                        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
                        iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMFullAccess'),
                    ],
                }),
                policy: custom.AwsCustomResourcePolicy.fromSdkCalls({ resources: custom.AwsCustomResourcePolicy.ANY_RESOURCE }),
            });
        }
        new cdk.CfnOutput(this, 'SSMParameterValue', {
            value: this.parameterValue,
        });
        new cdk.CfnOutput(this, 'SSMParameterName', {
            value: this.parameterName,
        });
    }
}
exports.SSMParameter = SSMParameter;
//# sourceMappingURL=data:application/json;base64,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