# AUTOGENERATED! DO NOT EDIT! File to edit: ../dataset_torch.ipynb.

# %% auto 0
__all__ = ['SequentialDataset', 'load_dataloaders']

# %% ../dataset_torch.ipynb 2
import torch
import recsys_slates_dataset.data_helper as data_helper
from torch.utils.data import Dataset, DataLoader
import torch
import json
import numpy as np
import logging
logging.basicConfig(format='%(asctime)s %(message)s', level='INFO')

class SequentialDataset(Dataset):
    ''' A Pytorch Dataset for the FINN Recsys Slates Dataset.
    Attributes:
      data: [Dict] A dictionary with tensors of the dataset. First dimension in each tensor must be the batch dimension. Requires the keys "click" and "slate". Additional elements can be added.
      sample_candidate_items: [int] Number of negative item examples sampled from the item universe for each interaction. If positive, the dataset provide an additional dictionary item "allitem". Often also called uniform candidate sampling. See Eide et. al. 2021 for more information.
    '''
    def __init__(self, data, sample_candidate_items=0):

        self.data = data
        self.num_items = self.data['slate'].max()+1
        self.sample_candidate_items = sample_candidate_items
        self.mask2ind = {'train' : 1, 'valid' : 2, 'test' : 3}

        logging.info(
            "Loading dataset with slate size={} and number of negative samples={}"
            .format(self.data['slate'].size(), self.sample_candidate_items))

        # Performs some checks on the dataset to make sure it is valid:
        assert "slate" in data.keys(), "Slate tensor is not in dataset. This is required."
        assert "click" in data.keys(), "Click tensor is not in dataset. This is required."
        assert all([val.size(0)==data['slate'].size(0) for key, val in data.items()]), "Not all data tensors have the same batch dimension"

    def __getitem__(self, idx):
        batch = {key: val[idx] for key, val in self.data.items()}

        if self.sample_candidate_items:
            # Sample actions uniformly (3 is the first non-special item)
            batch['allitem'] = torch.randint(
                size=(batch['click'].size(0), self.sample_candidate_items),
                low=3, high=self.num_items, device = batch['click'].device
                )

        return batch

    def __len__(self):
        return len(self.data['click'])

# %% ../dataset_torch.ipynb 3
def load_dataloaders(data_dir= "dat",
                     batch_size=1024,
                     num_workers= 0,
                     sample_candidate_items=False,
                     valid_pct= 0.05,
                     test_pct= 0.05,
                     t_testsplit= 5,
                     limit_num_users=None,
                     seed=0):
    """
    Loads pytorch dataloaders to be used in training. If used with standard settings, the train/val/test split is equivalent to Eide et. al. 2021.

    Attributes:
      data_dir: [str] where download and store data if not already downloaded.
      batch_size: [int] Batch size given by dataloaders.
      num_workers: [int] How many threads should be used to prepare batches of data.
      sample_candidate_items: [int] Number of negative item examples sampled from the item universe for each interaction. If positive, the dataset provide an additional dictionary item "allitem". Often also called uniform candidate sampling. See Eide et. al. 2021 for more information.
      valid_pct: [float] Percentage of users allocated to validation dataset.
      test_pct: [float] Percentage of users allocated to test dataset.
      t_testsplit: [int] For users allocated to validation and test datasets, how many initial interactions should be part of the training dataset.
      limit_num_users: [int] For debugging purposes, only return some users.
      seed: [int] Seed used to sample users/items.

    """

    logging.info("Download data if not in data folder..")
    data_helper.download_data_files(data_dir=data_dir)

    logging.info('Load data..')
    with np.load("{}/data.npz".format(data_dir)) as data_np:
        data = {key: torch.tensor(val) for key, val in data_np.items()}

    if limit_num_users is not None:
        logging.info("Limiting dataset to only return the first {} users.".format(limit_num_users))
        data = {key : val[:limit_num_users] for key, val in data.items()}

    with open('{}/ind2val.json'.format(data_dir), 'rb') as handle:
        # Use string2int object_hook found here: https://stackoverflow.com/a/54112705
        ind2val = json.load(
            handle,
            object_hook=lambda d: {
                int(k) if k.lstrip('-').isdigit() else k: v
                for k, v in d.items()
                }
            )

    num_users = len(data['click'])
    num_validusers = int(num_users * valid_pct)
    num_testusers = int(num_users * test_pct)
    torch.manual_seed(seed)
    perm_user = torch.randperm(num_users)
    valid_user_idx = perm_user[:num_validusers]
    test_user_idx  = perm_user[num_validusers:(num_validusers+num_testusers)]
    train_user_idx = perm_user[(num_validusers+num_testusers):]

    # Split dictionary into train/valid/test with a phase mask that shows which interactions are in different sets
    # (as some users have both train and valid data)
    data_train = data
    data_train['phase_mask'] = torch.ones_like(data['click']).bool()
    data_train['phase_mask'][test_user_idx,t_testsplit:]=False
    data_train['phase_mask'][valid_user_idx,t_testsplit:]=False

    data_valid = {key: val[valid_user_idx] for key, val in data.items()}
    data_valid['phase_mask'] = torch.zeros_like(data_valid['click']).bool()
    data_valid['phase_mask'][:,t_testsplit:] = True

    data_test = {key: val[test_user_idx] for key, val in data.items()}
    data_test['phase_mask'] = torch.zeros_like(data_test['click']).bool()
    data_test['phase_mask'][:,t_testsplit:] = True

    data_dicts = {
        "train" : data_train,
        "valid" : data_valid,
        "test" : data_test}

    datasets = {
        phase : SequentialDataset(data, sample_candidate_items)
        for phase, data in data_dicts.items()
        }


    # Build dataloaders for each data subset:
    dataloaders = {
        phase: DataLoader(ds, batch_size=batch_size, shuffle=(phase=="train"), num_workers=num_workers)
        for phase, ds in datasets.items()
    }
    for key, dl in dataloaders.items():
        logging.info(
            "In {}: num_users: {}, num_batches: {}".format(key, len(dl.dataset), len(dl))
        )

    # Load item attributes:
    with np.load('{}/itemattr.npz'.format(data_dir), mmap_mode=None) as itemattr_file:
        itemattr = {key : val for key, val in itemattr_file.items()}

    return ind2val, itemattr, dataloaders
