import os
from typing import BinaryIO, Dict, Optional, Tuple, Union

import click
import requests
from requests import HTTPError, Session, Timeout

from launchable.utils.http_client import _HttpClient, _join_paths
from launchable.utils.tracking import Tracking, TrackingClient  # type: ignore

from ..app import Application
from .authentication import get_org_workspace
from .env_keys import REPORT_ERROR_KEY


class LaunchableClient:
    def __init__(self, tracking_client: Optional[TrackingClient] = None, base_url: str = "", session: Optional[Session] = None,
                 test_runner: Optional[str] = "", app: Optional[Application] = None):
        self.http_client = _HttpClient(
            base_url=base_url,
            session=session,
            test_runner=test_runner,
            app=app
        )
        self.tracking_client = tracking_client
        self.organization, self.workspace = get_org_workspace()
        if self.organization is None or self.workspace is None:
            raise ValueError(
                "Could not identify a Launchable organization/workspace. "
                "Confirm that you set LAUNCHABLE_TOKEN "
                "(or LAUNCHABLE_ORGANIZATION and LAUNCHABLE_WORKSPACE) environment variable(s)\n"
                "See https://docs.launchableinc.com/getting-started#setting-your-api-key")
        self._workspace_state_cache: dict = {}

    def request(
        self,
        method: str,
        sub_path: str,
        payload: Optional[Union[Dict, BinaryIO]] = None,
        params: Optional[Dict] = None,
        timeout: Tuple[int, int] = (5, 60),
        compress: bool = False,
        additional_headers: Optional[Dict] = None,
    ) -> requests.Response:
        path = _join_paths(
            "/intake/organizations/{}/workspaces/{}".format(self.organization, self.workspace),
            sub_path
        )

        # report an error and bail out
        def track(event_name: Tracking.ErrorEvent, e: Exception):
            if self.tracking_client:
                self.tracking_client.send_error_event(
                    event_name=event_name,
                    stack_trace=str(e),
                    api=sub_path,
                )
            raise e

        try:
            response = self.http_client.request(
                method=method,
                path=path,
                payload=payload,
                params=params,
                timeout=timeout,
                compress=compress,
                additional_headers=additional_headers
            )
            return response
        except ConnectionError as e:
            track(Tracking.ErrorEvent.NETWORK_ERROR, e)
        except Timeout as e:
            track(Tracking.ErrorEvent.TIMEOUT_ERROR, e)
        except HTTPError as e:
            track(Tracking.ErrorEvent.UNEXPECTED_HTTP_STATUS_ERROR, e)
        except Exception as e:
            track(Tracking.ErrorEvent.INTERNAL_SERVER_ERROR, e)

        # should never come here, but needed to make type checker happy
        assert False

    def print_exception_and_recover(self, e: Exception, warning: Optional[str] = None, warning_color='yellow'):
        """
        Print the exception raised from the request method, then recover from it

        :param warning: optional warning message to contextualize the HTTP error
        """

        # a diagnostics flag to abort and report the details
        if os.getenv(REPORT_ERROR_KEY):
            raise e

        click.echo(e, err=True)
        if isinstance(e, HTTPError):
            # if the payload is present, report that as well to assist troubleshooting
            res = e.response
            if res and res.text:
                click.echo(res.text, err=True)

        if warning:
            click.echo(click.style(warning, fg=warning_color), err=True)

    def is_fail_fast_mode(self) -> bool:
        if 'fail_fast_mode' in self._workspace_state_cache:
            return self._workspace_state_cache['fail_fast_mode']
        # TODO: call api and set the result to cache
        try:
            res = self.request("get", "state")
            if res.status_code == 200:
                self._workspace_state_cache['fail_fast_mode'] = res.json().get('isFailFastMode', False)
                return self._workspace_state_cache['fail_fast_mode']
        except Exception as e:
            self.print_exception_and_recover(e, "Failed to check fail-fast mode status")

        return False
