#!/usr/bin/env python3
"""
Storage Module - JSONL directory structure and session persistence

Implements the standardized storage layout:
    ~/.mcp-audit/sessions/<platform>/<YYYY-MM-DD>/<session-id>.jsonl

This module provides:
- Platform-separated session storage
- Date-based organization for efficient queries
- Index files for cross-session discovery
- Migration helpers from v0.x format
"""

import json
from dataclasses import asdict, dataclass, field
from datetime import date, datetime
from pathlib import Path
from typing import Any, Dict, Iterator, List, Literal, Optional

# Schema version for storage format
STORAGE_SCHEMA_VERSION = "1.0.0"

# Supported platforms
Platform = Literal["claude_code", "codex_cli", "gemini_cli", "ollama_cli", "custom"]
SUPPORTED_PLATFORMS: List[Platform] = [
    "claude_code",
    "codex_cli",
    "gemini_cli",
    "ollama_cli",
    "custom",
]


def get_default_base_dir() -> Path:
    """
    Get the default base directory for mcp-audit data.

    Returns:
        Path to ~/.mcp-audit/sessions/
    """
    return Path.home() / ".mcp-audit" / "sessions"


@dataclass
class SessionIndex:
    """
    Index entry for a single session.

    Used for efficient cross-session queries without loading full session data.
    """

    schema_version: str
    session_id: str
    platform: Platform
    date: str  # YYYY-MM-DD format
    started_at: str  # ISO 8601 timestamp
    ended_at: Optional[str]  # ISO 8601 timestamp or None if incomplete
    project: Optional[str]
    total_tokens: int
    total_cost: float
    tool_count: int
    server_count: int
    is_complete: bool
    file_path: str  # Relative path from base_dir
    file_size_bytes: int

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary for JSON serialization."""
        return asdict(self)

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> "SessionIndex":
        """Create from dictionary."""
        return cls(**data)


@dataclass
class DailyIndex:
    """
    Index file for a single day's sessions.

    Stored at: <platform>/<YYYY-MM-DD>/.index.json
    """

    schema_version: str
    platform: Platform
    date: str  # YYYY-MM-DD
    sessions: List[SessionIndex] = field(default_factory=list)
    total_tokens: int = 0
    total_cost: float = 0.0
    session_count: int = 0
    last_updated: Optional[str] = None

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary for JSON serialization."""
        return {
            "schema_version": self.schema_version,
            "platform": self.platform,
            "date": self.date,
            "sessions": [s.to_dict() for s in self.sessions],
            "total_tokens": self.total_tokens,
            "total_cost": self.total_cost,
            "session_count": self.session_count,
            "last_updated": self.last_updated,
        }

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> "DailyIndex":
        """Create from dictionary."""
        sessions = [SessionIndex.from_dict(s) for s in data.get("sessions", [])]
        return cls(
            schema_version=data["schema_version"],
            platform=data["platform"],
            date=data["date"],
            sessions=sessions,
            total_tokens=data.get("total_tokens", 0),
            total_cost=data.get("total_cost", 0.0),
            session_count=data.get("session_count", 0),
            last_updated=data.get("last_updated"),
        )

    def add_session(self, session_index: SessionIndex) -> None:
        """Add a session to the daily index."""
        self.sessions.append(session_index)
        self.total_tokens += session_index.total_tokens
        self.total_cost += session_index.total_cost
        self.session_count += 1
        self.last_updated = datetime.now().isoformat()

    def recalculate_totals(self) -> None:
        """Recalculate aggregate totals from sessions."""
        self.total_tokens = sum(s.total_tokens for s in self.sessions)
        self.total_cost = sum(s.total_cost for s in self.sessions)
        self.session_count = len(self.sessions)


@dataclass
class PlatformIndex:
    """
    Index file for a platform's sessions.

    Stored at: <platform>/.index.json
    Provides quick access to date ranges and totals without scanning directories.
    """

    schema_version: str
    platform: Platform
    dates: List[str] = field(default_factory=list)  # List of YYYY-MM-DD with sessions
    total_sessions: int = 0
    total_tokens: int = 0
    total_cost: float = 0.0
    first_session_date: Optional[str] = None
    last_session_date: Optional[str] = None
    last_updated: Optional[str] = None

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary for JSON serialization."""
        return asdict(self)

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> "PlatformIndex":
        """Create from dictionary."""
        return cls(**data)


class StorageManager:
    """
    Manages session storage with the standardized directory structure.

    Directory Layout:
        ~/.mcp-audit/
        ├── sessions/
        │   ├── claude_code/
        │   │   ├── .index.json              # Platform-level index
        │   │   ├── 2025-11-24/
        │   │   │   ├── .index.json          # Daily index
        │   │   │   ├── session-abc123.jsonl # Session events
        │   │   │   └── session-def456.jsonl
        │   │   └── 2025-11-25/
        │   │       └── ...
        │   ├── codex_cli/
        │   │   └── ...
        │   └── gemini_cli/
        │       └── ...
        └── config/
            └── mcp-analyze.toml             # User configuration

    File Formats:
        - .jsonl files: Line-delimited JSON events (one event per line)
        - .index.json files: Metadata indexes for efficient queries
    """

    def __init__(self, base_dir: Optional[Path] = None):
        """
        Initialize storage manager.

        Args:
            base_dir: Base directory for session storage.
                      Defaults to ~/.mcp-audit/sessions/
        """
        self.base_dir = base_dir or get_default_base_dir()
        self.base_dir.mkdir(parents=True, exist_ok=True)

    # =========================================================================
    # Path Generation
    # =========================================================================

    def get_platform_dir(self, platform: Platform) -> Path:
        """Get directory for a specific platform."""
        if platform not in SUPPORTED_PLATFORMS:
            raise ValueError(
                f"Unsupported platform: {platform}. Must be one of {SUPPORTED_PLATFORMS}"
            )
        return self.base_dir / platform

    def get_date_dir(self, platform: Platform, session_date: date) -> Path:
        """Get directory for a specific date."""
        date_str = session_date.strftime("%Y-%m-%d")
        return self.get_platform_dir(platform) / date_str

    def get_session_path(self, platform: Platform, session_date: date, session_id: str) -> Path:
        """
        Get the path for a session file.

        Args:
            platform: Platform identifier
            session_date: Date of the session
            session_id: Unique session identifier

        Returns:
            Path to the session .jsonl file
        """
        return self.get_date_dir(platform, session_date) / f"{session_id}.jsonl"

    def generate_session_id(self, platform: Platform, timestamp: Optional[datetime] = None) -> str:
        """
        Generate a unique session ID.

        Format: session-{timestamp}-{random}
        Example: session-20251124T143052-a1b2c3

        Args:
            platform: Platform identifier (for context, not included in ID)
            timestamp: Session start time (defaults to now)

        Returns:
            Unique session identifier
        """
        import secrets

        ts = timestamp or datetime.now()
        ts_str = ts.strftime("%Y%m%dT%H%M%S")
        random_suffix = secrets.token_hex(3)  # 6 hex chars

        return f"session-{ts_str}-{random_suffix}"

    # =========================================================================
    # Session Writing
    # =========================================================================

    def create_session_file(
        self, platform: Platform, session_id: str, session_date: Optional[date] = None
    ) -> Path:
        """
        Create a new session file and its parent directories.

        Args:
            platform: Platform identifier
            session_id: Unique session identifier
            session_date: Date for the session (defaults to today)

        Returns:
            Path to the created session file
        """
        session_date = session_date or date.today()
        session_path = self.get_session_path(platform, session_date, session_id)

        # Create parent directories
        session_path.parent.mkdir(parents=True, exist_ok=True)

        # Create empty file
        session_path.touch()

        return session_path

    def append_event(self, session_path: Path, event: Dict[str, Any]) -> None:
        """
        Append an event to a session file.

        Args:
            session_path: Path to the session .jsonl file
            event: Event data to append
        """
        with open(session_path, "a") as f:
            f.write(json.dumps(event, default=str) + "\n")

    def write_session_events(self, session_path: Path, events: List[Dict[str, Any]]) -> None:
        """
        Write all events to a session file (overwrites existing).

        Args:
            session_path: Path to the session .jsonl file
            events: List of events to write
        """
        with open(session_path, "w") as f:
            for event in events:
                f.write(json.dumps(event, default=str) + "\n")

    # =========================================================================
    # Session Reading
    # =========================================================================

    def read_session_events(self, session_path: Path) -> Iterator[Dict[str, Any]]:
        """
        Read events from a session file as an iterator.

        Args:
            session_path: Path to the session .jsonl file

        Yields:
            Event dictionaries, one at a time
        """
        if not session_path.exists():
            return

        with open(session_path) as f:
            for line_num, line in enumerate(f, 1):
                line = line.strip()
                if not line:
                    continue
                try:
                    yield json.loads(line)
                except json.JSONDecodeError as e:
                    # Log warning but continue (graceful degradation)
                    print(f"Warning: Invalid JSON at {session_path}:{line_num}: {e}")

    def load_session_events(self, session_path: Path) -> List[Dict[str, Any]]:
        """
        Load all events from a session file into memory.

        Args:
            session_path: Path to the session .jsonl file

        Returns:
            List of event dictionaries
        """
        return list(self.read_session_events(session_path))

    # =========================================================================
    # Index Management
    # =========================================================================

    def get_daily_index_path(self, platform: Platform, session_date: date) -> Path:
        """Get path to daily index file."""
        return self.get_date_dir(platform, session_date) / ".index.json"

    def get_platform_index_path(self, platform: Platform) -> Path:
        """Get path to platform index file."""
        return self.get_platform_dir(platform) / ".index.json"

    def load_daily_index(self, platform: Platform, session_date: date) -> Optional[DailyIndex]:
        """
        Load daily index for a specific date.

        Args:
            platform: Platform identifier
            session_date: Date to load index for

        Returns:
            DailyIndex if exists, None otherwise
        """
        index_path = self.get_daily_index_path(platform, session_date)
        if not index_path.exists():
            return None

        try:
            with open(index_path) as f:
                data = json.load(f)
            return DailyIndex.from_dict(data)
        except (json.JSONDecodeError, KeyError) as e:
            print(f"Warning: Invalid daily index at {index_path}: {e}")
            return None

    def save_daily_index(self, index: DailyIndex) -> Path:
        """
        Save daily index to disk.

        Args:
            index: DailyIndex to save

        Returns:
            Path to saved index file
        """
        session_date = datetime.strptime(index.date, "%Y-%m-%d").date()
        index_path = self.get_daily_index_path(index.platform, session_date)

        # Ensure directory exists
        index_path.parent.mkdir(parents=True, exist_ok=True)

        with open(index_path, "w") as f:
            json.dump(index.to_dict(), f, indent=2)

        return index_path

    def load_platform_index(self, platform: Platform) -> Optional[PlatformIndex]:
        """
        Load platform index.

        Args:
            platform: Platform identifier

        Returns:
            PlatformIndex if exists, None otherwise
        """
        index_path = self.get_platform_index_path(platform)
        if not index_path.exists():
            return None

        try:
            with open(index_path) as f:
                data = json.load(f)
            return PlatformIndex.from_dict(data)
        except (json.JSONDecodeError, KeyError) as e:
            print(f"Warning: Invalid platform index at {index_path}: {e}")
            return None

    def save_platform_index(self, index: PlatformIndex) -> Path:
        """
        Save platform index to disk.

        Args:
            index: PlatformIndex to save

        Returns:
            Path to saved index file
        """
        index_path = self.get_platform_index_path(index.platform)

        # Ensure directory exists
        index_path.parent.mkdir(parents=True, exist_ok=True)

        with open(index_path, "w") as f:
            json.dump(index.to_dict(), f, indent=2)

        return index_path

    def update_indexes_for_session(
        self, platform: Platform, session_date: date, session_index: SessionIndex
    ) -> None:
        """
        Update both daily and platform indexes after adding/updating a session.

        Args:
            platform: Platform identifier
            session_date: Date of the session
            session_index: Index entry for the session
        """
        date_str = session_date.strftime("%Y-%m-%d")

        # Update daily index
        daily_index = self.load_daily_index(platform, session_date)
        if daily_index is None:
            daily_index = DailyIndex(
                schema_version=STORAGE_SCHEMA_VERSION,
                platform=platform,
                date=date_str,
            )

        # Check if session already exists (update) or is new (add)
        existing_idx = next(
            (
                i
                for i, s in enumerate(daily_index.sessions)
                if s.session_id == session_index.session_id
            ),
            None,
        )
        if existing_idx is not None:
            daily_index.sessions[existing_idx] = session_index
            daily_index.recalculate_totals()
        else:
            daily_index.add_session(session_index)

        self.save_daily_index(daily_index)

        # Update platform index
        platform_index = self.load_platform_index(platform)
        if platform_index is None:
            platform_index = PlatformIndex(
                schema_version=STORAGE_SCHEMA_VERSION,
                platform=platform,
            )

        if date_str not in platform_index.dates:
            platform_index.dates.append(date_str)
            platform_index.dates.sort()

        platform_index.total_sessions = sum(
            1
            for d in platform_index.dates
            if (idx := self.load_daily_index(platform, datetime.strptime(d, "%Y-%m-%d").date()))
            for _ in (idx.sessions if idx else [])
        )
        platform_index.first_session_date = (
            platform_index.dates[0] if platform_index.dates else None
        )
        platform_index.last_session_date = (
            platform_index.dates[-1] if platform_index.dates else None
        )
        platform_index.last_updated = datetime.now().isoformat()

        # Recalculate totals
        platform_index.total_tokens = 0
        platform_index.total_cost = 0.0
        platform_index.total_sessions = 0
        for date_str in platform_index.dates:
            daily = self.load_daily_index(platform, datetime.strptime(date_str, "%Y-%m-%d").date())
            if daily:
                platform_index.total_tokens += daily.total_tokens
                platform_index.total_cost += daily.total_cost
                platform_index.total_sessions += daily.session_count

        self.save_platform_index(platform_index)

    # =========================================================================
    # Session Discovery
    # =========================================================================

    def list_platforms(self) -> List[Platform]:
        """
        List all platforms with stored sessions.

        Returns:
            List of platform identifiers
        """
        platforms = []
        for platform in SUPPORTED_PLATFORMS:
            platform_dir = self.get_platform_dir(platform)
            if platform_dir.exists() and any(platform_dir.iterdir()):
                platforms.append(platform)
        return platforms

    def list_dates(self, platform: Platform) -> List[date]:
        """
        List all dates with sessions for a platform.

        Args:
            platform: Platform identifier

        Returns:
            List of dates, sorted newest first
        """
        platform_dir = self.get_platform_dir(platform)
        if not platform_dir.exists():
            return []

        dates = []
        for item in platform_dir.iterdir():
            if item.is_dir() and not item.name.startswith("."):
                try:
                    session_date = datetime.strptime(item.name, "%Y-%m-%d").date()
                    dates.append(session_date)
                except ValueError:
                    continue

        dates.sort(reverse=True)
        return dates

    def list_sessions(
        self,
        platform: Optional[Platform] = None,
        start_date: Optional[date] = None,
        end_date: Optional[date] = None,
        limit: Optional[int] = None,
    ) -> List[Path]:
        """
        List session files with optional filtering.

        Args:
            platform: Filter by platform (None for all)
            start_date: Filter sessions on or after this date
            end_date: Filter sessions on or before this date
            limit: Maximum number of sessions to return

        Returns:
            List of session file paths, sorted by date (newest first)
        """
        sessions = []

        platforms_to_check = [platform] if platform else self.list_platforms()

        for p in platforms_to_check:
            for session_date in self.list_dates(p):
                # Apply date filters
                if start_date and session_date < start_date:
                    continue
                if end_date and session_date > end_date:
                    continue

                date_dir = self.get_date_dir(p, session_date)
                for session_file in date_dir.glob("*.jsonl"):
                    if not session_file.name.startswith("."):
                        sessions.append(session_file)

        # Sort by modification time (newest first)
        sessions.sort(key=lambda p: p.stat().st_mtime, reverse=True)

        if limit:
            sessions = sessions[:limit]

        return sessions

    def find_session(self, session_id: str) -> Optional[Path]:
        """
        Find a session file by ID across all platforms and dates.

        Args:
            session_id: Session identifier to find

        Returns:
            Path to session file if found, None otherwise
        """
        for platform in self.list_platforms():
            for session_date in self.list_dates(platform):
                session_path = self.get_session_path(platform, session_date, session_id)
                if session_path.exists():
                    return session_path
        return None

    # =========================================================================
    # Statistics
    # =========================================================================

    def get_storage_stats(self) -> Dict[str, Any]:
        """
        Get storage statistics across all platforms.

        Returns:
            Dictionary with storage statistics
        """
        stats = {
            "base_dir": str(self.base_dir),
            "platforms": {},
            "total_sessions": 0,
            "total_size_bytes": 0,
        }

        for platform in self.list_platforms():
            platform_stats = {
                "session_count": 0,
                "date_count": 0,
                "size_bytes": 0,
            }

            dates = self.list_dates(platform)
            platform_stats["date_count"] = len(dates)

            for session_date in dates:
                date_dir = self.get_date_dir(platform, session_date)
                for session_file in date_dir.glob("*.jsonl"):
                    platform_stats["session_count"] += 1
                    platform_stats["size_bytes"] += session_file.stat().st_size

            stats["platforms"][platform] = platform_stats
            stats["total_sessions"] += platform_stats["session_count"]
            stats["total_size_bytes"] += platform_stats["size_bytes"]

        return stats


# =============================================================================
# Migration Helpers
# =============================================================================


def migrate_v0_session(
    v0_session_dir: Path, storage: StorageManager, platform: Platform = "claude_code"
) -> Optional[Path]:
    """
    Migrate a v0.x session directory to v1.x format.

    v0.x format: logs/sessions/{project}-{timestamp}/
        - summary.json
        - mcp-{server}.json
        - events.jsonl

    v1.x format: ~/.mcp-audit/sessions/<platform>/<YYYY-MM-DD>/<session-id>.jsonl

    Args:
        v0_session_dir: Path to v0.x session directory
        storage: StorageManager instance for v1.x storage
        platform: Platform to assign to migrated session

    Returns:
        Path to new session file if successful, None otherwise
    """
    # Check for events.jsonl (primary source)
    events_file = v0_session_dir / "events.jsonl"
    summary_file = v0_session_dir / "summary.json"

    if not events_file.exists() and not summary_file.exists():
        print(f"Warning: No events.jsonl or summary.json in {v0_session_dir}")
        return None

    # Extract date from directory name
    # Format: {project}-{YYYY}-{MM}-{DD}-{HHMMSS}
    dir_name = v0_session_dir.name
    try:
        # Try to extract date from the directory name
        parts = dir_name.rsplit("-", 4)
        if len(parts) >= 4:
            year = int(parts[-4])
            month = int(parts[-3])
            day = int(parts[-2])
            session_date = date(year, month, day)
        else:
            # Fallback to today
            session_date = date.today()
    except (ValueError, IndexError):
        session_date = date.today()

    # Generate new session ID
    session_id = storage.generate_session_id(platform)

    # Create new session file
    new_session_path = storage.create_session_file(platform, session_id, session_date)

    # Copy events to new format
    if events_file.exists():
        with open(events_file) as src, open(new_session_path, "w") as dst:
            for line in src:
                dst.write(line)

    # If we have summary.json, extract metadata for index
    if summary_file.exists():
        try:
            with open(summary_file) as f:
                summary = json.load(f)

            # Create index entry
            session_index = SessionIndex(
                schema_version=STORAGE_SCHEMA_VERSION,
                session_id=session_id,
                platform=platform,
                date=session_date.strftime("%Y-%m-%d"),
                started_at=summary.get("timestamp", datetime.now().isoformat()),
                ended_at=summary.get("end_timestamp"),
                project=summary.get("project"),
                total_tokens=summary.get("token_usage", {}).get("total_tokens", 0),
                total_cost=summary.get("cost_estimate", 0.0),
                tool_count=summary.get("mcp_tool_calls", {}).get("unique_tools", 0),
                server_count=len(summary.get("server_sessions", {})),
                is_complete=summary.get("end_timestamp") is not None,
                file_path=str(new_session_path.relative_to(storage.base_dir)),
                file_size_bytes=new_session_path.stat().st_size,
            )

            # Update indexes
            storage.update_indexes_for_session(platform, session_date, session_index)

        except (json.JSONDecodeError, KeyError) as e:
            print(f"Warning: Could not parse summary.json: {e}")

    return new_session_path


def migrate_all_v0_sessions(
    v0_base_dir: Path, storage: StorageManager, platform: Platform = "claude_code"
) -> Dict[str, Any]:
    """
    Migrate all v0.x sessions from a directory.

    Args:
        v0_base_dir: Base directory containing v0.x sessions (e.g., logs/sessions/)
        storage: StorageManager instance for v1.x storage
        platform: Default platform for migrated sessions

    Returns:
        Migration results dictionary
    """
    results = {
        "total": 0,
        "migrated": 0,
        "failed": 0,
        "skipped": 0,
        "errors": [],
    }

    if not v0_base_dir.exists():
        return results

    for session_dir in v0_base_dir.iterdir():
        if not session_dir.is_dir():
            continue

        results["total"] += 1

        # Detect platform from directory name if possible
        detected_platform = platform
        if "codex" in session_dir.name.lower():
            detected_platform = "codex_cli"
        elif "gemini" in session_dir.name.lower():
            detected_platform = "gemini_cli"
        elif "ollama" in session_dir.name.lower():
            detected_platform = "ollama_cli"

        try:
            new_path = migrate_v0_session(session_dir, storage, detected_platform)
            if new_path:
                results["migrated"] += 1
            else:
                results["skipped"] += 1
        except Exception as e:
            results["failed"] += 1
            results["errors"].append(f"{session_dir.name}: {e}")

    return results


# =============================================================================
# Testing
# =============================================================================

if __name__ == "__main__":
    import tempfile

    print("Storage Module Tests")
    print("=" * 60)

    # Use temporary directory for tests
    with tempfile.TemporaryDirectory() as temp_dir:
        storage = StorageManager(base_dir=Path(temp_dir))

        # Test 1: Generate session ID
        session_id = storage.generate_session_id("claude_code")
        print(f"✓ Generated session ID: {session_id}")

        # Test 2: Create session file
        session_path = storage.create_session_file(
            platform="claude_code", session_id=session_id, session_date=date.today()
        )
        print(f"✓ Created session file: {session_path}")

        # Test 3: Write events
        events = [
            {"type": "start", "timestamp": datetime.now().isoformat()},
            {"type": "tool_call", "tool": "mcp__zen__chat", "tokens": 1000},
            {"type": "end", "timestamp": datetime.now().isoformat()},
        ]
        storage.write_session_events(session_path, events)
        print(f"✓ Wrote {len(events)} events")

        # Test 4: Read events
        loaded_events = storage.load_session_events(session_path)
        assert len(loaded_events) == len(events)
        print(f"✓ Read {len(loaded_events)} events")

        # Test 5: Create and save daily index
        session_index = SessionIndex(
            schema_version=STORAGE_SCHEMA_VERSION,
            session_id=session_id,
            platform="claude_code",
            date=date.today().strftime("%Y-%m-%d"),
            started_at=datetime.now().isoformat(),
            ended_at=datetime.now().isoformat(),
            project="test-project",
            total_tokens=1000,
            total_cost=0.05,
            tool_count=1,
            server_count=1,
            is_complete=True,
            file_path=str(session_path.relative_to(storage.base_dir)),
            file_size_bytes=session_path.stat().st_size,
        )
        storage.update_indexes_for_session("claude_code", date.today(), session_index)
        print("✓ Updated indexes")

        # Test 6: List sessions
        sessions = storage.list_sessions()
        assert len(sessions) == 1
        print(f"✓ Listed {len(sessions)} session(s)")

        # Test 7: Get storage stats
        stats = storage.get_storage_stats()
        print(
            f"✓ Storage stats: {stats['total_sessions']} sessions, {stats['total_size_bytes']} bytes"
        )

        # Test 8: Load daily index
        daily_index = storage.load_daily_index("claude_code", date.today())
        assert daily_index is not None
        assert daily_index.session_count == 1
        print(
            f"✓ Daily index: {daily_index.session_count} session(s), {daily_index.total_tokens} tokens"
        )

        # Test 9: Load platform index
        platform_index = storage.load_platform_index("claude_code")
        assert platform_index is not None
        print(f"✓ Platform index: {platform_index.total_sessions} session(s)")

    print("\n" + "=" * 60)
    print("All tests passed!")
