#!/usr/bin/python
"""
    Copyright (c) 2016-present,
    Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.

    This file is part of Catapult.

    Catapult is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Catapult is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with Catapult. If not, see <http://www.gnu.org/licenses/>.
"""

from __future__ import annotations
from .GeneratorUtils import GeneratorUtils
from .NamespaceIdDto import NamespaceIdDto
from .ReceiptBuilder import ReceiptBuilder
from .ReceiptTypeDto import ReceiptTypeDto


class NamespaceExpiryReceiptBuilder(ReceiptBuilder):
    """Binary layout for a namespace expiry receipt."""

    def __init__(self, version: int, type_: ReceiptTypeDto, artifactId: NamespaceIdDto):
        """Constructor.
        Args:
            version: Receipt version.
            type_: Receipt type.
            artifactId: Expiring namespace id.
        """
        super().__init__(version, type_)
        self.artifactId = artifactId

    @classmethod
    def loadFromBinary(cls, payload: bytes) -> NamespaceExpiryReceiptBuilder:
        """Creates an instance of NamespaceExpiryReceiptBuilder from binary payload.
        Args:
            payload: Byte payload to use to serialize the object.
        Returns:
            Instance of NamespaceExpiryReceiptBuilder.
        """
        bytes_ = bytes(payload)
        superObject = ReceiptBuilder.loadFromBinary(bytes_)
        bytes_ = bytes_[superObject.getSize():]
        # gen: _load_from_binary_custom
        artifactId = NamespaceIdDto.loadFromBinary(bytes_)
        bytes_ = bytes_[artifactId.getSize():]
        return NamespaceExpiryReceiptBuilder(superObject.version, superObject.type_, artifactId)

    def getArtifactId(self) -> NamespaceIdDto:
        """Gets expiring namespace id.
        Returns:
            Expiring namespace id.
        """
        return self.artifactId

    def getSize(self) -> int:
        """Gets the size of the object.
        Returns:
            Size in bytes.
        """
        size: int = super().getSize()
        size += self.artifactId.getSize()
        return size

    def serialize(self) -> bytes:
        """Serializes an object to bytes.
        Returns:
            Serialized bytes.
        """
        bytes_ = bytes()
        superBytes = super().serialize()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, superBytes)
        artifactIdBytes = self.artifactId.serialize()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, artifactIdBytes)
        return bytes_
