# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['adcircpy',
 'adcircpy.cmd',
 'adcircpy.forcing',
 'adcircpy.forcing.tides',
 'adcircpy.forcing.waves',
 'adcircpy.forcing.winds',
 'adcircpy.forcing.winds._parametric',
 'adcircpy.mesh',
 'adcircpy.mesh.parsers',
 'adcircpy.outputs',
 'adcircpy.server']

package_data = \
{'': ['*']}

install_requires = \
['appdirs',
 'geopandas<0.11',
 'haversine',
 'matplotlib',
 'netCDF4<1.6',
 'numpy',
 'pandas',
 'paramiko',
 'pooch',
 'psutil',
 'pyproj>=2.6',
 'requests',
 'scipy',
 'shapely',
 'stormevents>=1.4.0,<2.0.0',
 'typepigeon<2',
 'utm']

extras_require = \
{'development': ['isort', 'oitnb'],
 'documentation': ['m2r2',
                   'sphinx',
                   'sphinx-rtd-theme',
                   'sphinxcontrib-programoutput',
                   'sphinxcontrib-bibtex'],
 'testing': ['pytest',
             'pytest-cov',
             'pytest-mock',
             'pytest-socket',
             'pytest-xdist']}

entry_points = \
{'console_scripts': ['best_track_file = adcircpy.cmd.best_track_file:main',
                     'best_track_run = adcircpy.cmd.best_track_run:main',
                     'fort63 = adcircpy.cmd.fort63:main',
                     'plot_fort61 = adcircpy.cmd.plot_fort61:main',
                     'plot_maxele = adcircpy.cmd.plot_maxele:main',
                     'plot_mesh = adcircpy.cmd.plot_mesh:main',
                     'tidal_run = adcircpy.cmd.tidal_run:main',
                     'tide_gen = adcircpy.cmd.tide_gen:main']}

setup_kwargs = {
    'name': 'adcircpy',
    'version': '1.2.3',
    'description': 'Python package for working with ADCIRC input and output files',
    'long_description': '# ADCIRCPy\n\n### Python library for automating ADCIRC model runs.\n\n[![tests](https://github.com/noaa-ocs-modeling/adcircpy/workflows/tests/badge.svg)](https://github.com/noaa-ocs-modeling/adcircpy/actions?query=workflow%3Atests)\n[![build](https://github.com/noaa-ocs-modeling/adcircpy/workflows/build/badge.svg)](https://github.com/noaa-ocs-modeling/adcircpy/actions?query=workflow%3Abuild)\n[![codecov](https://codecov.io/gh/noaa-ocs-modeling/adcircpy/branch/main/graph/badge.svg?token=BQWB1QKJ3Q)](https://codecov.io/gh/noaa-ocs-modeling/adcircpy)\n[![version](https://img.shields.io/pypi/v/adcircpy)](https://pypi.org/project/adcircpy)\n[![license](https://img.shields.io/github/license/noaa-ocs-modeling/adcircpy)](https://opensource.org/licenses/gpl-license)\n[![style](https://sourceforge.net/p/oitnb/code/ci/default/tree/_doc/_static/oitnb.svg?format=raw)](https://sourceforge.net/p/oitnb/code)\n[![documentation](https://readthedocs.org/projects/adcircpy/badge/?version=latest)](https://adcircpy.readthedocs.io/en/latest/?badge=latest)\n\nDocumentation can be found at https://adcircpy.readthedocs.io\n\n## Organization / Responsibility\n\nADCIRCpy is currently maintained by the [Coastal Marine Modeling Branch (CMMB)](https://coastaloceanmodels.noaa.gov) of the\nOffice of Coast Survey (OCS), a part of the [National Oceanic and Atmospheric Administration (NOAA)](https://www.noaa.gov), an\nagency of the United States federal government. It was majorly developed by [@jreniel](https://github.com/jreniel).\n\n- Zachary Burnett (**lead**) - zachary.burnett@noaa.gov\n- Jaime Calzada - jrcalzada@vims.edu\n\n## Installation\n\nPlease use a virtual environment with Python>=3.6. You may use conda or the OS\'s Python to provide a virtual environment for\nthe application.\n\nYou may install the application though pip. This will install the latest tagged version.\n<!--pytest-codeblocks:skip-->\n\n```bash\npip install adcircpy\n```\n\nAlternatively, you many manually install the repo by cloning it and then running\n<!--pytest-codeblocks:skip-->\n\n```bash\npip install .\n```\n\n## Usage\n\n### Command Line Interface (CLI)\n\nThis program exposes a few commands available from the command line interface. You may pass the `-h` flag to any of this\ncommands to explore their functionality.\n\n* `tide_gen`\n* `plot_mesh`\n* `tidal_run`\n* `best_track_run`\n* `best_track_file`\n* `plot_maxele`\n* `plot_fort61`\n* `fort63`\n\n#### examples\n\n##### generate tidal constituent template\n\nYou can quickly create a tidal component table for your your mesh by executing the `tide_gen` command and by passing a mesh, a\nstart date and number of run days as arguments. This functions sources data from\nthe [HAMTIDE](https://icdc.cen.uni-hamburg.de/en/hamtide.html) database by default.\n<!--pytest-codeblocks:skip-->\n\n```bash\ntide_gen \\\n    /path/to/your/fort.14 \\\n    2021-02-26T00:00:00 \\\n    15 \\\n    --mesh-crs=\'epsg:4326\'\n```\n\n##### run best-track run for Hurricane Sandy (AL182012)\n\nTo create the ADCIRC input files includes both tides and storm data for Hurricane Sandy:\n<!--pytest-codeblocks:skip-->\n\n```bash\nbest_track_run \\\n    /path/to/your/fort.14 \\\n    Sandy2012 \\\n    --fort13=/path/to/your/fort.13 \\\n    --crs=EPSG:4326 \\\n    --output-directory=/path/where/you/want/the/files \\\n    --constituents=all \\\n    --spinup-days=15.0 \\\n    --elev=30. \\\n    --mete=30. \\\n    --velo=30. \\\n    --skip-run\n```\n\nNote that the --crs flag is required due to the fort.14 not containing Coordinate Reference System information which is\nrequired for correct operation. [EPSG:4326](https://spatialreference.org/ref/epsg/wgs-84/) means that the mesh is in WGS84 (\nlat/lon). Note that the backlash represents "continue on next line" for the shell. You may write the command above on a single\nline after excluding the backslashes.\n\n##### plot results\n\nThese are two examples for doing quick plots with the package. These are given here as illustrative examples only. There is\nsupport for more file types than this examples, but the program does not yet support every output input/output file type. As a\nuser, you are encouraged to explore what\'s available and suggest and contribute your improvements.\n<!--pytest-codeblocks:skip-->\n\n```bash\nplot_fort61 /path/to/fort.61.nc MSL --show --coops-only\n```\n\n<!--pytest-codeblocks:skip-->\n\n```bash\nplot_mesh /path/to/fort.14 --show-elements\n```\n\n### Python API\n\nSee the [examples](./examples) directory for usage examples.\n\n#### `example_1.py`\n\nThe following code builds a simple ADCIRC run configuration by doing the following:\n\n1. reads a `fort.14` mesh file (specifically a test mesh for Shinnecock Inlet)\n2. adds tidal forcing to the mesh\n3. creates an `AdcircRun` driver object with the mesh, including start and end dates\n4. overrides default model options in the resulting `fort.15`\n5. runs ADCIRC if present, otherwise writes configuration to disk\n\n```python\nfrom datetime import datetime, timedelta\nfrom pathlib import Path\nimport shutil\n\nfrom adcircpy import AdcircMesh, AdcircRun, Tides\nfrom adcircpy.utilities import download_mesh, get_logger\n\nLOGGER = get_logger(__name__)\n\nDATA_DIRECTORY = Path(__file__).parent.absolute() / \'data\'\nINPUT_DIRECTORY = DATA_DIRECTORY / \'input\' / \'shinnecock\'\nOUTPUT_DIRECTORY = DATA_DIRECTORY / \'output\' / \'example_1\'\n\nMESH_DIRECTORY = INPUT_DIRECTORY / \'shinnecock\'\n\ndownload_mesh(\n    url=\'https://www.dropbox.com/s/1wk91r67cacf132/NetCDF_shinnecock_inlet.tar.bz2?dl=1\',\n    directory=MESH_DIRECTORY,\n    known_hash=\'99d764541983bfee60d4176af48ed803d427dea61243fa22d3f4003ebcec98f4\',\n)\n\n# open mesh file\nmesh = AdcircMesh.open(MESH_DIRECTORY / \'fort.14\', crs=4326)\n\n# initialize tidal forcing and constituents\ntidal_forcing = Tides()\ntidal_forcing.use_constituent(\'M2\')\ntidal_forcing.use_constituent(\'N2\')\ntidal_forcing.use_constituent(\'S2\')\ntidal_forcing.use_constituent(\'K1\')\ntidal_forcing.use_constituent(\'O1\')\nmesh.add_forcing(tidal_forcing)\n\n# set simulation dates\nduration = timedelta(days=5)\nstart_date = datetime(2015, 12, 14)\nend_date = start_date + duration\n\n# instantiate driver object\ndriver = AdcircRun(mesh, start_date, end_date)\n\n# request outputs\ndriver.set_elevation_surface_output(sampling_rate=timedelta(minutes=30))\ndriver.set_velocity_surface_output(sampling_rate=timedelta(minutes=30))\n\n# override default options so the resulting `fort.15` matches the original Shinnecock test case options\ndriver.timestep = 6.0\ndriver.DRAMP = 2.0\ndriver.TOUTGE = 3.8\ndriver.TOUTGV = 3.8\ndriver.smagorinsky = False\ndriver.horizontal_mixing_coefficient = 5.0\ndriver.gwce_solution_scheme = \'semi-implicit-legacy\'\n\nif shutil.which(\'padcirc\') is not None:\n    driver.run(OUTPUT_DIRECTORY, overwrite=True)\nelif shutil.which(\'adcirc\') is not None:\n    driver.run(OUTPUT_DIRECTORY, overwrite=True, nproc=1)\nelse:\n    LOGGER.warning(\n        \'ADCIRC binaries were not found in PATH. \'\n        \'ADCIRC will not run. Writing files to disk...\'\n    )\n    driver.write(OUTPUT_DIRECTORY, overwrite=True)\n```\n\n#### `example_2.py`\n\nThe following code is similar to `example_1.py`, above, except it adds a static Manning\'s N coefficient to the mesh.\n\n```python\nfrom datetime import datetime, timedelta\nfrom pathlib import Path\nimport shutil\n\nimport numpy\n\nfrom adcircpy import AdcircMesh, AdcircRun, Tides\nfrom adcircpy.utilities import download_mesh, get_logger\n\nLOGGER = get_logger(__name__)\n\nDATA_DIRECTORY = Path(__file__).parent.absolute() / \'data\'\nINPUT_DIRECTORY = DATA_DIRECTORY / \'input\'\nOUTPUT_DIRECTORY = DATA_DIRECTORY / \'output\' / \'example_2\'\n\nMESH_DIRECTORY = INPUT_DIRECTORY / \'shinnecock\'\n\ndownload_mesh(\n    url=\'https://www.dropbox.com/s/1wk91r67cacf132/NetCDF_shinnecock_inlet.tar.bz2?dl=1\',\n    directory=MESH_DIRECTORY,\n    known_hash=\'99d764541983bfee60d4176af48ed803d427dea61243fa22d3f4003ebcec98f4\',\n)\n\n# open mesh file\nmesh = AdcircMesh.open(MESH_DIRECTORY / \'fort.14\', crs=4326)\n\n# generate tau0 factor\nmesh.generate_tau0()\n\n# also add Manning\'s N to the domain (constant for this example)\nmesh.mannings_n_at_sea_floor = numpy.full(mesh.values.shape, 0.025)\n\n# initialize tidal forcing and constituents\ntidal_forcing = Tides()\ntidal_forcing.use_constituent(\'M2\')\ntidal_forcing.use_constituent(\'N2\')\ntidal_forcing.use_constituent(\'S2\')\ntidal_forcing.use_constituent(\'K1\')\ntidal_forcing.use_constituent(\'O1\')\nmesh.add_forcing(tidal_forcing)\n\n# set simulation dates\nspinup_time = timedelta(days=2)\nduration = timedelta(days=3)\nstart_date = datetime(2015, 12, 14) + spinup_time\nend_date = start_date + duration\n\n# instantiate driver object\ndriver = AdcircRun(mesh, start_date, end_date, spinup_time)\n\n# request outputs\ndriver.set_elevation_surface_output(sampling_rate=timedelta(minutes=30))\ndriver.set_velocity_surface_output(sampling_rate=timedelta(minutes=30))\n\n# override default options\ndriver.timestep = 4.0\n\nif shutil.which(\'padcirc\') is not None:\n    driver.run(OUTPUT_DIRECTORY, overwrite=True)\nelif shutil.which(\'adcirc\') is not None:\n    driver.run(OUTPUT_DIRECTORY, overwrite=True, nproc=1)\nelse:\n    LOGGER.warning(\n        \'ADCIRC binaries were not found in PATH. \'\n        \'ADCIRC will not run. Writing files to disk...\'\n    )\n    driver.write(OUTPUT_DIRECTORY, overwrite=True)\n```\n\n#### `example_3.py`\n\nThe following code is similar to `example_1.py`, above, except it adds HURDAT BestTrack wind forcing and also builds a Slurm\njob script for submission to a job manager.\n\n```python\nfrom datetime import datetime, timedelta\nfrom pathlib import Path\n\nfrom adcircpy import AdcircMesh, AdcircRun, Tides\nfrom adcircpy.forcing.winds import BestTrackForcing\nfrom adcircpy.server import SlurmConfig\nfrom adcircpy.utilities import download_mesh\n\nDATA_DIRECTORY = Path(__file__).parent.absolute() / \'data\'\nINPUT_DIRECTORY = DATA_DIRECTORY / \'input\'\nOUTPUT_DIRECTORY = DATA_DIRECTORY / \'output\' / \'example_3\'\n\nMESH_DIRECTORY = INPUT_DIRECTORY / \'shinnecock\'\n\ndownload_mesh(\n    url=\'https://www.dropbox.com/s/1wk91r67cacf132/NetCDF_shinnecock_inlet.tar.bz2?dl=1\',\n    directory=MESH_DIRECTORY,\n    known_hash=\'99d764541983bfee60d4176af48ed803d427dea61243fa22d3f4003ebcec98f4\',\n)\n\n# open mesh file\nmesh = AdcircMesh.open(MESH_DIRECTORY / \'fort.14\', crs=4326)\n\n# initialize tidal forcing and constituents\ntidal_forcing = Tides()\ntidal_forcing.use_all()\nmesh.add_forcing(tidal_forcing)\n\n# initialize wind forcing\nwind_forcing = BestTrackForcing(\'Sandy2012\')\nmesh.add_forcing(wind_forcing)\n\n# initialize Slurm configuration\nslurm = SlurmConfig(\n    account=\'account\',\n    ntasks=1000,\n    run_name=\'adcircpy/examples/example_3.py\',\n    partition=\'partition\',\n    walltime=timedelta(hours=8),\n    mail_type=\'all\',\n    mail_user=\'example@email.gov\',\n    log_filename=\'example_3.log\',\n    modules=[\'intel/2020\', \'impi/2020\', \'netcdf/4.7.2-parallel\'],\n    path_prefix=\'$HOME/adcirc/build\',\n)\n\n# set simulation dates\nspinup_time = timedelta(days=15)\nduration = timedelta(days=3)\nstart_date = datetime(2012, 10, 21, 18)\nend_date = start_date + duration\n\n# instantiate driver object\ndriver = AdcircRun(mesh, start_date, end_date, spinup_time, server_config=slurm)\n\n# write driver state to disk\ndriver.write(OUTPUT_DIRECTORY, overwrite=True)\n```\n\n#### `example_4.py`\n\nThe following code is similar to `example_3.py`, above, except it uses ATMESH wind forcing and WW3DATA wave forcing.\n\n```python\nfrom datetime import datetime, timedelta\nfrom pathlib import Path\n\nfrom adcircpy import AdcircMesh, AdcircRun, Tides\nfrom adcircpy.forcing.waves.ww3 import WaveWatch3DataForcing\nfrom adcircpy.forcing.winds.atmesh import AtmosphericMeshForcing\nfrom adcircpy.server import SlurmConfig\nfrom adcircpy.utilities import download_mesh\n\nDATA_DIRECTORY = Path(__file__).parent.absolute() / \'data\'\nINPUT_DIRECTORY = DATA_DIRECTORY / \'input\'\nOUTPUT_DIRECTORY = DATA_DIRECTORY / \'output\' / \'example_4\'\n\nMESH_DIRECTORY = INPUT_DIRECTORY / \'shinnecock\'\n\ndownload_mesh(\n    url=\'https://www.dropbox.com/s/1wk91r67cacf132/NetCDF_shinnecock_inlet.tar.bz2?dl=1\',\n    directory=MESH_DIRECTORY,\n    known_hash=\'99d764541983bfee60d4176af48ed803d427dea61243fa22d3f4003ebcec98f4\',\n)\n\n# open mesh file\nmesh = AdcircMesh.open(MESH_DIRECTORY / \'fort.14\', crs=4326)\n\n# initialize tidal forcing and constituents\ntidal_forcing = Tides()\ntidal_forcing.use_all()\nmesh.add_forcing(tidal_forcing)\n\n# initialize atmospheric mesh forcings (for NUOPC coupling)\nwind_forcing = AtmosphericMeshForcing(\n    filename=\'Wind_HWRF_SANDY_Nov2018_ExtendedSmoothT.nc\', nws=17, interval_seconds=3600,\n)\nmesh.add_forcing(wind_forcing)\n\n# initialize wave mesh forcings (for NUOPC coupling)\nwave_forcing = WaveWatch3DataForcing(\n    filename=\'ww3.HWRF.NOV2018.2012_sxy.nc\', nrs=5, interval_seconds=3600,\n)\nmesh.add_forcing(wave_forcing)\n\n# initialize Slurm configuration\nslurm = SlurmConfig(\n    account=\'account\',\n    ntasks=1000,\n    run_name=\'adcircpy/examples/example_4.py\',\n    partition=\'partition\',\n    walltime=timedelta(hours=8),\n    mail_type=\'all\',\n    mail_user=\'example@email.gov\',\n    log_filename=\'example_4.log\',\n    modules=[\'intel/2020\', \'impi/2020\', \'netcdf/4.7.2-parallel\'],\n    path_prefix=\'$HOME/adcirc/build\',\n)\n\n# instantiate driver object\ndriver = AdcircRun(\n    mesh=mesh,\n    start_date=datetime.now(),\n    end_date=timedelta(days=7),\n    spinup_time=timedelta(days=5),\n    server_config=slurm,\n)\n\n# write driver state to disk\ndriver.write(OUTPUT_DIRECTORY, overwrite=True)\n```\n\n## Citation\n\n```\nCalzada, J., Burnett, Z., Moghimi, S., Myers, E., & Pe’eri, S. (2021). ADCIRCpy: A Python API to generate ADCIRC model input files (Technical Memorandum No. 41; NOAA NOS OCS). National Oceanic and Atmospheric Administation.\n```\n\n```bibtex\n@techreport{calzadaADCIRCpyPythonAPI2021,\n    type = {Technical {{Memorandum}}},\n    title = {{{ADCIRCpy}}: A {{Python API}} to Generate {{ADCIRC}} Model Input Files},\n    author = {Calzada, Jaime and Burnett, Zachary and Moghimi, Saeed and Myers, Edward and Pe\'eri, Shachak},\n    year = {2021},\n    month = dec,\n    number = {41},\n    institution = {{National Oceanic and Atmospheric Administation}},\n    abstract = {The Advanced Circulation Model (ADCIRC) is a Fortran program used for modeling ocean circulation due to tides, surface waves and atmospheric forcings. However, the input formats and configuration are inflexible and not straight forward for operational implementation, making rapid iteration of model testing, ensemble configuration, and model coupling complicated. Here, we introduce a flexible abstraction of model inputs and outputs written in Python, called ADCIRCpy, that provides a simpler user interface for automatically generating ADCIRC configuration to a variety of inputs and model scenarios. This documentation outlines 1. the needs for such an abstraction, 2. the peculiarities and challenges with the ADCIRC model that necessitate custom logic, and 3. methodologies for generalizing user input in such a way as to make generating model configurations consistent, fast, and efficient.}\n}\n```\n\n## Acknowledgements\n\nThe majority of ADCIRCpy was written by Jaime Calzada [@jreniel](https://github.com/jreniel).\n',
    'author': 'Zach Burnett',
    'author_email': 'zachary.burnett@noaa.gov',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/noaa-ocs-modeling/adcircpy.git',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
