# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['wagtail_meilisearch',
 'wagtail_meilisearch.management',
 'wagtail_meilisearch.management.commands']

package_data = \
{'': ['*']}

install_requires = \
['arrow>=0.15.0,<0.16.0', 'meilisearch>=0.9.0,<0.10.0']

setup_kwargs = {
    'name': 'wagtail-meilisearch',
    'version': '0.1.5',
    'description': 'A MeiliSearch backend for Wagatil',
    'long_description': '# Wagtail MeiliSearch\n\nThis is a (beta) Wagtail search backend for the [MeiliSearch](https://github.com/meilisearch/MeiliSearch) search engine.\n\n\n## Installation\n\n`poetry add wagtail_meilisearch` or `pip install wagtail_meilisearch`\n\n## Configuration\n\nSee the [MeiliSearch docs](https://docs.meilisearch.com/guides/advanced_guides/installation.html#environment-variables-and-flags) for info on the values you want to add here.\n\n```\nWAGTAILSEARCH_BACKENDS = {\n    \'default\': {\n        \'BACKEND\': \'wagtail_meilisearch.backend\',\n        \'HOST\': os.environ.get(\'MEILISEARCH_HOST\', \'http://127.0.0.1\'),\n        \'PORT\': os.environ.get(\'MEILISEARCH_PORT\', \'7700\'),\n        \'MASTER_KEY\': os.environ.get(\'MEILI_MASTER_KEY\', \'\')\n    },\n}\n```\n\n## Update strategies\n\nIndexing a very large site with `python manage.py update_index` can be pretty taxing on the CPU, take quite a long time, and reduce the responsiveness of the MeiliSearch server. Wagtail-MeiliSearch offers two update strategies, `soft` and `hard`. The default, `soft` strategy will do an "add or update" call for each document sent to it, while the `hard` strategy will delete every document in the index and then replace them.\n\nThere are tradeoffs with either strategy - `hard` will guarantee that your search data matches your model data, but be hard work on the CPU for longer. `soft` will be faster and less CPU intensive, but if a field is removed from your model between indexings, that field data will remain in the search index.\n\nOne useful trick is to tell Wagtail that you have two search backends, with the default backend set to do `soft` updates that you can run nightly, and a second backend with `hard` updates that you can run less frequently.\n\n```\nWAGTAILSEARCH_BACKENDS = {\n    \'default\': {\n        \'BACKEND\': \'wagtail_meilisearch.backend\',\n        \'HOST\': os.environ.get(\'MEILISEARCH_HOST\', \'http://127.0.0.1\'),\n        \'PORT\': os.environ.get(\'MEILISEARCH_PORT\', \'7700\'),\n        \'MASTER_KEY\': os.environ.get(\'MEILI_MASTER_KEY\', \'\')\n    },\n    \'hard\': {\n        \'BACKEND\': \'wagtail_meilisearch.backend\',\n        \'HOST\': os.environ.get(\'MEILISEARCH_HOST\', \'http://127.0.0.1\'),\n        \'PORT\': os.environ.get(\'MEILISEARCH_PORT\', \'7700\'),\n        \'MASTER_KEY\': os.environ.get(\'MEILI_MASTER_KEY\', \'\'),\n        \'UPDATE_STRATEGY\': \'hard\'\n    }\n}\n```\n\nIf you use this technique, remember to pass the backend name into the `update_index` command otherwise both will run.\n\n`python manage.py update_index --backend default` for a soft update\n`python manage.py update_index --backend hard` for a hard update\n\n### Delta strategy\n\nThe `delta` strategy is useful if you habitually add created_at and updated_at timestamps to your models. This strategy will check the fields...\n\n* `first_published_at`\n* `last_published_at`\n* `created_at`\n* `updated_at`\n\nAnd only update the records for objects where one or more of these fields has a date more recent than the time delta specified in the settings.\n\n```\nWAGTAILSEARCH_BACKENDS = {\n    \'default\': {\n        \'BACKEND\': \'wagtail_meilisearch.backend\',\n        \'HOST\': os.environ.get(\'MEILISEARCH_HOST\', \'http://127.0.0.1\'),\n        \'PORT\': os.environ.get(\'MEILISEARCH_PORT\', \'7700\'),\n        \'MASTER_KEY\': os.environ.get(\'MEILI_MASTER_KEY\', \'\')\n        \'UPDATE_STRATEGY\': delta,\n        \'UPDATE_DELTA\': {\n            \'weeks\': -1\n        }\n    }\n}\n```\n\nIf the delta is set to `{\'weeks\': -1}`, wagtail-meilisearch will only update indexes for documents where one of the timestamp fields has a date within the last week. Your time delta _must_ be a negative.\n\nUnder the hood we use [Arrow](https://arrow.readthedocs.io), so you can use any keyword args supported by [Arrow\'s `shift()`](https://arrow.readthedocs.io/en/latest/index.html#replace-shift).\n\nIf you set `UPDATE_STRATEGY` to `delta` but don\'t provide a value for `UPDATE_DELTA` wagtail-meilisearch will default to `{\'weeks\': -1}`.\n\n## Skip models\n\nSometimes you might have a site where a certain page model is guaranteed not to change, for instance an archive section. After creating your initial search index, you can add a `SKIP_MODELS` key to the config to tell wagtail-meilisearch to ignore specific models when running `update_index`. Behind the scenes wagtail-meilisearch returns a dummy model index to the `update_index` management command for every model listed in your `SKIP_MODELS` - this ensures that this setting only affects `update_index`, so if you manually edit one of the models listed it should get re-indexed with the update signal.\n\n```\nWAGTAILSEARCH_BACKENDS = {\n    \'default\': {\n        \'BACKEND\': \'wagtail_meilisearch.backend\',\n        \'HOST\': os.environ.get(\'MEILISEARCH_HOST\', \'http://127.0.0.1\'),\n        \'PORT\': os.environ.get(\'MEILISEARCH_PORT\', \'7700\'),\n        \'MASTER_KEY\': os.environ.get(\'MEILI_MASTER_KEY\', \'\'),\n        \'UPDATE_STRATEGY\': \'delta\',\n        \'SKIP_MODELS\': [\n            \'core.ArchivePage\',\n        ]\n    }\n}\n```\n\n## Stop Words\n\nStop words are words for which we don\'t want to place significance on their frequency. For instance, the search query `tom and jerry` would return far less relevant results if the word `and` was given the same importance as `tom` and `jerry`. There\'s a fairly sane list of English language stop words supplied, but you can also supply your own. This is particularly useful if you have a lot of content in any other language.\n\n```\nMY_STOP_WORDS = [\'a\', \'list\', \'of\', \'words\']\n\nWAGTAILSEARCH_BACKENDS = {\n    \'default\': {\n        \'BACKEND\': \'wagtail_meilisearch.backend\',\n        [...]\n        \'STOP_WORDS\': MY_STOP_WORDS\n    },\n}\n```\n\nOr alternatively, you can extend the built in list.\n\n```\nfrom wagtail_meilisearch.settings import STOP_WORDS\n\nMY_STOP_WORDS = STOP_WORDS + WELSH_STOP_WORDS + FRENCH_STOP_WORDS\n\nWAGTAILSEARCH_BACKENDS = {\n    \'default\': {\n        \'BACKEND\': \'wagtail_meilisearch.backend\',\n        [...]\n        \'STOP_WORDS\': MY_STOP_WORDS\n    },\n}\n```\n\n\n## Contributing\n\nIf you want to help with the development I\'d be more than happy. The vast majority of the heavy lifting is done by MeiliSearch itself, but there is a TODO list...\n\n\n### TODO\n\n* Faceting\n* Implement boosting in the sort algorithm\n* Write tests\n* Performance improvements\n* ~~Implement stop words~~\n* ~~Search results~~\n* ~~Add support for the autocomplete api~~\n* ~~Ensure we\'re getting results by relevance~~\n\n### Thanks\n\nThank you to the devs of [Wagtail-Whoosh](https://github.com/wagtail/wagtail-whoosh). Reading the code over there was the only way I could work out how Wagtail Search backends are supposed to work.\n',
    'author': 'Hactar',
    'author_email': 'systems@hactar.is',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/hactar-is/wagtail-meilisearch',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
