# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['satispaython']

package_data = \
{'': ['*']}

install_requires = \
['cryptography>=3.4,<4.0', 'httpx>=0.16,<0.17']

setup_kwargs = {
    'name': 'satispaython',
    'version': '0.3.2',
    'description': 'A simple library to manage Satispay payments following the Web-button flow.',
    'long_description': "# satispaython\n\n[![GitHub Workflow Status](https://img.shields.io/github/workflow/status/otto-torino/satispaython/Test?style=flat-square)](https://github.com/otto-torino/satispaython/actions)\n[![Codecov](https://img.shields.io/codecov/c/github/otto-torino/satispaython?style=flat-square)](https://codecov.io/gh/otto-torino/satispaython)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/satispaython?style=flat-square)](https://pypi.org/project/satispaython)\n\nA simple library to manage Satispay payments following the [Web-button flow](https://developers.satispay.com/docs/web-button-pay).\n\n## Requirements\n\n* [`cryptography`](https://github.com/pyca/cryptography) >= 3.3\n* [`httpx`](https://github.com/encode/httpx) >= 0.16\n\n## Installation\n\nYou can install this package with pip: `pip install satispaython`.\n\n## Usage\n\n### Key generation\n\nFirst of all you need a RSA private key. You may generate the key by yourself or you may use the provided utility functions:\n\n```python\nfrom satispaython.utils import generate_key, write_key\n\nrsa_key = generate_key()\nwrite_key(rsa_key, 'path/to/file.pem')\n```\n\nIn order to load the key from a PEM encoded file you may use the utility function:\n\n```python\nfrom satispaython.utils import load_key\n\nrsa_key = load_key('path/to/file.pem')\n```\n\n> :information_source: The function `write_key` stores the key in the PEM format. If you generate the key with any other method and you would like to use the `load_key` function, please make sure the key is stored within a file in the PEM format.\n\n> :information_source: Satispaython key management is based on `cryptography` so all the functions which require an RSA key parameter expect an object of the class [`RSAPrivateKey`](https://cryptography.io/en/latest/hazmat/primitives/asymmetric/rsa.html#cryptography.hazmat.primitives.asymmetric.rsa.RSAPrivateKey). If you don't use the `load_key` function then make sure your key is an instance of `RSAPrivateKey`.\n\nYou may protect your key with a password simply adding the `password` parameter:\n\n```python\nwrite_key(rsa_key, 'path/to/file.pem', password='mypassword')\nrsa_key = load_key('path/to/file.pem', password='mypassword')\n```\n\n### Satispay API\n\nSatispaython web requests are based on `httpx` so the following functions return an instance of [`Response`](https://www.python-httpx.org/api/#response). On success, the Satispay API responds with a JSON encoded body, so you can simply check for the `response.status_code` and eventually get the content with `response.json()`.\n\n> :information_source: If you need to use the Sandbox endpoints be sure to read the [proper section](https://github.com/otto-torino/satispaython#sandbox-endpoints).\n\nIn order to use the [Satispay API](https://developers.satispay.com/reference) simply import satispaython:\n\n```python\nimport satispaython\n```\n\nThen you can:\n\n#### Obtain a key-id using a token\n\n```python\nresponse = satispaython.obtain_key_id(token, rsa_key)\n```\n\n> :information_source: The token is the activation code that can be generated from the Satispay Dashboard (or provided manually for Sandbox account).\n\n> :warning: Tokens are disposable! The key-id should be saved right after its creation.\n\n#### Make an authentication test\n\n```python\nresponse = satispaython.test_authentication(key_id, rsa_key)\n```\n\n> :information_source: Authentication tests work on Sandbox endpoints only.\n\n#### Create a payment\n\n```python\nresponse = satispaython.create_payment(key_id, rsa_key, amount_unit, currency, body_params=None, headers=None)\n```\n\nYou may use the utility function `format_datetime` to get a correctly formatted `expiration_date` to supply to the request:\n\n```python\nfrom datetime import datetime, timezone, timedelta\nfrom satispaython.utils import format_datetime\n\nexpiration_date = datetime.now(timezone.utc) + timedelta(hours=1)\nexpiration_date = format_datetime(expiration_date)\n```\n\n#### Get payment details\n\n```python\nresponse = satispaythonsatispaython.get_payment_details(key_id, rsa_key, payment_id, headers=None)\n```\n\n### Sandbox endpoints\n\nBy default satispaython points to the production Satispay API. If you need to use the [Sandbox](https://developers.satispay.com/docs/sandbox-account) endpoints, simply set the `staging` parameter to `True`:\n\n```python\nresponse = satispaython.obtain_key_id(rsa_key, token, staging=True)\nresponse = satispaython.create_payment(key_id, rsa_key, amount_unit, currency, body_params=None, headers=None, staging=True)\nresponse = satispaython.get_payment_details(key_id, rsa_key, payment_id, headers=None, staging=True)\n```\n\n## Clients, AsyncClients and Auth\n\nSatispaython offers specialized versions of `httpx`'s [`Client`](https://www.python-httpx.org/api/#client), [`AsyncClient`](https://www.python-httpx.org/api/#asyncclient) and [`Auth`](https://www.python-httpx.org/advanced/#customizing-authentication) classes:\n\n```python\nfrom satispaython import SatispayClient\n\nwith SatispayClient(key_id, rsa_key, staging=True) as client:\n    response = client.create_payment(100, 'EUR', body_params=None, headers=None)\n    response = client.get_payment_details(payment_id, headers=None)\n```\n\n```python\nfrom satispaython import AsyncSatispayClient\n\nasync with AsyncSatispayClient(key_id, rsa_key, staging=True) as client:\n    response = await client.create_payment(100, 'EUR', body_params=None, headers=None)\n    response = await client.get_payment_details(payment_id, headers=None)\n```\n\n```python\nimport httpx\nfrom satispaython import SatispayAuth\n\nauth = SatispayAuth(key_id, rsa_key)\nresponse = httpx.post('https://staging.authservices.satispay.com/wally-services/protocol/tests/signature', auth=auth)\n```\n",
    'author': 'Daniele Pira',
    'author_email': 'daniele.pira@otto.to.it',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/otto-torino/satispaython',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
