"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerImageBuilder = exports.ImageBuilderComponent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const constructs_1 = require("constructs");
const utils_1 = require("../../utils");
const common_1 = require("../common");
const dockerfileTemplate = `FROM {{{ imagebuilder:parentImage }}}
ENV RUNNER_VERSION=___RUNNER_VERSION___
{{{ imagebuilder:environments }}}
{{{ imagebuilder:components }}}`;
function uniqueName(scope) {
    return cdk.Names.uniqueResourceName(scope, { maxLength: 126, separator: '-', allowedSpecialCharacters: '_-' });
}
class ImageBuilderObjectBase extends cdk.Resource {
    constructor(scope, id) {
        super(scope, id);
    }
    version(type, name, data) {
        return new aws_cdk_lib_1.CustomResource(this, 'Version', {
            serviceToken: this.versionFunction().functionArn,
            resourceType: `Custom::ImageBuilder-${type}-Version`,
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            properties: {
                ObjectType: type,
                ObjectName: name,
                VersionedData: data,
            },
        }).ref;
    }
    versionFunction() {
        return utils_1.BundledNodejsFunction.singleton(this, 'aws-image-builder-versioner', {
            description: 'Custom resource handler that bumps up Image Builder versions',
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: [
                        'imagebuilder:ListComponents',
                        'imagebuilder:ListContainerRecipes',
                        'imagebuilder:ListImageRecipes',
                    ],
                    resources: ['*'],
                }),
            ],
        });
    }
}
/**
 * Components are a set of commands to run and optional files to add to an image. Components are the building blocks of images built by Image Builder.
 *
 * Example:
 *
 * ```
 * new ImageBuilderComponent(this, 'AWS CLI', {
 *   platform: 'Windows',
 *   displayName: 'AWS CLI',
 *   description: 'Install latest version of AWS CLI',
 *   commands: [
 *     '$ErrorActionPreference = \'Stop\'',
 *     'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
 *   ],
 * }
 * ```
 */
class ImageBuilderComponent extends ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.assets = [];
        this.platform = props.platform;
        let steps = [];
        if (props.assets) {
            let inputs = [];
            let extractCommands = [];
            for (const asset of props.assets) {
                this.assets.push(asset.asset);
                if (asset.asset.isFile) {
                    inputs.push({
                        source: asset.asset.s3ObjectUrl,
                        destination: asset.path,
                    });
                }
                else if (asset.asset.isZipArchive) {
                    inputs.push({
                        source: asset.asset.s3ObjectUrl,
                        destination: `${asset.path}.zip`,
                    });
                    if (props.platform === 'Windows') {
                        extractCommands.push('$ErrorActionPreference = \'Stop\'');
                        extractCommands.push(`Expand-Archive "${asset.path}.zip" -DestinationPath "${asset.path}"`);
                        extractCommands.push(`del "${asset.path}.zip"`);
                    }
                    else {
                        extractCommands.push(`unzip "${asset.path}.zip" -d "${asset.path}"`);
                        extractCommands.push(`rm "${asset.path}.zip"`);
                    }
                }
                else {
                    throw new Error(`Unknown asset type: ${asset.asset}`);
                }
            }
            steps.push({
                name: 'Download',
                action: 'S3Download',
                inputs,
            });
            if (extractCommands.length > 0) {
                steps.push({
                    name: 'Extract',
                    action: props.platform === 'Linux' ? 'ExecuteBash' : 'ExecutePowerShell',
                    inputs: {
                        commands: extractCommands,
                    },
                });
            }
        }
        steps.push({
            name: 'Run',
            action: props.platform === 'Linux' ? 'ExecuteBash' : 'ExecutePowerShell',
            inputs: {
                commands: props.commands,
            },
        });
        const data = {
            name: props.displayName,
            description: props.description,
            schemaVersion: '1.0',
            phases: [
                {
                    name: 'build',
                    steps,
                },
            ],
        };
        const name = uniqueName(this);
        const component = new aws_cdk_lib_1.aws_imagebuilder.CfnComponent(this, 'Component', {
            name: name,
            platform: props.platform,
            version: this.version('Component', name, {
                platform: props.platform,
                data,
            }),
            data: JSON.stringify(data),
        });
        this.arn = component.attrArn;
    }
    /**
     * Grants read permissions to the principal on the assets buckets.
     *
     * @param grantee
     */
    grantAssetsRead(grantee) {
        for (const asset of this.assets) {
            asset.grantRead(grantee);
        }
    }
}
exports.ImageBuilderComponent = ImageBuilderComponent;
_a = JSII_RTTI_SYMBOL_1;
ImageBuilderComponent[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.ImageBuilderComponent", version: "0.5.5" };
/**
 * Image builder recipe for a Docker container image.
 */
class ContainerRecipe extends ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        const name = uniqueName(this);
        let components = props.components.map(component => {
            return {
                componentArn: component.arn,
            };
        });
        const recipe = new aws_cdk_lib_1.aws_imagebuilder.CfnContainerRecipe(this, 'Recipe', {
            name: name,
            version: this.version('ContainerRecipe', name, {
                platform: props.platform,
                components,
                dockerfileTemplate,
            }),
            parentImage: props.parentImage ?? 'mcr.microsoft.com/windows/servercore:ltsc2019-amd64',
            components,
            containerType: 'DOCKER',
            targetRepository: {
                service: 'ECR',
                repositoryName: props.targetRepository.repositoryName,
            },
            dockerfileTemplateData: props.dockerfileTemplate,
        });
        this.arn = recipe.attrArn;
        this.name = name;
    }
}
/**
 * An image builder that uses Image Builder to build Docker images pre-baked with all the GitHub Actions runner requirements. Builders can be used with runner providers.
 *
 * The CodeBuild builder is better and faster. Only use this one if you have no choice. For example, if you need Windows containers.
 *
 * Each builder re-runs automatically at a set interval to make sure the images contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the image used to spin-up runners. Some runner providers may require custom components. Check the runner provider documentation. The default components work with CodeBuild and Fargate.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the Fargate provider, use:
 *
 * ```
 * const builder = new ContainerImageBuilder(this, 'Builder', {
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * new CodeBuildRunner(this, 'CodeBuild provider', {
 *     label: 'windows-codebuild',
 *     imageBuilder: builder,
 * });
 * ```
 */
class ContainerImageBuilder extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.components = [];
        // set platform
        this.architecture = props?.architecture ?? common_1.Architecture.X86_64;
        if (!this.architecture.is(common_1.Architecture.X86_64)) {
            throw new Error(`Unsupported architecture: ${this.architecture}. Consider CodeBuild for faster image builds.`);
        }
        this.os = props?.os ?? common_1.Os.LINUX;
        if (this.os.is(common_1.Os.WINDOWS)) {
            this.platform = 'Windows';
        }
        else {
            throw new Error(`Unsupported OS: ${this.os}. Consider CodeBuild for faster image builds.`);
        }
        this.parentImage = props?.parentImage;
        // set builder options
        this.rebuildInterval = props?.rebuildInterval ?? aws_cdk_lib_1.Duration.days(7);
        if (props?.vpc && props?.subnetSelection) {
            this.subnetId = props.vpc.selectSubnets(props.subnetSelection).subnetIds[0];
        }
        if (props?.securityGroup) {
            this.securityGroupIds = [props.securityGroup.securityGroupId];
        }
        this.instanceTypes = [props?.instanceType?.toString() ?? 'm5.large'];
        this.description = `Build image for GitHub Actions runner ${this.node.path} (${this.os.name}/${this.architecture.name})`;
        this.logRetention = props?.logRetention ?? aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH;
        this.logRemovalPolicy = props?.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY;
        // runner version
        this.runnerVersion = props?.runnerVersion ?? common_1.RunnerVersion.latest();
        // create repository that only keeps one tag
        this.repository = new aws_cdk_lib_1.aws_ecr.Repository(this, 'Repository', {
            imageScanOnPush: true,
            imageTagMutability: aws_ecr_1.TagMutability.MUTABLE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            lifecycleRules: [
                {
                    description: 'Remove all but the latest image',
                    tagStatus: aws_ecr_1.TagStatus.ANY,
                    maxImageCount: 1,
                },
            ],
        });
        // add all basic components
        this.addBaseWindowsComponents();
    }
    addBaseWindowsComponents() {
        this.addComponent(new ImageBuilderComponent(this, 'AWS CLI', {
            platform: 'Windows',
            displayName: 'AWS CLI',
            description: 'Install latest version of AWS CLI',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
            ],
        }));
        this.addComponent(new ImageBuilderComponent(this, 'GitHub CLI', {
            platform: 'Windows',
            displayName: 'GitHub CLI',
            description: 'Install latest version of gh',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/cli/cli/releases/latest > $Env:TEMP\\latest-gh',
                '$LatestUrl = Get-Content $Env:TEMP\\latest-gh',
                '$GH_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                '$ProgressPreference = \'SilentlyContinue\'',
                'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/cli/cli/releases/download/v${GH_VERSION}/gh_${GH_VERSION}_windows_amd64.msi" -OutFile gh.msi',
                'Start-Process msiexec.exe -Wait -ArgumentList \'/i gh.msi /qn\'',
                'del gh.msi',
            ],
        }));
        this.addComponent(new ImageBuilderComponent(this, 'git', {
            platform: 'Windows',
            displayName: 'Git',
            description: 'Install latest version of git',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                '$ProgressPreference = \'SilentlyContinue\'',
                'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/git-for-windows/git/releases/latest > $Env:TEMP\\latest-git',
                '$LatestUrl = Get-Content $Env:TEMP\\latest-git',
                '$GIT_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                '$GIT_VERSION_SHORT = ($GIT_VERSION -Split \'.windows.\')[0]',
                'Invoke-WebRequest -UseBasicParsing -Uri https://github.com/git-for-windows/git/releases/download/v${GIT_VERSION}/Git-${GIT_VERSION_SHORT}-64-bit.exe -OutFile git-setup.exe',
                'Start-Process git-setup.exe -Wait -ArgumentList \'/VERYSILENT\'',
                'del git-setup.exe',
            ],
        }));
        let runnerCommands;
        if (this.runnerVersion.version == common_1.RunnerVersion.latest().version) {
            runnerCommands = [
                'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/actions/runner/releases/latest > $Env:TEMP\\latest-gha',
                '$LatestUrl = Get-Content $Env:TEMP\\latest-gha',
                '$RUNNER_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
            ];
        }
        else {
            runnerCommands = [`$RUNNER_VERSION = '${this.runnerVersion.version}'`];
        }
        this.addComponent(new ImageBuilderComponent(this, 'GitHub Actions Runner', {
            platform: 'Windows',
            displayName: 'GitHub Actions Runner',
            description: 'Install latest version of GitHub Actions Runner',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                '$ProgressPreference = \'SilentlyContinue\'',
            ].concat(runnerCommands, [
                'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/actions/runner/releases/download/v${RUNNER_VERSION}/actions-runner-win-x64-${RUNNER_VERSION}.zip" -OutFile actions.zip',
                'Expand-Archive actions.zip -DestinationPath C:\\actions',
                'del actions.zip',
            ]),
        }));
    }
    /**
     * Add a component to be installed before any other components. Useful for required system settings like certificates or proxy settings.
     * @param component
     */
    prependComponent(component) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components = [component].concat(this.components);
    }
    /**
     * Add a component to be installed.
     * @param component
     */
    addComponent(component) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components.push(component);
    }
    /**
     * Add extra trusted certificates. This helps deal with self-signed certificates for GitHub Enterprise Server.
     *
     * All first party Dockerfiles support this. Others may not.
     *
     * @param path path to directory containing a file called certs.pem containing all the required certificates
     */
    addExtraCertificates(path) {
        this.prependComponent(new ImageBuilderComponent(this, 'Extra Certs', {
            platform: this.platform,
            displayName: 'GitHub Actions Runner',
            description: 'Install latest version of GitHub Actions Runner',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                'Import-Certificate -FilePath certs\\certs.pem -CertStoreLocation Cert:\\LocalMachine\\Root',
            ],
            assets: [
                {
                    path: 'certs',
                    asset: new aws_cdk_lib_1.aws_s3_assets.Asset(this, 'Extra Certs Asset', { path }),
                },
            ],
        }));
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the image builder.
     */
    bind() {
        if (this.boundImage) {
            return this.boundImage;
        }
        const infra = this.infrastructure();
        const dist = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'Distribution', {
            name: uniqueName(this),
            description: this.description,
            distributions: [
                {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    containerDistributionConfiguration: {
                        ContainerTags: ['latest'],
                        TargetRepository: {
                            Service: 'ECR',
                            RepositoryName: this.repository.repositoryName,
                        },
                    },
                },
            ],
        });
        const recipe = new ContainerRecipe(this, 'Container Recipe', {
            platform: this.platform,
            components: this.components,
            targetRepository: this.repository,
            dockerfileTemplate: dockerfileTemplate.replace('___RUNNER_VERSION___', this.runnerVersion.version),
            parentImage: this.parentImage,
        });
        const log = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Log', {
            logGroupName: `/aws/imagebuilder/${recipe.name}`,
            retention: this.logRetention,
            removalPolicy: this.logRemovalPolicy,
        });
        const image = new aws_cdk_lib_1.aws_imagebuilder.CfnImage(this, 'Image', {
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            containerRecipeArn: recipe.arn,
        });
        image.node.addDependency(log);
        this.imageCleaner(image, recipe.name);
        let scheduleOptions;
        if (this.rebuildInterval.toDays() > 0) {
            scheduleOptions = {
                scheduleExpression: aws_cdk_lib_1.aws_events.Schedule.rate(this.rebuildInterval).expressionString,
                pipelineExecutionStartCondition: 'EXPRESSION_MATCH_ONLY',
            };
        }
        const pipeline = new aws_cdk_lib_1.aws_imagebuilder.CfnImagePipeline(this, 'Pipeline', {
            name: uniqueName(this),
            description: this.description,
            containerRecipeArn: recipe.arn,
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            schedule: scheduleOptions,
        });
        pipeline.node.addDependency(log);
        this.boundImage = {
            // There are simpler ways to get the ARN, but we want an image object that depends on the newly built image.
            // We want whoever is using this image to automatically wait for Image Builder to finish building before using the image.
            imageRepository: aws_cdk_lib_1.aws_ecr.Repository.fromRepositoryName(this, 'Dependable Image', 
            // we can't use image.attrName because it comes up with upper case
            cdk.Fn.split(':', cdk.Fn.split('/', image.attrImageUri, 2)[1], 2)[0]),
            imageTag: 'latest',
            os: this.os,
            architecture: this.architecture,
            logGroup: log,
        };
        return this.boundImage;
    }
    infrastructure() {
        let role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
            ],
        });
        for (const component of this.components) {
            component.grantAssetsRead(role);
        }
        return new aws_cdk_lib_1.aws_imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: uniqueName(this),
            description: this.description,
            subnetId: this.subnetId,
            securityGroupIds: this.securityGroupIds,
            instanceTypes: this.instanceTypes,
            instanceProfileName: new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(this, 'Instance Profile', {
                roles: [
                    role.roleName,
                ],
            }).ref,
        });
    }
    imageCleaner(image, recipeName) {
        const crHandler = utils_1.BundledNodejsFunction.singleton(this, 'build-image', {
            description: 'Custom resource handler that triggers CodeBuild to build runner images, and cleans-up images on deletion',
            timeout: cdk.Duration.minutes(3),
        });
        const policy = new aws_cdk_lib_1.aws_iam.Policy(this, 'CR Policy', {
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ecr:BatchDeleteImage', 'ecr:ListImages'],
                    resources: [this.repository.repositoryArn],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['imagebuilder:ListImages', 'imagebuilder:ListImageBuildVersions', 'imagebuilder:DeleteImage'],
                    resources: ['*'],
                }),
            ],
        });
        crHandler.role?.attachInlinePolicy(policy);
        const cr = new aws_cdk_lib_1.CustomResource(this, 'Deleter', {
            serviceToken: crHandler.functionArn,
            resourceType: 'Custom::ImageDeleter',
            properties: {
                RepoName: this.repository.repositoryName,
                ImageBuilderName: recipeName,
                DeleteOnly: true,
            },
        });
        // add dependencies to make sure resources are there when we need them
        cr.node.addDependency(image);
        cr.node.addDependency(policy);
        cr.node.addDependency(crHandler);
        return cr;
    }
}
exports.ContainerImageBuilder = ContainerImageBuilder;
_b = JSII_RTTI_SYMBOL_1;
ContainerImageBuilder[_b] = { fqn: "@cloudsnorkel/cdk-github-runners.ContainerImageBuilder", version: "0.5.5" };
//# sourceMappingURL=data:application/json;base64,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