import { aws_codebuild as codebuild, aws_ec2 as ec2, aws_iam as iam, aws_stepfunctions as stepfunctions, Duration } from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { IImageBuilder, IRunnerProvider, RunnerImage, RunnerProviderProps, RunnerRuntimeParameters } from './common';
export interface CodeBuildRunnerProps extends RunnerProviderProps {
    /**
     * Provider running an image to run inside CodeBuild with GitHub runner pre-configured. A user named `runner` is expected to exist with access to Docker-in-Docker.
     *
     * @default image builder with `CodeBuildRunner.LINUX_X64_DOCKERFILE_PATH` as Dockerfile
     */
    readonly imageBuilder?: IImageBuilder;
    /**
     * GitHub Actions label used for this provider.
     *
     * @default 'codebuild'
     */
    readonly label?: string;
    /**
     * VPC to launch the runners in.
     *
     * @default no VPC
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Security Group to assign to this instance.
     *
     * @default public project with no security group
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * Where to place the network interfaces within the VPC.
     *
     * @default no subnet
     */
    readonly subnetSelection?: ec2.SubnetSelection;
    /**
     * The type of compute to use for this build.
     * See the {@link ComputeType} enum for the possible values.
     *
     * @default {@link ComputeType#SMALL}
     */
    readonly computeType?: codebuild.ComputeType;
    /**
     * The number of minutes after which AWS CodeBuild stops the build if it's
     * not complete. For valid values, see the timeoutInMinutes field in the AWS
     * CodeBuild User Guide.
     *
     * @default Duration.hours(1)
     */
    readonly timeout?: Duration;
}
/**
 * GitHub Actions runner provider using CodeBuild to execute the actions.
 *
 * Creates a project that gets started for each job.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
export declare class CodeBuildRunner extends Construct implements IRunnerProvider {
    /**
     * Path to Dockerfile for Linux x64 with all the requirements for CodeBuild runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
     *
     * Available build arguments that can be set in the image builder:
     * * `BASE_IMAGE` sets the `FROM` line. This should be an Ubuntu compatible image.
     * * `EXTRA_PACKAGES` can be used to install additional packages.
     * * `DOCKER_CHANNEL` overrides the channel from which Docker will be downloaded. Defaults to `"stsable"`.
     * * `DIND_COMMIT` overrides the commit where dind is found.
     * * `DOCKER_VERSION` overrides the installed Docker version.
     * * `DOCKER_COMPOSE_VERSION` overrides the installed docker-compose version.
     */
    static readonly LINUX_X64_DOCKERFILE_PATH: string;
    /**
     * Path to Dockerfile for Linux ARM64 with all the requirements for CodeBuild runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
     *
     * Available build arguments that can be set in the image builder:
     * * `BASE_IMAGE` sets the `FROM` line. This should be an Ubuntu compatible image.
     * * `EXTRA_PACKAGES` can be used to install additional packages.
     * * `DOCKER_CHANNEL` overrides the channel from which Docker will be downloaded. Defaults to `"stsable"`.
     * * `DIND_COMMIT` overrides the commit where dind is found.
     * * `DOCKER_VERSION` overrides the installed Docker version.
     * * `DOCKER_COMPOSE_VERSION` overrides the installed docker-compose version.
     */
    static readonly LINUX_ARM64_DOCKERFILE_PATH: string;
    /**
     * CodeBuild project hosting the runner.
     */
    readonly project: codebuild.Project;
    /**
     * Label associated with this provider.
     */
    readonly label: string;
    /**
     * VPC used for hosting the project.
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Security group attached to the task.
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * Grant principal used to add permissions to the runner role.
     */
    readonly grantPrincipal: iam.IPrincipal;
    /**
     * Docker image in CodeBuild project.
     */
    readonly image: RunnerImage;
    constructor(scope: Construct, id: string, props: CodeBuildRunnerProps);
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters: RunnerRuntimeParameters): stepfunctions.IChainable;
    /**
     * The network connections associated with this resource.
     */
    get connections(): ec2.Connections;
}
