"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HostPathVolumeType = exports.AzureDiskPersistentVolumeCachingMode = exports.AzureDiskPersistentVolumeKind = exports.EmptyDirMedium = exports.Volume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const k8s = require("./imports/k8s");
/**
 * Volume represents a named volume in a pod that may be accessed by any
 * container in the pod.
 *
 * Docker also has a concept of volumes, though it is somewhat looser and less
 * managed. In Docker, a volume is simply a directory on disk or in another
 * Container. Lifetimes are not managed and until very recently there were only
 * local-disk-backed volumes. Docker now provides volume drivers, but the
 * functionality is very limited for now (e.g. as of Docker 1.7 only one volume
 * driver is allowed per Container and there is no way to pass parameters to
 * volumes).
 *
 * A Kubernetes volume, on the other hand, has an explicit lifetime - the same
 * as the Pod that encloses it. Consequently, a volume outlives any Containers
 * that run within the Pod, and data is preserved across Container restarts. Of
 * course, when a Pod ceases to exist, the volume will cease to exist, too.
 * Perhaps more importantly than this, Kubernetes supports many types of
 * volumes, and a Pod can use any number of them simultaneously.
 *
 * At its core, a volume is just a directory, possibly with some data in it,
 * which is accessible to the Containers in a Pod. How that directory comes to
 * be, the medium that backs it, and the contents of it are determined by the
 * particular volume type used.
 *
 * To use a volume, a Pod specifies what volumes to provide for the Pod (the
 * .spec.volumes field) and where to mount those into Containers (the
 * .spec.containers[*].volumeMounts field).
 *
 * A process in a container sees a filesystem view composed from their Docker
 * image and volumes. The Docker image is at the root of the filesystem
 * hierarchy, and any volumes are mounted at the specified paths within the
 * image. Volumes can not mount onto other volumes
 */
class Volume extends constructs_1.Construct {
    constructor(scope, id, name, config) {
        super(scope, id);
        this.name = name;
        this.config = config;
        this.name = name.slice(0, 63);
    }
    /**
     * Mounts an Amazon Web Services (AWS) EBS volume into your pod.
     * Unlike emptyDir, which is erased when a pod is removed, the contents of an EBS volume are
     * persisted and the volume is unmounted. This means that an EBS volume can be pre-populated with data,
     * and that data can be shared between pods.
     *
     * There are some restrictions when using an awsElasticBlockStore volume:
     *
     * - the nodes on which pods are running must be AWS EC2 instances.
     * - those instances need to be in the same region and availability zone as the EBS volume.
     * - EBS only supports a single EC2 instance mounting a volume.
     */
    static fromAwsElasticBlockStore(scope, id, volumeId, options = {}) {
        return new Volume(scope, id, options.name ?? `ebs-${volumeId}`, {
            awsElasticBlockStore: {
                volumeId,
                fsType: options.fsType ?? 'ext4',
                partition: options.partition,
                readOnly: options.readOnly ?? false,
            },
        });
    }
    /**
     * Mounts a Microsoft Azure Data Disk into a pod.
     */
    static fromAzureDisk(scope, id, diskName, diskUri, options = {}) {
        return new Volume(scope, id, options.name ?? `azuredisk-${diskName}`, {
            azureDisk: {
                diskName,
                diskUri,
                cachingMode: options.cachingMode ?? AzureDiskPersistentVolumeCachingMode.NONE,
                fsType: options.fsType ?? 'ext4',
                kind: options.kind ?? AzureDiskPersistentVolumeKind.SHARED,
                readOnly: options.readOnly ?? false,
            },
        });
    }
    /**
     * Mounts a Google Compute Engine (GCE) persistent disk (PD) into your Pod.
     * Unlike emptyDir, which is erased when a pod is removed, the contents of a PD are
     * preserved and the volume is merely unmounted. This means that a PD can be pre-populated
     * with data, and that data can be shared between pods.
     *
     * There are some restrictions when using a gcePersistentDisk:
     *
     * - the nodes on which Pods are running must be GCE VMs
     * - those VMs need to be in the same GCE project and zone as the persistent disk
     */
    static fromGcePersistentDisk(scope, id, pdName, options = {}) {
        return new Volume(scope, id, options.name ?? `gcedisk-${pdName}`, {
            gcePersistentDisk: {
                pdName,
                fsType: options.fsType ?? 'ext4',
                partition: options.partition,
                readOnly: options.readOnly ?? false,
            },
        });
    }
    /**
     * Populate the volume from a ConfigMap.
     *
     * The configMap resource provides a way to inject configuration data into
     * Pods. The data stored in a ConfigMap object can be referenced in a volume
     * of type configMap and then consumed by containerized applications running
     * in a Pod.
     *
     * When referencing a configMap object, you can simply provide its name in the
     * volume to reference it. You can also customize the path to use for a
     * specific entry in the ConfigMap.
     *
     * @param configMap The config map to use to populate the volume.
     * @param options Options
     */
    static fromConfigMap(scope, id, configMap, options = {}) {
        return new Volume(scope, id, options.name ?? `configmap-${configMap.name}`, {
            configMap: {
                name: configMap.name,
                defaultMode: options.defaultMode,
                optional: options.optional,
                items: Volume.renderItems(options.items),
            },
        });
    }
    /**
     * An emptyDir volume is first created when a Pod is assigned to a Node, and
     * exists as long as that Pod is running on that node. As the name says, it is
     * initially empty. Containers in the Pod can all read and write the same
     * files in the emptyDir volume, though that volume can be mounted at the same
     * or different paths in each Container. When a Pod is removed from a node for
     * any reason, the data in the emptyDir is deleted forever.
     *
     * @see http://kubernetes.io/docs/user-guide/volumes#emptydir
     *
     * @param options - Additional options.
     */
    static fromEmptyDir(scope, id, name, options = {}) {
        return new Volume(scope, id, name, {
            emptyDir: {
                medium: options.medium,
                sizeLimit: options.sizeLimit
                    ? k8s.Quantity.fromString(`${options.sizeLimit.toMebibytes()}Mi`)
                    : undefined,
            },
        });
    }
    /**
     * Populate the volume from a Secret.
     *
     * A secret volume is used to pass sensitive information, such as passwords, to Pods.
     * You can store secrets in the Kubernetes API and mount them as files for use by pods
     * without coupling to Kubernetes directly.
     *
     * secret volumes are backed by tmpfs (a RAM-backed filesystem)
     * so they are never written to non-volatile storage.
     *
     * @see https://kubernetes.io/docs/concepts/storage/volumes/#secret
     *
     * @param secr The secret to use to populate the volume.
     * @param options Options
     */
    static fromSecret(scope, id, secr, options = {}) {
        return new Volume(scope, id, options.name ?? `secret-${secr.name}`, {
            secret: {
                secretName: secr.name,
                defaultMode: options.defaultMode,
                optional: options.optional,
                items: Volume.renderItems(options.items),
            },
        });
    }
    /**
     * Used to mount a PersistentVolume into a Pod.
     * PersistentVolumeClaims are a way for users to "claim" durable storage (such as a GCE PersistentDisk or an iSCSI volume)
     * without knowing the details of the particular cloud environment.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/
     */
    static fromPersistentVolumeClaim(scope, id, claim, options = {}) {
        return new Volume(scope, id, options.name ?? `pvc-${claim.name}`, {
            persistentVolumeClaim: {
                claimName: claim.name,
                readOnly: options.readOnly ?? false,
            },
        });
    }
    /**
     * Used to mount a file or directory from the host node's filesystem into a Pod.
     * This is not something that most Pods will need, but it offers a powerful
     * escape hatch for some applications.
     *
     * @see https://kubernetes.io/docs/concepts/storage/volumes/#hostpath
     */
    static fromHostPath(scope, id, name, options) {
        return new Volume(scope, id, name, {
            hostPath: {
                path: options.path,
                type: options.type ?? HostPathVolumeType.DEFAULT,
            },
        });
    }
    /**
     * Used to mount an NFS share into a Pod.
     *
     * @see https://kubernetes.io/docs/concepts/storage/volumes/#nfs
     */
    static fromNfs(scope, id, name, options) {
        return new Volume(scope, id, name, {
            nfs: {
                server: options.server,
                path: options.path,
                readOnly: options.readOnly,
            },
        });
    }
    /**
     * Populate the volume from a CSI driver, for example the Secrets Store CSI
     * Driver: https://secrets-store-csi-driver.sigs.k8s.io/introduction.html.
     * Which in turn needs an associated provider to source the secrets, such as
     * the AWS Secrets Manager and Systems Manager Parameter Store provider:
     * https://aws.github.io/secrets-store-csi-driver-provider-aws/.
     *
     * @param driver The name of the CSI driver to use to populate the volume.
     * @param options Options for the CSI volume, including driver-specific ones.
     */
    static fromCsi(scope, id, driver, options = {}) {
        return new Volume(scope, id, options.name ?? cdk8s_1.Names.toDnsLabel(scope, { extra: [id] }), {
            csi: {
                driver: driver,
                fsType: options.fsType,
                readOnly: options.readOnly,
                volumeAttributes: options.attributes,
            },
        });
    }
    asVolume() {
        return this;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            name: this.name,
            ...this.config,
        };
    }
}
exports.Volume = Volume;
_a = JSII_RTTI_SYMBOL_1;
Volume[_a] = { fqn: "cdk8s-plus-27.Volume", version: "2.7.36" };
/**
  * @internal
 */
Volume.renderItems = (items) => {
    if (!items) {
        return undefined;
    }
    const result = new Array();
    for (const key of Object.keys(items).sort()) {
        result.push({
            key,
            path: items[key].path,
            mode: items[key].mode,
        });
    }
    return result;
};
/**
 * The medium on which to store the volume.
 */
var EmptyDirMedium;
(function (EmptyDirMedium) {
    /**
     * The default volume of the backing node.
     */
    EmptyDirMedium["DEFAULT"] = "";
    /**
     * Mount a tmpfs (RAM-backed filesystem) for you instead. While tmpfs is very
     * fast, be aware that unlike disks, tmpfs is cleared on node reboot and any
     * files you write will count against your Container's memory limit.
     */
    EmptyDirMedium["MEMORY"] = "Memory";
})(EmptyDirMedium = exports.EmptyDirMedium || (exports.EmptyDirMedium = {}));
/**
 * Azure Disk kinds.
 */
var AzureDiskPersistentVolumeKind;
(function (AzureDiskPersistentVolumeKind) {
    /**
     * Multiple blob disks per storage account.
     */
    AzureDiskPersistentVolumeKind["SHARED"] = "Shared";
    /**
     * Single blob disk per storage account.
     */
    AzureDiskPersistentVolumeKind["DEDICATED"] = "Dedicated";
    /**
     * Azure managed data disk.
     */
    AzureDiskPersistentVolumeKind["MANAGED"] = "Managed";
})(AzureDiskPersistentVolumeKind = exports.AzureDiskPersistentVolumeKind || (exports.AzureDiskPersistentVolumeKind = {}));
/**
 * Azure disk caching modes.
 */
var AzureDiskPersistentVolumeCachingMode;
(function (AzureDiskPersistentVolumeCachingMode) {
    /**
     * None.
     */
    AzureDiskPersistentVolumeCachingMode["NONE"] = "None";
    /**
     * ReadOnly.
     */
    AzureDiskPersistentVolumeCachingMode["READ_ONLY"] = "ReadOnly";
    /**
     * ReadWrite.
     */
    AzureDiskPersistentVolumeCachingMode["READ_WRITE"] = "ReadWrite";
})(AzureDiskPersistentVolumeCachingMode = exports.AzureDiskPersistentVolumeCachingMode || (exports.AzureDiskPersistentVolumeCachingMode = {}));
/**
 * Host path types.
 */
var HostPathVolumeType;
(function (HostPathVolumeType) {
    /**
     * Empty string (default) is for backward compatibility, which means that no
     * checks will be performed before mounting the hostPath volume.
     */
    HostPathVolumeType["DEFAULT"] = "";
    /**
     * If nothing exists at the given path, an empty directory will be created
     * there as needed with permission set to 0755, having the same group and
     * ownership with Kubelet.
     */
    HostPathVolumeType["DIRECTORY_OR_CREATE"] = "DirectoryOrCreate";
    /**
     * A directory must exist at the given path.
     */
    HostPathVolumeType["DIRECTORY"] = "Directory";
    /**
     * If nothing exists at the given path, an empty file will be created there
     * as needed with permission set to 0644, having the same group and ownership
     * with Kubelet.
     */
    HostPathVolumeType["FILE_OR_CREATE"] = "FileOrCreate";
    /**
     * A file must exist at the given path.
     */
    HostPathVolumeType["FILE"] = "File";
    /**
     * A UNIX socket must exist at the given path.
     */
    HostPathVolumeType["SOCKET"] = "Socket";
    /**
     * A character device must exist at the given path.
     */
    HostPathVolumeType["CHAR_DEVICE"] = "CharDevice";
    /**
     * A block device must exist at the given path.
     */
    HostPathVolumeType["BLOCK_DEVICE"] = "BlockDevice";
})(HostPathVolumeType = exports.HostPathVolumeType || (exports.HostPathVolumeType = {}));
//# sourceMappingURL=data:application/json;base64,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