"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.GitHubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const helpers_internal_1 = require("aws-cdk-lib/pipelines/lib/helpers-internal");
const decamelize = require("decamelize");
const aws_credentials_1 = require("./private/aws-credentials");
const github_action_step_1 = require("./steps/github-action-step");
const github = require("./workflows-model");
const yaml_file_1 = require("./yaml-file");
const CDKOUT_ARTIFACT = 'cdk.out';
const ASSET_HASH_NAME = 'asset-hash';
/**
 * CDK Pipelines for GitHub workflows.
 */
class GitHubWorkflow extends pipelines_1.PipelineBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.jobOutputs = {};
        this.assetHashMap = {};
        this.stackProperties = {};
        this.cdkCliVersion = props.cdkCliVersion;
        this.preSynthed = props.preSynthed ?? false;
        this.buildContainer = props.buildContainer;
        this.preBuildSteps = props.preBuildSteps ?? [];
        this.postBuildSteps = props.postBuildSteps ?? [];
        this.gitHubActionRoleArn = props.gitHubActionRoleArn;
        this.useGitHubActionRole = this.gitHubActionRoleArn ? true : false;
        this.jobSettings = props.jobSettings;
        this.awsCredentials = props.awsCredentials ?? {
            accessKeyId: 'AWS_ACCESS_KEY_ID',
            secretAccessKey: 'AWS_SECRET_ACCESS_KEY',
        };
        this.dockerCredentials = props.dockerCredentials ?? [];
        this.workflowPath = props.workflowPath ?? '.github/workflows/deploy.yml';
        if (!this.workflowPath.endsWith('.yml') && !this.workflowPath.endsWith('.yaml')) {
            throw new Error('workflow file is expected to be a yaml file');
        }
        if (!this.workflowPath.includes('.github/workflows/')) {
            throw new Error('workflow files must be stored in the \'.github/workflows\' directory of your repository');
        }
        this.workflowFile = new yaml_file_1.YamlFile(this.workflowPath);
        this.workflowName = props.workflowName ?? 'deploy';
        this.workflowTriggers = props.workflowTriggers ?? {
            push: { branches: ['main'] },
            workflowDispatch: {},
        };
        this.runner = props.runner ?? github.Runner.UBUNTU_LATEST;
        this.publishAssetsAuthRegion = props.publishAssetsAuthRegion ?? 'us-west-2';
    }
    /**
     * Deploy a single Stage by itself with options for further GitHub configuration.
     *
     * Add a Stage to the pipeline, to be deployed in sequence with other Stages added to the pipeline.
     * All Stacks in the stage will be deployed in an order automatically determined by their relative dependencies.
     */
    addStageWithGitHubOptions(stage, options) {
        const stageDeployment = this.addStage(stage, options);
        // keep track of GitHub specific options
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, 'environment', options?.gitHubEnvironment);
        this.addStackProps(stacks, 'capabilities', options?.stackCapabilities);
        this.addStackProps(stacks, 'settings', options?.jobSettings);
        return stageDeployment;
    }
    addStackProps(stacks, key, value) {
        if (value === undefined) {
            return;
        }
        for (const stack of stacks) {
            this.stackProperties[stack.stackArtifactId] = {
                ...this.stackProperties[stack.stackArtifactId],
                [key]: value,
            };
        }
    }
    doBuildPipeline() {
        const app = aws_cdk_lib_1.Stage.of(this);
        if (!app) {
            throw new Error('The GitHub Workflow must be defined in the scope of an App');
        }
        const cdkoutDir = app.outdir;
        const jobs = new Array();
        const structure = new helpers_internal_1.PipelineGraph(this, {
            selfMutation: false,
            publishTemplate: true,
            prepareStep: false,
        });
        for (const stageNode of flatten(structure.graph.sortedChildren())) {
            if (!helpers_internal_1.isGraph(stageNode)) {
                throw new Error(`Top-level children must be graphs, got '${stageNode}'`);
            }
            const tranches = stageNode.sortedLeaves();
            for (const tranche of tranches) {
                for (const node of tranche) {
                    const job = this.jobForNode(node, {
                        assemblyDir: cdkoutDir,
                        structure,
                    });
                    if (job) {
                        jobs.push(job);
                    }
                }
            }
        }
        // convert jobs to a map and make sure there are no duplicates
        const jobmap = {};
        for (const job of jobs) {
            if (job.id in jobmap) {
                throw new Error(`duplicate job id ${job.id}`);
            }
            jobmap[job.id] = snakeCaseKeys(job.definition);
        }
        // Update jobs with late-bound output requests
        this.insertJobOutputs(jobmap);
        const workflow = {
            name: this.workflowName,
            on: snakeCaseKeys(this.workflowTriggers, '_'),
            jobs: jobmap,
        };
        // write as a yaml file
        this.workflowFile.update(workflow);
        // create directory if it does not exist
        fs_1.mkdirSync(path.dirname(this.workflowPath), { recursive: true });
        // GITHUB_WORKFLOW is set when GitHub Actions is running the workflow.
        // see: https://docs.github.com/en/actions/learn-github-actions/environment-variables#default-environment-variables
        const diffProtection = this.node.tryGetContext('cdk-pipelines-github:diffProtection') ?? true;
        if (diffProtection && process.env.GITHUB_WORKFLOW === this.workflowName) {
            // check if workflow file has changed
            if (!fs_1.existsSync(this.workflowPath) || this.workflowFile.toYaml() !== fs_1.readFileSync(this.workflowPath, 'utf8')) {
                throw new Error(`Please commit the updated workflow file ${path.relative(__dirname, this.workflowPath)} when you change your pipeline definition.`);
            }
        }
        this.workflowFile.writeFile();
    }
    insertJobOutputs(jobmap) {
        for (const [jobId, jobOutputs] of Object.entries(this.jobOutputs)) {
            jobmap[jobId] = {
                ...jobmap[jobId],
                outputs: {
                    ...jobmap[jobId].outputs,
                    ...this.renderJobOutputs(jobOutputs),
                },
            };
        }
    }
    renderJobOutputs(outputs) {
        const renderedOutputs = {};
        for (const output of outputs) {
            renderedOutputs[output.outputName] = `\${{ steps.${output.stepId}.outputs.${output.outputName} }}`;
        }
        return renderedOutputs;
    }
    /**
     * Make an action from the given node and/or step
     */
    jobForNode(node, options) {
        switch (node.data?.type) {
            // Nothing for these, they are groupings (shouldn't even have popped up here)
            case 'group':
            case 'stack-group':
            case undefined:
                throw new Error(`jobForNode: did not expect to get group nodes: ${node.data?.type}`);
            case 'self-update':
                throw new Error('GitHub Workflows does not support self mutation');
            case 'publish-assets':
                return this.jobForAssetPublish(node, node.data.assets, options);
            case 'prepare':
                throw new Error('"prepare" is not supported by GitHub Workflows');
            case 'execute':
                return this.jobForDeploy(node, node.data.stack, node.data.captureOutputs);
            case 'step':
                if (node.data.isBuildStep) {
                    return this.jobForBuildStep(node, node.data.step);
                }
                else if (node.data.step instanceof pipelines_1.ShellStep) {
                    return this.jobForScriptStep(node, node.data.step);
                }
                else if (node.data.step instanceof github_action_step_1.GitHubActionStep) {
                    return this.jobForGitHubActionStep(node, node.data.step);
                }
                else {
                    throw new Error(`unsupported step type: ${node.data.step.constructor.name}`);
                }
            default:
                // The 'as any' is temporary, until the change upstream rolls out
                throw new Error(`GitHubWorfklow does not support graph nodes of type '${node.data?.type}'. You are probably using a feature this CDK Pipelines implementation does not support.`);
        }
    }
    jobForAssetPublish(node, assets, options) {
        if (assets.length === 0) {
            throw new Error('Asset Publish step must have at least 1 asset');
        }
        const installSuffix = this.cdkCliVersion ? `@${this.cdkCliVersion}` : '';
        const cdkoutDir = options.assemblyDir;
        const jobId = node.uniqueId;
        const assetId = assets[0].assetId;
        // check if asset is docker asset and if we have docker credentials
        const dockerLoginSteps = [];
        if (node.uniqueId.includes('DockerAsset') && this.dockerCredentials.length > 0) {
            for (const creds of this.dockerCredentials) {
                dockerLoginSteps.push(...this.stepsToConfigureDocker(creds));
            }
        }
        // create one file and make one step
        const relativeToAssembly = (p) => path.posix.join(cdkoutDir, path.relative(path.resolve(cdkoutDir), p));
        const fileContents = ['set -ex'].concat(assets.map((asset) => {
            return `npx cdk-assets --path "${relativeToAssembly(asset.assetManifestPath)}" --verbose publish "${asset.assetSelector}"`;
        }));
        // we need the jobId to reference the outputs later
        this.assetHashMap[assetId] = jobId;
        fileContents.push(`echo '::set-output name=${ASSET_HASH_NAME}::${assetId}'`);
        const publishStepFile = path.join(cdkoutDir, `publish-${jobId}-step.sh`);
        fs_1.mkdirSync(path.dirname(publishStepFile), { recursive: true });
        fs_1.writeFileSync(publishStepFile, fileContents.join('\n'), { encoding: 'utf-8' });
        const publishStep = {
            id: 'Publish',
            name: `Publish ${jobId}`,
            run: `/bin/bash ./cdk.out/${path.relative(cdkoutDir, publishStepFile)}`,
        };
        return {
            id: jobId,
            definition: {
                name: `Publish Assets ${jobId}`,
                ...this.jobSettings,
                needs: this.renderDependencies(node),
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                runsOn: this.runner.runsOn,
                outputs: {
                    [ASSET_HASH_NAME]: `\${{ steps.Publish.outputs.${ASSET_HASH_NAME} }}`,
                },
                steps: [
                    ...this.stepsToDownloadAssembly(cdkoutDir),
                    {
                        name: 'Install',
                        run: `npm install --no-save cdk-assets${installSuffix}`,
                    },
                    ...this.stepsToConfigureAws(this.useGitHubActionRole, { region: this.publishAssetsAuthRegion }),
                    ...dockerLoginSteps,
                    publishStep,
                ],
            },
        };
    }
    jobForDeploy(node, stack, _captureOutputs) {
        const region = stack.region;
        const account = stack.account;
        if (!region || !account) {
            throw new Error('"account" and "region" are required');
        }
        if (!stack.templateUrl) {
            throw new Error(`unable to determine template URL for stack ${stack.stackArtifactId}`);
        }
        const resolve = (s) => {
            return cx_api_1.EnvironmentPlaceholders.replace(s, {
                accountId: account,
                region: region,
                partition: 'aws',
            });
        };
        const replaceAssetHash = (template) => {
            const hash = path.parse(template.split('/').pop() ?? '').name;
            if (this.assetHashMap[hash] === undefined) {
                throw new Error(`Template asset hash ${hash} not found.`);
            }
            return template.replace(hash, `\${{ needs.${this.assetHashMap[hash]}.outputs.${ASSET_HASH_NAME} }}`);
        };
        const params = {
            'name': stack.stackName,
            'template': replaceAssetHash(resolve(stack.templateUrl)),
            'no-fail-on-empty-changeset': '1',
        };
        const capabilities = this.stackProperties[stack.stackArtifactId]?.capabilities;
        if (capabilities) {
            params.capabilities = Array(capabilities).join(',');
        }
        if (stack.executionRoleArn) {
            params['role-arn'] = resolve(stack.executionRoleArn);
        }
        const assumeRoleArn = stack.assumeRoleArn ? resolve(stack.assumeRoleArn) : undefined;
        return {
            id: node.uniqueId,
            definition: {
                name: `Deploy ${stack.stackArtifactId}`,
                ...this.jobSettings,
                ...this.stackProperties[stack.stackArtifactId]?.settings,
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                ...(this.stackProperties[stack.stackArtifactId]?.environment ? {
                    environment: this.stackProperties[stack.stackArtifactId].environment,
                } : {}),
                needs: this.renderDependencies(node),
                runsOn: this.runner.runsOn,
                steps: [
                    ...this.stepsToConfigureAws(this.useGitHubActionRole, { region, assumeRoleArn }),
                    {
                        id: 'Deploy',
                        uses: 'aws-actions/aws-cloudformation-github-deploy@v1',
                        with: params,
                    },
                ],
            },
        };
    }
    jobForBuildStep(node, step) {
        if (!(step instanceof pipelines_1.ShellStep)) {
            throw new Error('synthStep must be a ScriptStep');
        }
        if (step.inputs.length > 0) {
            throw new Error('synthStep cannot have inputs');
        }
        if (step.outputs.length > 1) {
            throw new Error('synthStep must have a single output');
        }
        if (!step.primaryOutput) {
            throw new Error('synthStep requires a primaryOutput which contains cdk.out');
        }
        const cdkOut = step.outputs[0];
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: 'Synthesize',
                ...this.jobSettings,
                permissions: {
                    contents: github.JobPermission.READ,
                    // The Synthesize job does not use the GitHub Action Role on its own, but it's possible
                    // that it is being used in the preBuildSteps.
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: step.env,
                container: this.buildContainer,
                steps: [
                    ...this.stepsToCheckout(),
                    ...this.preBuildSteps,
                    ...installSteps,
                    {
                        name: 'Build',
                        run: step.commands.join('\n'),
                    },
                    ...this.postBuildSteps,
                    ...this.stepsToUploadAssembly(cdkOut.directory),
                ],
            },
        };
    }
    /**
     * Searches for the stack that produced the output via the current
     * job's dependencies.
     *
     * This function should always find a stack, since it is guaranteed
     * that a CfnOutput comes from a referenced stack.
     */
    findStackOfOutput(ref, node) {
        for (const dep of node.allDeps) {
            if (dep.data?.type === 'execute' && ref.isProducedBy(dep.data.stack)) {
                return dep.uniqueId;
            }
        }
        // Should never happen
        throw new Error(`The output ${ref.outputName} is not referenced by any of the dependent stacks!`);
    }
    addJobOutput(jobId, output) {
        if (this.jobOutputs[jobId] === undefined) {
            this.jobOutputs[jobId] = [output];
        }
        else {
            this.jobOutputs[jobId].push(output);
        }
    }
    jobForScriptStep(node, step) {
        const envVariables = {};
        for (const [envName, ref] of Object.entries(step.envFromCfnOutputs)) {
            const jobId = this.findStackOfOutput(ref, node);
            this.addJobOutput(jobId, {
                outputName: ref.outputName,
                stepId: 'Deploy',
            });
            envVariables[envName] = `\${{ needs.${jobId}.outputs.${ref.outputName} }}`;
        }
        const downloadInputs = new Array();
        const uploadOutputs = new Array();
        for (const input of step.inputs) {
            downloadInputs.push({
                uses: 'actions/download-artifact@v2',
                with: {
                    name: input.fileSet.id,
                    path: input.directory,
                },
            });
        }
        for (const output of step.outputs) {
            uploadOutputs.push({
                uses: 'actions/upload-artifact@v2.1.1',
                with: {
                    name: output.fileSet.id,
                    path: output.directory,
                },
            });
        }
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                ...this.jobSettings,
                permissions: {
                    contents: github.JobPermission.READ,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: {
                    ...step.env,
                    ...envVariables,
                },
                steps: [
                    ...downloadInputs,
                    ...installSteps,
                    { run: step.commands.join('\n') },
                    ...uploadOutputs,
                ],
            },
        };
    }
    jobForGitHubActionStep(node, step) {
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                ...this.jobSettings,
                permissions: {
                    contents: github.JobPermission.WRITE,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: step.env,
                steps: step.jobSteps,
            },
        };
    }
    stepsToConfigureAws(openId, { region, assumeRoleArn }) {
        function getDeployRole(arn) {
            return arn.replace('cfn-exec', 'deploy');
        }
        let steps = [];
        if (openId) {
            steps.push(aws_credentials_1.awsCredentialStep('Authenticate Via OIDC Role', {
                region,
                gitHubActionRoleArn: this.gitHubActionRoleArn,
            }));
            if (assumeRoleArn) {
                // Result of initial credentials with GitHub Action role are these environment variables
                steps.push(aws_credentials_1.awsCredentialStep('Assume CDK Deploy Role', {
                    region,
                    accessKeyId: '${{ env.AWS_ACCESS_KEY_ID }}',
                    secretAccessKey: '${{ env.AWS_SECRET_ACCESS_KEY }}',
                    sessionToken: '${{ env.AWS_SESSION_TOKEN }}',
                    roleToAssume: getDeployRole(assumeRoleArn),
                }));
            }
        }
        else {
            steps.push(aws_credentials_1.awsCredentialStep('Authenticate Via GitHub Secrets', {
                region,
                accessKeyId: `\${{ secrets.${this.awsCredentials.accessKeyId} }}`,
                secretAccessKey: `\${{ secrets.${this.awsCredentials.secretAccessKey} }}`,
                ...(this.awsCredentials.sessionToken ? {
                    sessionToken: `\${{ secrets.${this.awsCredentials.sessionToken} }}`,
                } : undefined),
                roleToAssume: assumeRoleArn,
            }));
        }
        return steps;
    }
    stepsToConfigureDocker(dockerCredential) {
        let params;
        if (dockerCredential.name === 'docker') {
            params = {
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        else if (dockerCredential.name === 'ecr') {
            params = {
                registry: dockerCredential.registry,
            };
        }
        else {
            params = {
                registry: dockerCredential.registry,
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        return [
            {
                uses: 'docker/login-action@v1',
                with: params,
            },
        ];
    }
    stepsToDownloadAssembly(targetDir) {
        if (this.preSynthed) {
            return this.stepsToCheckout();
        }
        return [{
                name: `Download ${CDKOUT_ARTIFACT}`,
                uses: 'actions/download-artifact@v2',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: targetDir,
                },
            }];
    }
    stepsToCheckout() {
        return [{
                name: 'Checkout',
                uses: 'actions/checkout@v2',
            }];
    }
    stepsToUploadAssembly(dir) {
        if (this.preSynthed) {
            return [];
        }
        return [{
                name: `Upload ${CDKOUT_ARTIFACT}`,
                uses: 'actions/upload-artifact@v2.1.1',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: dir,
                },
            }];
    }
    renderDependencies(node) {
        const deps = new Array();
        for (const d of node.allDeps) {
            if (d instanceof helpers_internal_1.Graph) {
                deps.push(...d.allLeaves().nodes);
            }
            else {
                deps.push(d);
            }
        }
        return deps.map(x => x.uniqueId);
    }
}
exports.GitHubWorkflow = GitHubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GitHubWorkflow[_a] = { fqn: "cdk-pipelines-github.GitHubWorkflow", version: "0.3.5" };
function snakeCaseKeys(obj, sep = '-') {
    if (typeof obj !== 'object' || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(o => snakeCaseKeys(o, sep));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        // we don't want to snake case environment variables
        if (k !== 'env' && typeof v === 'object' && v != null) {
            v = snakeCaseKeys(v);
        }
        result[decamelize(k, { separator: sep })] = v;
    }
    return result;
}
function* flatten(xs) {
    for (const x of xs) {
        for (const y of x) {
            yield y;
        }
    }
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,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