"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloneRepository = exports.Ec2Environment = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const cloud9_generated_1 = require("../lib/cloud9.generated");
/**
 * A Cloud9 Environment with Amazon EC2
 * @resource AWS::Cloud9::EnvironmentEC2
 */
class Ec2Environment extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.vpc = props.vpc;
        if (!props.subnetSelection && this.vpc.publicSubnets.length === 0) {
            throw new Error('no subnetSelection specified and no public subnet found in the vpc, please specify subnetSelection');
        }
        const vpcSubnets = (_a = props.subnetSelection) !== null && _a !== void 0 ? _a : { subnetType: ec2.SubnetType.PUBLIC };
        const c9env = new cloud9_generated_1.CfnEnvironmentEC2(this, 'Resource', {
            name: props.ec2EnvironmentName,
            description: props.description,
            instanceType: (_c = (_b = props.instanceType) === null || _b === void 0 ? void 0 : _b.toString()) !== null && _c !== void 0 ? _c : ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.MICRO).toString(),
            subnetId: this.vpc.selectSubnets(vpcSubnets).subnetIds[0],
            repositories: props.clonedRepositories ? props.clonedRepositories.map(r => ({
                repositoryUrl: r.repositoryUrl,
                pathComponent: r.pathComponent,
            })) : undefined,
        });
        this.environmentId = c9env.ref;
        this.ec2EnvironmentArn = c9env.getAtt('Arn').toString();
        this.ec2EnvironmentName = c9env.getAtt('Name').toString();
        this.ideUrl = `https://${this.stack.region}.console.aws.amazon.com/cloud9/ide/${this.environmentId}`;
    }
    /**
     * import from EnvironmentEc2Name
     */
    static fromEc2EnvironmentName(scope, id, ec2EnvironmentName) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.ec2EnvironmentName = ec2EnvironmentName;
                this.ec2EnvironmentArn = cdk.Stack.of(this).formatArn({
                    service: 'cloud9',
                    resource: 'environment',
                    resourceName: this.ec2EnvironmentName,
                });
            }
        }
        return new Import(scope, id);
    }
}
exports.Ec2Environment = Ec2Environment;
/**
 * The class for different repository providers
 */
class CloneRepository {
    constructor(repositoryUrl, pathComponent) {
        this.repositoryUrl = repositoryUrl;
        this.pathComponent = pathComponent;
    }
    /**
     * import repository to cloud9 environment from AWS CodeCommit
     *
     * @param repository the codecommit repository to clone from
     * @param path  the target path in cloud9 environment
     */
    static fromCodeCommit(repository, path) {
        return {
            repositoryUrl: repository.repositoryCloneUrlHttp,
            pathComponent: path,
        };
    }
}
exports.CloneRepository = CloneRepository;
//# sourceMappingURL=data:application/json;base64,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