"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sqs = require("@aws-cdk/aws-sqs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
let queue;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    queue = new sqs.Queue(stack, 'Queue');
});
test('Send message to queue', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Send', {
        task: new tasks.SendToQueue(queue, {
            messageBody: sfn.TaskInput.fromText('Send this message'),
            messageDeduplicationId: sfn.JsonPath.stringAt('$.deduping'),
        }),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sqs:sendMessage',
                ],
            ],
        },
        End: true,
        Parameters: {
            'QueueUrl': { Ref: 'Queue4A7E3555' },
            'MessageBody': 'Send this message',
            'MessageDeduplicationId.$': '$.deduping',
        },
    });
});
test('Send message to SQS queue with task token', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Send', {
        task: new tasks.SendToQueue(queue, {
            integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            messageBody: sfn.TaskInput.fromObject({
                Input: 'Send this message',
                Token: sfn.JsonPath.taskToken,
            }),
        }),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sqs:sendMessage.waitForTaskToken',
                ],
            ],
        },
        End: true,
        Parameters: {
            QueueUrl: { Ref: 'Queue4A7E3555' },
            MessageBody: {
                'Input': 'Send this message',
                'Token.$': '$$.Task.Token',
            },
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied but task token is not included in messageBody', () => {
    expect(() => {
        // WHEN
        new sfn.Task(stack, 'Send', {
            task: new tasks.SendToQueue(queue, {
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
                messageBody: sfn.TaskInput.fromText('Send this message'),
            }),
        });
        // THEN
    }).toThrow(/Task Token is missing in messageBody/i);
});
test('Message body can come from state', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Send', {
        task: new tasks.SendToQueue(queue, {
            messageBody: sfn.TaskInput.fromDataAt('$.theMessage'),
        }),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sqs:sendMessage',
                ],
            ],
        },
        End: true,
        Parameters: {
            'QueueUrl': { Ref: 'Queue4A7E3555' },
            'MessageBody.$': '$.theMessage',
        },
    });
});
test('Message body can be an object', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Send', {
        task: new tasks.SendToQueue(queue, {
            messageBody: sfn.TaskInput.fromObject({
                literal: 'literal',
                SomeInput: sfn.JsonPath.stringAt('$.theMessage'),
            }),
        }),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sqs:sendMessage',
                ],
            ],
        },
        End: true,
        Parameters: {
            QueueUrl: { Ref: 'Queue4A7E3555' },
            MessageBody: {
                'literal': 'literal',
                'SomeInput.$': '$.theMessage',
            },
        },
    });
});
test('Message body object can contain references', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Send', {
        task: new tasks.SendToQueue(queue, {
            messageBody: sfn.TaskInput.fromObject({
                queueArn: queue.queueArn,
            }),
        }),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sqs:sendMessage',
                ],
            ],
        },
        End: true,
        Parameters: {
            QueueUrl: { Ref: 'Queue4A7E3555' },
            MessageBody: {
                queueArn: { 'Fn::GetAtt': ['Queue4A7E3555', 'Arn'] },
            },
        },
    });
});
test('Task throws if SYNC is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'Send', {
            task: new tasks.SendToQueue(queue, {
                integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
                messageBody: sfn.TaskInput.fromText('Send this message'),
            }),
        });
    }).toThrow(/Invalid Service Integration Pattern: SYNC is not supported to call SQS./i);
});
//# sourceMappingURL=data:application/json;base64,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