"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateTrainingJob = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
const base_types_1 = require("./base-types");
/**
 * Class representing the SageMaker Create Training Job task.
 *
 * @experimental
 */
class SageMakerCreateTrainingJob extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        /**
         * Allows specify security group connections for instances of this fleet.
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateTrainingJob.SUPPORTED_INTEGRATION_PATTERNS);
        // set the default resource config if not defined.
        this.resourceConfig = props.resourceConfig || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
            volumeSize: core_1.Size.gibibytes(10),
        };
        // set the stopping condition if not defined
        this.stoppingCondition = props.stoppingCondition || {
            maxRuntime: core_1.Duration.hours(1),
        };
        // check that either algorithm name or image is defined
        if (!props.algorithmSpecification.algorithmName && !props.algorithmSpecification.trainingImage) {
            throw new Error('Must define either an algorithm name or training image URI in the algorithm specification');
        }
        // set the input mode to 'File' if not defined
        this.algorithmSpecification = props.algorithmSpecification.trainingInputMode
            ? props.algorithmSpecification
            : { ...props.algorithmSpecification, trainingInputMode: base_types_1.InputMode.FILE };
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.inputDataConfig = props.inputDataConfig.map((config) => {
            if (!config.dataSource.s3DataSource.s3DataType) {
                return {
                    ...config,
                    dataSource: { s3DataSource: { ...config.dataSource.s3DataSource, s3DataType: base_types_1.S3DataType.S3_PREFIX } },
                };
            }
            else {
                return config;
            }
        });
        // add the security groups to the connections object
        if (props.vpcConfig) {
            this.vpc = props.vpcConfig.vpc;
            this.subnets = props.vpcConfig.subnets ? this.vpc.selectSubnets(props.vpcConfig.subnets).subnetIds : this.vpc.selectSubnets().subnetIds;
        }
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * The execution role for the Sagemaker training job.
     *
     * Only available after task has been added to a state machine.
     */
    get role() {
        if (this._role === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._role;
    }
    get grantPrincipal() {
        if (this._grantPrincipal === undefined) {
            throw new Error('Principal not available yet--use the object in a Task first');
        }
        return this._grantPrincipal;
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createTrainingJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    renderParameters() {
        return {
            TrainingJobName: this.props.trainingJobName,
            RoleArn: this._role.roleArn,
            ...this.renderAlgorithmSpecification(this.algorithmSpecification),
            ...this.renderInputDataConfig(this.inputDataConfig),
            ...this.renderOutputDataConfig(this.props.outputDataConfig),
            ...this.renderResourceConfig(this.resourceConfig),
            ...this.renderStoppingCondition(this.stoppingCondition),
            ...this.renderHyperparameters(this.props.hyperparameters),
            ...this.renderTags(this.props.tags),
            ...this.renderVpcConfig(this.props.vpcConfig),
        };
    }
    renderAlgorithmSpecification(spec) {
        return {
            AlgorithmSpecification: {
                TrainingInputMode: spec.trainingInputMode,
                ...(spec.trainingImage ? { TrainingImage: spec.trainingImage.bind(this).imageUri } : {}),
                ...(spec.algorithmName ? { AlgorithmName: spec.algorithmName } : {}),
                ...(spec.metricDefinitions
                    ? { MetricDefinitions: spec.metricDefinitions.map((metric) => ({ Name: metric.name, Regex: metric.regex })) }
                    : {}),
            },
        };
    }
    renderInputDataConfig(config) {
        return {
            InputDataConfig: config.map((channel) => ({
                ChannelName: channel.channelName,
                DataSource: {
                    S3DataSource: {
                        S3Uri: channel.dataSource.s3DataSource.s3Location.bind(this, { forReading: true }).uri,
                        S3DataType: channel.dataSource.s3DataSource.s3DataType,
                        ...(channel.dataSource.s3DataSource.s3DataDistributionType
                            ? { S3DataDistributionType: channel.dataSource.s3DataSource.s3DataDistributionType }
                            : {}),
                        ...(channel.dataSource.s3DataSource.attributeNames ? { AttributeNames: channel.dataSource.s3DataSource.attributeNames } : {}),
                    },
                },
                ...(channel.compressionType ? { CompressionType: channel.compressionType } : {}),
                ...(channel.contentType ? { ContentType: channel.contentType } : {}),
                ...(channel.inputMode ? { InputMode: channel.inputMode } : {}),
                ...(channel.recordWrapperType ? { RecordWrapperType: channel.recordWrapperType } : {}),
            })),
        };
    }
    renderOutputDataConfig(config) {
        return {
            OutputDataConfig: {
                S3OutputPath: config.s3OutputLocation.bind(this, { forWriting: true }).uri,
                ...(config.encryptionKey ? { KmsKeyId: config.encryptionKey.keyArn } : {}),
            },
        };
    }
    renderResourceConfig(config) {
        return {
            ResourceConfig: {
                InstanceCount: config.instanceCount,
                InstanceType: 'ml.' + config.instanceType,
                VolumeSizeInGB: config.volumeSize.toGibibytes(),
                ...(config.volumeEncryptionKey ? { VolumeKmsKeyId: config.volumeEncryptionKey.keyArn } : {}),
            },
        };
    }
    renderStoppingCondition(config) {
        return {
            StoppingCondition: {
                MaxRuntimeInSeconds: config.maxRuntime && config.maxRuntime.toSeconds(),
            },
        };
    }
    renderHyperparameters(params) {
        return params ? { HyperParameters: params } : {};
    }
    renderTags(tags) {
        return tags ? { Tags: Object.keys(tags).map((key) => ({ Key: key, Value: tags[key] })) } : {};
    }
    renderVpcConfig(config) {
        return config
            ? {
                VpcConfig: {
                    SecurityGroupIds: core_1.Lazy.listValue({ produce: () => this.securityGroups.map((sg) => sg.securityGroupId) }),
                    Subnets: this.subnets,
                },
            }
            : {};
    }
    makePolicyStatements() {
        // set the sagemaker role or create new one
        this._grantPrincipal = this._role =
            this.props.role ||
                new iam.Role(this, 'SagemakerRole', {
                    assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
                    inlinePolicies: {
                        CreateTrainingJob: new iam.PolicyDocument({
                            statements: [
                                new iam.PolicyStatement({
                                    actions: [
                                        'cloudwatch:PutMetricData',
                                        'logs:CreateLogStream',
                                        'logs:PutLogEvents',
                                        'logs:CreateLogGroup',
                                        'logs:DescribeLogStreams',
                                        'ecr:GetAuthorizationToken',
                                        ...(this.props.vpcConfig
                                            ? [
                                                'ec2:CreateNetworkInterface',
                                                'ec2:CreateNetworkInterfacePermission',
                                                'ec2:DeleteNetworkInterface',
                                                'ec2:DeleteNetworkInterfacePermission',
                                                'ec2:DescribeNetworkInterfaces',
                                                'ec2:DescribeVpcs',
                                                'ec2:DescribeDhcpOptions',
                                                'ec2:DescribeSubnets',
                                                'ec2:DescribeSecurityGroups',
                                            ]
                                            : []),
                                    ],
                                    resources: ['*'],
                                }),
                            ],
                        }),
                    },
                });
        if (this.props.outputDataConfig.encryptionKey) {
            this.props.outputDataConfig.encryptionKey.grantEncrypt(this._role);
        }
        if (this.props.resourceConfig && this.props.resourceConfig.volumeEncryptionKey) {
            this.props.resourceConfig.volumeEncryptionKey.grant(this._role, 'kms:CreateGrant');
        }
        // create a security group if not defined
        if (this.vpc && this.securityGroup === undefined) {
            this.securityGroup = new ec2.SecurityGroup(this, 'TrainJobSecurityGroup', {
                vpc: this.vpc,
            });
            this.connections.addSecurityGroup(this.securityGroup);
            this.securityGroups.push(this.securityGroup);
        }
        const stack = core_1.Stack.of(this);
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTrainingJob', 'sagemaker:DescribeTrainingJob', 'sagemaker:StopTrainingJob'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'training-job',
                        // If the job name comes from input, we cannot target the policy to a particular ARN prefix reliably...
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.trainingJobName) ? '*' : `${this.props.trainingJobName}*`,
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this._role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTrainingJobsRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.SageMakerCreateTrainingJob = SageMakerCreateTrainingJob;
SageMakerCreateTrainingJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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