"""
Runs the installer

When this is done, the directory structure will look like this
sdk/Android:
    build-tools
    cmdline-tools
    emulator       # Generated by the Android SDK Manager
    flutter
    java
    licenses       # Generated by the Android SDK Manager
    patcher        # Generated by the Android SDK Manager
    platforms      # Generated by the Android SDK Manager
    platform-tools # Generated by the Android SDK Manager
    system-images  # Generated by the Android SDK Manager
    tools          # Generated by the Android SDK Manager
"""

# pylint: disable=missing-function-docstring

import argparse
import os
import sys
from pathlib import Path
import shutil
import subprocess
from download import download  # type: ignore

from pyflutterinstall.resources import (
    JAVA_SDK_URL,
    FLUTTER_GIT_DOWNLOAD,
    ANDROID_SDK_URL,
    CHROME_URL,
)

from pyflutterinstall.envset import add_system_path, set_env_var


assert (
    shutil.which("git") is not None
), "Git is not installed, please install, add it to the path then continue."

DELETE_PREVIOUS_INSTALL = True

PROJECT_ROOT = Path(os.getcwd())
INSTALL_DIR = PROJECT_ROOT / "sdk"
DOWNLOAD_DIR = PROJECT_ROOT / ".downloads"

ANDROID_SDK = INSTALL_DIR / "Android"
FLUTTER_TARGET = ANDROID_SDK / "flutter"
JAVA_DIR = ANDROID_SDK / "java"


def make_dirs() -> None:
    os.makedirs(INSTALL_DIR, exist_ok=True)
    os.makedirs(DOWNLOAD_DIR, exist_ok=True)
    os.makedirs(ANDROID_SDK, exist_ok=True)
    os.makedirs(JAVA_DIR, exist_ok=True)

    INSTALL_DIR.mkdir(parents=True, exist_ok=True)
    DOWNLOAD_DIR.mkdir(parents=True, exist_ok=True)


def execute(command, cwd=None):
    print(f"Executing\n  {command}")
    if cwd:
        print(f"  CWD={cwd}")
    subprocess.check_call(command, cwd=cwd, shell=True)


def install_java_sdk() -> None:
    print(f"Install Java SDK from {JAVA_SDK_URL} to {INSTALL_DIR}")
    java_sdk_zip_file = Path(
        download(JAVA_SDK_URL, DOWNLOAD_DIR / os.path.basename(JAVA_SDK_URL))
    )

    print(f"Unpacking {java_sdk_zip_file} to {JAVA_DIR}")
    shutil.unpack_archive(java_sdk_zip_file, JAVA_DIR)
    base_java_dir = JAVA_DIR / os.listdir(JAVA_DIR)[0]
    print(base_java_dir)
    java_bin_dir = base_java_dir / "bin"
    print(java_bin_dir)
    add_system_path(java_bin_dir)
    print("Java SDK installed.\n")


def install_android_sdk() -> None:
    print(
        f"Install Android commandline-tools SDK from {ANDROID_SDK_URL} to {INSTALL_DIR}"
    )
    path = download(ANDROID_SDK_URL, DOWNLOAD_DIR / os.path.basename(ANDROID_SDK_URL))
    print(f"Unpacking {path} to {INSTALL_DIR}")
    shutil.unpack_archive(path, ANDROID_SDK / "cmdline-tools" / "tools")
    cmd_tools_path = ANDROID_SDK / "cmdline-tools" / "tools" / "cmdline-tools" / "bin"
    sdkmanager_path = cmd_tools_path / "sdkmanager.bat"
    # install latest
    execute(f'{sdkmanager_path} --sdk_root="{ANDROID_SDK}" --install "platform-tools"')
    add_system_path(cmd_tools_path)
    set_env_var("ANDROID_SDK_ROOT", ANDROID_SDK)
    set_env_var("ANDROID_HOME", ANDROID_SDK)
    # update tools
    print(f"Updating Android SDK with {sdkmanager_path}")
    execute(f'{sdkmanager_path} --sdk_root="{ANDROID_SDK}" --update')
    tools_to_install = [
        "system-images;android-28;default;x86_64",
        "cmdline-tools;latest",
        "platform-tools",
        "build-tools;28.0.3",
        "platforms;android-30",
        "emulator",
        "tools",
    ]
    tools_to_install = [f'"{tool}"' for tool in tools_to_install]
    tools_list_str = " ".join(tools_to_install)
    # sdkmanager --install "cmdline-tools;latest"
    execute(f'{sdkmanager_path} --sdk_root="{ANDROID_SDK}" --install {tools_list_str}')
    execute(f'{sdkmanager_path} --licenses --sdk_root="{ANDROID_SDK}"')


def install_flutter() -> None:

    print(f"Install Flutter from {FLUTTER_GIT_DOWNLOAD} to {FLUTTER_TARGET}")
    if not FLUTTER_TARGET.exists():
        execute(f'{FLUTTER_GIT_DOWNLOAD} "{FLUTTER_TARGET}"')
    else:
        print(f"Flutter already installed at {FLUTTER_TARGET}")
    # Add flutter to path
    add_system_path(FLUTTER_TARGET / "bin")
    execute(f'flutter config --android-sdk "{ANDROID_SDK}"')
    execute("flutter doctor --android-licenses")
    print("Flutter installed.\n")


def install_chrome() -> None:
    # Install chrome for windows
    stdout = subprocess.check_output("flutter doctor", shell=True, text=True)
    if "Cannot find Chrome" in stdout:
        print("Chrome not found, installing")
        print(f"Install Chrome from {CHROME_URL} to {INSTALL_DIR}")
        path = download(CHROME_URL, DOWNLOAD_DIR / os.path.basename(CHROME_URL))
        print(f"Downloaded chrome at {path}")
        # install it
        execute(f'"{path}"')


def main():

    parser = argparse.ArgumentParser(description="Installs Flutter Dependencies")
    parser.add_argument("--skip-java", action="store_true", help="Skip Java SDK")
    parser.add_argument("--skip-android", action="store_true", help="Skip Android SDK")
    parser.add_argument("--skip-flutter", action="store_true", help="Skip Flutter SDK")
    parser.add_argument("--skip-chrome", action="store_true", help="Skip Chrome")
    args = parser.parse_args()

    # check for win32
    if sys.platform != "win32":
        print("This script is only for Windows")
        sys.exit(1)
    print("\nInstalling Flutter SDK and dependencies\n")
    make_dirs()
    if not args.skip_java:
        install_java_sdk()
    if not args.skip_android:
        install_android_sdk()
    if not args.skip_flutter:
        install_flutter()
    if not args.skip_chrome:
        install_chrome()
    print("\nDone installing Flutter SDK and dependencies\n")
    os.system("flutter doctor")


if __name__ == "__main__":
    main()
