import glob
import logging
import os
import re

import imageio

from utoolbox.cli.prompt import prompt_float
from utoolbox.data.datastore import ImageFolderDatastore
from ..base import MultiChannelDataset
from .error import MultipleSettingsError, SettingsNotFoundError
from .settings import ScanType, Settings
from .utils import refactor_datastore_keys

logger = logging.getLogger(__name__)

__all__ = ["SPIMDataset"]


class SPIMDataset(MultiChannelDataset):
    """
    Representation of an acquisition result from SPIM, containing software setup
    and collected data.

    Args:
        root (str): Source directory of the dataset.
        refactor (bool, optional): Refactor filenames of consectutive acquisitions. 

    Attributes:
        SETTINGS_PATTERN (str): Metadata filename generated by SPIM. Capture group is 
            named as "ds_name".
    """

    SETTINGS_PATTERN = r"(?P<ds_name>.+)_Settings.txt$"

    def __init__(self, root, refactor=True):
        self._refactor = refactor

        super().__init__(root)
        if not os.path.exists(self.root):
            raise FileNotFoundError("invalid dataset root")

    def _find_settings_file(self):
        """Find settings file generated by Lattice Scope."""
        # settings are .txt files
        search_pattern = os.path.join(self.root, "*.txt")
        filenames = glob.glob(search_pattern)

        ds_names = []
        for filename in filenames:
            basename = os.path.basename(filename)
            matches = re.match(SPIMDataset.SETTINGS_PATTERN, basename)
            if matches is None:
                continue
            ds_names.append((matches.group("ds_name"), filename))

        if not ds_names:
            raise SettingsNotFoundError("no known settings file")
        elif len(ds_names) > 1:
            logger.warning("diverged dataset, attempting to resolve it")

            # sort by name of dataset instead of actual path
            ds_names.sort(key=lambda t: t[0])

            # use the longest common prefix to resolve it
            ds_names_tr = list(zip(*ds_names))
            prefix = os.path.commonprefix(ds_names_tr[0])
            try:
                index = ds_names_tr[0].index(prefix)
            except ValueError:
                raise MultipleSettingsError(
                    "unable to determine which settings file to use"
                )
            return ds_names[index][1]
        else:
            # disable unnecessary refactor request
            self._refactor = False

            return ds_names[0][1]

    def _load_metadata(self):
        # TODO process all settings and compare for differences
        settings_file = self._find_settings_file()
        logger.debug('settings file "{}"'.format(settings_file))
        # NOTE some files have corrupted timestamp causing utf-8 decode error
        with open(settings_file, "r", errors="ignore") as fd:
            lines = fd.read()
        return Settings(lines)

    def _deserialize_info_from_metadata(self, dxy=0.103):
        metadata, info = self.metadata, self.info

        # time
        # .. iterate through channels
        n_stacks = set(channel.stacks for channel in metadata.waveform.channels)
        if len(n_stacks) > 1:
            logger.warning("number of saved stacks are inconsistent across channels")
            info.frames = max(n_stacks)
        else:
            info.frames = n_stacks.pop()

        # color
        info.channels = [ch.id for ch in metadata.waveform.channels]

        # stack, 2D
        lx, ly, rx, ry = metadata.camera.roi
        info.shape = (ry - ly, rx - lx)

        pxsize = prompt_float("What is the size of a single pixel? ")
        info.pixel_size = (pxsize,) * 2

        # stack, 3D
        if metadata.waveform.type == ScanType.OBJECTIVE:
            info.n_slices = metadata.waveform.obj_piezo_n_steps
            info.z_step = abs(metadata.waveform.obj_piezo_step_size)
        elif metadata.waveform.type == ScanType.SAMPLE:
            logger.warning("sample scanned data requires deskew")
            info.n_slices = metadata.waveform.sample_piezo_n_steps
            info.z_step = abs(metadata.waveform.sample_piezo_step_size)

    def _load_channel(self, channel):
        # NOTE
        # `imageio.volread` can adapt for both 2D and 3D TIFF files.
        ds = ImageFolderDatastore(
            self.root,
            read_func=imageio.volread,
            sub_dir=False,
            pattern="*_ch{}_*".format(channel),
        )

        if self._refactor:
            refactor_datastore_keys(ds)
        return ds
