"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResourceHandler = void 0;
/* eslint-disable no-console */
const https = require("https");
const url = require("url");
const client_sfn_1 = require("@aws-sdk/client-sfn");
class CustomResourceHandler {
    constructor(event, context) {
        this.event = event;
        this.context = context;
        this.timedOut = false;
        this.timeout = setTimeout(async () => {
            await this.respond({
                status: 'FAILED',
                reason: 'Lambda Function Timeout',
                data: this.context.logStreamName,
            });
            this.timedOut = true;
        }, context.getRemainingTimeInMillis() - 1200);
        this.event = event;
        this.physicalResourceId = extractPhysicalResourceId(event);
    }
    /**
     * Handles executing the custom resource event. If `stateMachineArn` is present
     * in the props then trigger the waiter statemachine
     */
    async handle() {
        try {
            if ('stateMachineArn' in this.event.ResourceProperties) {
                const req = {
                    stateMachineArn: this.event.ResourceProperties.stateMachineArn,
                    name: this.event.RequestId,
                    input: JSON.stringify(this.event),
                };
                await this.startExecution(req);
                return;
            }
            else {
                const response = await this.processEvent(this.event.ResourceProperties);
                return response;
            }
        }
        catch (e) {
            console.log(e);
            throw e;
        }
        finally {
            clearTimeout(this.timeout);
        }
    }
    /**
     * Handle async requests from the waiter state machine
     */
    async handleIsComplete() {
        try {
            const result = await this.processEvent(this.event.ResourceProperties);
            return result;
        }
        catch (e) {
            console.log(e);
            return;
        }
        finally {
            clearTimeout(this.timeout);
        }
    }
    /**
     * Start a step function state machine which will wait for the request
     * to be successful.
     */
    async startExecution(req) {
        try {
            const sfn = new client_sfn_1.SFN({});
            await sfn.startExecution(req);
        }
        finally {
            clearTimeout(this.timeout);
        }
    }
    respond(response) {
        if (this.timedOut) {
            return;
        }
        const cfResponse = {
            Status: response.status,
            Reason: response.reason,
            PhysicalResourceId: this.physicalResourceId,
            StackId: this.event.StackId,
            RequestId: this.event.RequestId,
            LogicalResourceId: this.event.LogicalResourceId,
            NoEcho: false,
            Data: response.data,
        };
        const responseBody = JSON.stringify(cfResponse);
        console.log('Responding to CloudFormation', responseBody);
        const parsedUrl = url.parse(this.event.ResponseURL);
        const requestOptions = {
            hostname: parsedUrl.hostname,
            path: parsedUrl.path,
            method: 'PUT',
            headers: {
                'content-type': '',
                'content-length': Buffer.byteLength(responseBody, 'utf8'),
            },
        };
        return new Promise((resolve, reject) => {
            try {
                const request = https.request(requestOptions, resolve);
                request.on('error', reject);
                request.write(responseBody);
                request.end();
            }
            catch (e) {
                reject(e);
            }
            finally {
                clearTimeout(this.timeout);
            }
        });
    }
}
exports.CustomResourceHandler = CustomResourceHandler;
function extractPhysicalResourceId(event) {
    switch (event.RequestType) {
        case 'Create':
            return event.LogicalResourceId;
        case 'Update':
        case 'Delete':
            return event.PhysicalResourceId;
    }
}
//# sourceMappingURL=data:application/json;base64,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