"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeployAssert = void 0;
const core_1 = require("aws-cdk-lib/core");
const constructs_1 = require("constructs");
const assertions_1 = require("../assertions");
const http_call_1 = require("../http-call");
const hash_1 = require("../private/hash");
const sdk_1 = require("../sdk");
const DEPLOY_ASSERT_SYMBOL = Symbol.for('@aws-cdk/integ-tests.DeployAssert');
/**
 * Construct that allows for registering a list of assertions
 * that should be performed on a construct
 */
class DeployAssert extends constructs_1.Construct {
    /**
     * Returns whether the construct is a DeployAssert construct
     */
    static isDeployAssert(x) {
        return x !== null && typeof (x) === 'object' && DEPLOY_ASSERT_SYMBOL in x;
    }
    /**
     * Finds a DeployAssert construct in the given scope
     */
    static of(construct) {
        const scopes = constructs_1.Node.of(constructs_1.Node.of(construct).root).findAll();
        const deployAssert = scopes.find(s => DeployAssert.isDeployAssert(s));
        if (!deployAssert) {
            throw new Error('No DeployAssert construct found in scopes');
        }
        return deployAssert;
    }
    constructor(scope, props) {
        super(scope, 'Default');
        this.assertionIdCounts = new Map();
        this.scope = props?.stack ?? new core_1.Stack(scope, 'DeployAssert');
        Object.defineProperty(this, DEPLOY_ASSERT_SYMBOL, { value: true });
    }
    awsApiCall(service, api, parameters, outputPaths) {
        let hash = '';
        try {
            hash = (0, hash_1.md5hash)(this.scope.resolve(parameters));
        }
        catch { }
        return new sdk_1.AwsApiCall(this.scope, this.uniqueAssertionId(`AwsApiCall${service}${api}${hash}`), {
            api,
            service,
            parameters,
            outputPaths,
        });
    }
    httpApiCall(url, options) {
        let hash = '';
        try {
            hash = (0, hash_1.md5hash)(this.scope.resolve({
                url,
                options,
            }));
        }
        catch { }
        let append = '';
        if (!core_1.Token.isUnresolved(url)) {
            const parsedUrl = new URL(url);
            append = `${parsedUrl.hostname}${parsedUrl.pathname}`;
        }
        return new http_call_1.HttpApiCall(this.scope, this.uniqueAssertionId(`HttpApiCall${append}${hash}`), {
            url,
            fetchOptions: options,
        });
    }
    invokeFunction(props) {
        const hash = (0, hash_1.md5hash)(this.scope.resolve(props));
        return new sdk_1.LambdaInvokeFunction(this.scope, this.uniqueAssertionId(`LambdaInvoke${hash}`), props);
    }
    expect(id, expected, actual) {
        new assertions_1.EqualsAssertion(this.scope, `EqualsAssertion${id}`, {
            expected,
            actual,
        });
    }
    /**
     * Gets a unique logical id based on a proposed assertion id.
     */
    uniqueAssertionId(id) {
        const count = this.assertionIdCounts.get(id);
        if (count === undefined) {
            // If we've never seen this id before, we'll return the id unchanged
            // to maintain backward compatibility.
            this.assertionIdCounts.set(id, 1);
            return id;
        }
        // Otherwise, we'll increment the counter and return a unique id.
        this.assertionIdCounts.set(id, count + 1);
        return `${id}${count}`;
    }
}
exports.DeployAssert = DeployAssert;
//# sourceMappingURL=data:application/json;base64,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