""" All exceptions for VeraCore REST API Client """


import logging


# pylint: disable=invalid-name
logger = logging.getLogger(__name__)


class VeraCoreException(Exception):
    """Base VeraCore API exception"""

    message = 'Unknown error occurred for {url}. Response content: {content}'

    def __init__(self, url, status, resource_name, content):
        """Initialize the VeraCoreException class
        VeraCoreException is the base class exception for all exceptions
        generated by this API client

        Arguments:
            url: VeraCore URL that was called
            status: Status code of the error response
            resource_name: Name of the VeraCore resource being queried
            content: content of the response
        """
        # pylint: disable=super-init-not-called
        self.url = url
        self.status = status
        self.resource_name = resource_name
        self.content = content

        if '{status}' in self.message:
            self.log_message = self.message.format(
                url=self.url, status=self.status, content=self.content)
        else:
            self.log_message = self.message.format(
                url=self.url, content=self.content)

        logger.error(self.log_message)

    def __str__(self):
        return self.message.format(url=self.url, content=self.content)


class VeraCoreRequestForbidden(VeraCoreException):
    """
    Error Code: 403
    The request has generated a forbidden access error.
    Verify that the service account has authenticated with a valid
    username, password, & system ID. Also verify that the token
    has not expired.
    """

    message = 'Request refused for {url}. Response content: {content}'


class VeraCoreRequestError(VeraCoreException):
    """
    Error Code: Dynamic
    The request completed with an error.
    """

    message = 'Request to {url} has failed with status {status}. {content}'

    def __str__(self):
        return self.message.format(
            url=self.url, status=self.status, content=self.content)


class VeraCoreNoSession(VeraCoreException):
    """
    Error Code: None
    A request was about to be attempted that didn't have a valid session.
    Try reinitiallizing or calling the login() method.
    """

    # pylint: disable=line-too-long
    message = 'No valid authentication token exists when attempting to call {url}. {content}'
