#!/usr/bin/env python
# -*- coding: utf-8; -*-

# Copyright (c) 2022 Oracle and/or its affiliates.
# Licensed under the Universal Permissive License v 1.0 as shown at https://oss.oracle.com/licenses/upl/


import os
import click
import yaml

from ads.opctl.utils import suppress_traceback
from ads.opctl.distributed.cmds import (
    initialize_workspace,
    show_config_info,
    cancel_distributed_run,
    generate_docker_compose_yaml,
)


@click.group("distributed-training")
@click.help_option("--help", "-h")
def commands():
    pass


@commands.command()
@click.option(
    "--framework",
    "-f",
    help="Distributed training framework type",
    type=click.Choice(["dask", "horovod-tensorflow", "horovod-pytorch", "pytorch"]),
    default=None,
    required=True,
)
@click.option(
    "--version",
    help="version number of the framework resource",
    default="v1",
    required=False,
)
@click.option("--debug", "-d", help="set debug mode", is_flag=True, default=False)
@click.help_option("--help", "-h")
def init(framework, version, debug):
    """
    Initialize workspace to build distributed framework compatible container images.
    Supported frameworks are - dask, horovod-tensorflow, horovod-pytorch and pytroch
    """
    initialize_workspace(
        framework,
        version,
    )


@commands.command()
@click.option(
    "--job-id",
    "-j",
    help="Job OCID of the cluster",
    default=None,
    required=False,
)
@click.option(
    "--work-dir",
    "-f",
    help="Working directory provided when the cluster was created.",
    default=None,
    required=False,
)
@click.option(
    "--file",
    "-f",
    help="Yaml file with cluster run information",
    default=None,
    required=False,
)
@click.option(
    "--worker-info", help="Show worker information", is_flag=True, default=False
)
@click.option("--debug", "-d", help="set debug mode", is_flag=True, default=False)
@click.option(
    "--oci-config",
    help="OCI Config file location",
    default=None,
    required=False,
)
@click.option(
    "--oci-profile",
    help="oci config profile",
    default=None,
    required=False,
)
def show_config(job_id, work_dir, file, worker_info, debug, **kwargs):
    """
    Displays the config json files create by the main node and the worker nodes. By default only the config generated by the Main node is displayed.
    """
    if not (job_id and work_dir) and not file:
        print("Either job ocid or yaml file with `jobId` key is required")
        return
    show_config_info(
        job_id,
        work_dir,
        file,
        worker_info,
        **kwargs,
    )


@commands.command()
@click.option(
    "--job-id",
    "-j",
    help="Job OCID of the cluster",
    default=None,
    required=False,
)
@click.option(
    "--file",
    "-f",
    help="Yaml file with cluster run information",
    default=None,
    required=False,
)
@click.option("--debug", "-d", help="set debug mode", is_flag=True, default=False)
@click.option(
    "--oci-config",
    help="OCI Config file location",
    default=None,
    required=False,
)
@click.option(
    "--oci-profile",
    help="oci config profile",
    default=None,
    required=False,
)
def cancel(job_id, file, debug, **kwargs):
    """Terminate the running cluster. Use Job ID of the main jobrun or use the file which has the yaml output displayed when cluster was created using ``ads opctl run -f <yaml>``"""
    if not (job_id or file):
        print("Either job ocid or yaml file with `jobId` key is required")
        return
    cancel_distributed_run(job_id, file, **kwargs)


@commands.command()
@click.option(
    "--file",
    "-f",
    help="YAML file with cluster run information",
    default=None,
    required=False,
)
@click.option(
    "--output",
    "-o",
    help="Output path for Docker compose YAML file. Defaults to docker-compose.yml in same directory as the YAML file.",
    default=None,
    required=False,
)
@click.option("--debug", "-d", help="set debug mode", is_flag=True, default=False)
@click.option(
    "--oci-config",
    help="OCI Config file location. Defaults to ~/.oci",
    default=None,
    required=False,
)
def generate_docker_compose(file, output, **kwargs):
    """Generates docker-compose.yml from ADS distributed training YAML specification."""
    debug = kwargs["debug"]
    if os.path.exists(file):
        if output is None:
            output = os.path.abspath(
                os.path.join(os.path.dirname(file), "docker-compose.yml")
            )
        with open(file, "r", encoding="utf-8") as f:
            config = suppress_traceback(debug)(yaml.safe_load)(f.read())
            generate_docker_compose_yaml(config, output, **kwargs)
    else:
        raise FileNotFoundError(f"{file} is not found")
