# -*- coding: utf-8 -*-
# Copyright 2022 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
from typing import Callable, Dict, Optional, Sequence, Tuple, Union
import warnings

from google.api_core import gapic_v1, grpc_helpers, operations_v1
import google.auth  # type: ignore
from google.auth import credentials as ga_credentials  # type: ignore
from google.auth.transport.grpc import SslCredentials  # type: ignore
from google.longrunning import operations_pb2  # type: ignore
import grpc  # type: ignore

from google.cloud.tpu_v2alpha1.types import cloud_tpu

from .base import DEFAULT_CLIENT_INFO, TpuTransport


class TpuGrpcTransport(TpuTransport):
    """gRPC backend transport for Tpu.

    Manages TPU nodes and other resources
    TPU API v2alpha1

    This class defines the same methods as the primary client, so the
    primary client can load the underlying transport implementation
    and call it.

    It sends protocol buffers over the wire using gRPC (which is built on
    top of HTTP/2); the ``grpcio`` package must be installed.
    """

    _stubs: Dict[str, Callable]

    def __init__(
        self,
        *,
        host: str = "tpu.googleapis.com",
        credentials: ga_credentials.Credentials = None,
        credentials_file: str = None,
        scopes: Sequence[str] = None,
        channel: grpc.Channel = None,
        api_mtls_endpoint: str = None,
        client_cert_source: Callable[[], Tuple[bytes, bytes]] = None,
        ssl_channel_credentials: grpc.ChannelCredentials = None,
        client_cert_source_for_mtls: Callable[[], Tuple[bytes, bytes]] = None,
        quota_project_id: Optional[str] = None,
        client_info: gapic_v1.client_info.ClientInfo = DEFAULT_CLIENT_INFO,
        always_use_jwt_access: Optional[bool] = False,
        api_audience: Optional[str] = None,
    ) -> None:
        """Instantiate the transport.

        Args:
            host (Optional[str]):
                 The hostname to connect to.
            credentials (Optional[google.auth.credentials.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify the application to the service; if none
                are specified, the client will attempt to ascertain the
                credentials from the environment.
                This argument is ignored if ``channel`` is provided.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is ignored if ``channel`` is provided.
            scopes (Optional(Sequence[str])): A list of scopes. This argument is
                ignored if ``channel`` is provided.
            channel (Optional[grpc.Channel]): A ``Channel`` instance through
                which to make calls.
            api_mtls_endpoint (Optional[str]): Deprecated. The mutual TLS endpoint.
                If provided, it overrides the ``host`` argument and tries to create
                a mutual TLS channel with client SSL credentials from
                ``client_cert_source`` or application default SSL credentials.
            client_cert_source (Optional[Callable[[], Tuple[bytes, bytes]]]):
                Deprecated. A callback to provide client SSL certificate bytes and
                private key bytes, both in PEM format. It is ignored if
                ``api_mtls_endpoint`` is None.
            ssl_channel_credentials (grpc.ChannelCredentials): SSL credentials
                for the grpc channel. It is ignored if ``channel`` is provided.
            client_cert_source_for_mtls (Optional[Callable[[], Tuple[bytes, bytes]]]):
                A callback to provide client certificate bytes and private key bytes,
                both in PEM format. It is used to configure a mutual TLS channel. It is
                ignored if ``channel`` or ``ssl_channel_credentials`` is provided.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            client_info (google.api_core.gapic_v1.client_info.ClientInfo):
                The client info used to send a user-agent string along with
                API requests. If ``None``, then default info will be used.
                Generally, you only need to set this if you're developing
                your own client library.
            always_use_jwt_access (Optional[bool]): Whether self signed JWT should
                be used for service account credentials.

        Raises:
          google.auth.exceptions.MutualTLSChannelError: If mutual TLS transport
              creation failed for any reason.
          google.api_core.exceptions.DuplicateCredentialArgs: If both ``credentials``
              and ``credentials_file`` are passed.
        """
        self._grpc_channel = None
        self._ssl_channel_credentials = ssl_channel_credentials
        self._stubs: Dict[str, Callable] = {}
        self._operations_client: Optional[operations_v1.OperationsClient] = None

        if api_mtls_endpoint:
            warnings.warn("api_mtls_endpoint is deprecated", DeprecationWarning)
        if client_cert_source:
            warnings.warn("client_cert_source is deprecated", DeprecationWarning)

        if channel:
            # Ignore credentials if a channel was passed.
            credentials = False
            # If a channel was explicitly provided, set it.
            self._grpc_channel = channel
            self._ssl_channel_credentials = None

        else:
            if api_mtls_endpoint:
                host = api_mtls_endpoint

                # Create SSL credentials with client_cert_source or application
                # default SSL credentials.
                if client_cert_source:
                    cert, key = client_cert_source()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )
                else:
                    self._ssl_channel_credentials = SslCredentials().ssl_credentials

            else:
                if client_cert_source_for_mtls and not ssl_channel_credentials:
                    cert, key = client_cert_source_for_mtls()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )

        # The base transport sets the host, credentials and scopes
        super().__init__(
            host=host,
            credentials=credentials,
            credentials_file=credentials_file,
            scopes=scopes,
            quota_project_id=quota_project_id,
            client_info=client_info,
            always_use_jwt_access=always_use_jwt_access,
            api_audience=api_audience,
        )

        if not self._grpc_channel:
            self._grpc_channel = type(self).create_channel(
                self._host,
                # use the credentials which are saved
                credentials=self._credentials,
                # Set ``credentials_file`` to ``None`` here as
                # the credentials that we saved earlier should be used.
                credentials_file=None,
                scopes=self._scopes,
                ssl_credentials=self._ssl_channel_credentials,
                quota_project_id=quota_project_id,
                options=[
                    ("grpc.max_send_message_length", -1),
                    ("grpc.max_receive_message_length", -1),
                ],
            )

        # Wrap messages. This must be done after self._grpc_channel exists
        self._prep_wrapped_messages(client_info)

    @classmethod
    def create_channel(
        cls,
        host: str = "tpu.googleapis.com",
        credentials: ga_credentials.Credentials = None,
        credentials_file: str = None,
        scopes: Optional[Sequence[str]] = None,
        quota_project_id: Optional[str] = None,
        **kwargs,
    ) -> grpc.Channel:
        """Create and return a gRPC channel object.
        Args:
            host (Optional[str]): The host for the channel to use.
            credentials (Optional[~.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify this application to the service. If
                none are specified, the client will attempt to ascertain
                the credentials from the environment.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is mutually exclusive with credentials.
            scopes (Optional[Sequence[str]]): A optional list of scopes needed for this
                service. These are only used when credentials are not specified and
                are passed to :func:`google.auth.default`.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            kwargs (Optional[dict]): Keyword arguments, which are passed to the
                channel creation.
        Returns:
            grpc.Channel: A gRPC channel object.

        Raises:
            google.api_core.exceptions.DuplicateCredentialArgs: If both ``credentials``
              and ``credentials_file`` are passed.
        """

        return grpc_helpers.create_channel(
            host,
            credentials=credentials,
            credentials_file=credentials_file,
            quota_project_id=quota_project_id,
            default_scopes=cls.AUTH_SCOPES,
            scopes=scopes,
            default_host=cls.DEFAULT_HOST,
            **kwargs,
        )

    @property
    def grpc_channel(self) -> grpc.Channel:
        """Return the channel designed to connect to this service."""
        return self._grpc_channel

    @property
    def operations_client(self) -> operations_v1.OperationsClient:
        """Create the client designed to process long-running operations.

        This property caches on the instance; repeated calls return the same
        client.
        """
        # Quick check: Only create a new client if we do not already have one.
        if self._operations_client is None:
            self._operations_client = operations_v1.OperationsClient(self.grpc_channel)

        # Return the client from cache.
        return self._operations_client

    @property
    def list_nodes(
        self,
    ) -> Callable[[cloud_tpu.ListNodesRequest], cloud_tpu.ListNodesResponse]:
        r"""Return a callable for the list nodes method over gRPC.

        Lists nodes.

        Returns:
            Callable[[~.ListNodesRequest],
                    ~.ListNodesResponse]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_nodes" not in self._stubs:
            self._stubs["list_nodes"] = self.grpc_channel.unary_unary(
                "/google.cloud.tpu.v2alpha1.Tpu/ListNodes",
                request_serializer=cloud_tpu.ListNodesRequest.serialize,
                response_deserializer=cloud_tpu.ListNodesResponse.deserialize,
            )
        return self._stubs["list_nodes"]

    @property
    def get_node(self) -> Callable[[cloud_tpu.GetNodeRequest], cloud_tpu.Node]:
        r"""Return a callable for the get node method over gRPC.

        Gets the details of a node.

        Returns:
            Callable[[~.GetNodeRequest],
                    ~.Node]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_node" not in self._stubs:
            self._stubs["get_node"] = self.grpc_channel.unary_unary(
                "/google.cloud.tpu.v2alpha1.Tpu/GetNode",
                request_serializer=cloud_tpu.GetNodeRequest.serialize,
                response_deserializer=cloud_tpu.Node.deserialize,
            )
        return self._stubs["get_node"]

    @property
    def create_node(
        self,
    ) -> Callable[[cloud_tpu.CreateNodeRequest], operations_pb2.Operation]:
        r"""Return a callable for the create node method over gRPC.

        Creates a node.

        Returns:
            Callable[[~.CreateNodeRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "create_node" not in self._stubs:
            self._stubs["create_node"] = self.grpc_channel.unary_unary(
                "/google.cloud.tpu.v2alpha1.Tpu/CreateNode",
                request_serializer=cloud_tpu.CreateNodeRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["create_node"]

    @property
    def delete_node(
        self,
    ) -> Callable[[cloud_tpu.DeleteNodeRequest], operations_pb2.Operation]:
        r"""Return a callable for the delete node method over gRPC.

        Deletes a node.

        Returns:
            Callable[[~.DeleteNodeRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "delete_node" not in self._stubs:
            self._stubs["delete_node"] = self.grpc_channel.unary_unary(
                "/google.cloud.tpu.v2alpha1.Tpu/DeleteNode",
                request_serializer=cloud_tpu.DeleteNodeRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["delete_node"]

    @property
    def stop_node(
        self,
    ) -> Callable[[cloud_tpu.StopNodeRequest], operations_pb2.Operation]:
        r"""Return a callable for the stop node method over gRPC.

        Stops a node. This operation is only available with
        single TPU nodes.

        Returns:
            Callable[[~.StopNodeRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "stop_node" not in self._stubs:
            self._stubs["stop_node"] = self.grpc_channel.unary_unary(
                "/google.cloud.tpu.v2alpha1.Tpu/StopNode",
                request_serializer=cloud_tpu.StopNodeRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["stop_node"]

    @property
    def start_node(
        self,
    ) -> Callable[[cloud_tpu.StartNodeRequest], operations_pb2.Operation]:
        r"""Return a callable for the start node method over gRPC.

        Starts a node.

        Returns:
            Callable[[~.StartNodeRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "start_node" not in self._stubs:
            self._stubs["start_node"] = self.grpc_channel.unary_unary(
                "/google.cloud.tpu.v2alpha1.Tpu/StartNode",
                request_serializer=cloud_tpu.StartNodeRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["start_node"]

    @property
    def update_node(
        self,
    ) -> Callable[[cloud_tpu.UpdateNodeRequest], operations_pb2.Operation]:
        r"""Return a callable for the update node method over gRPC.

        Updates the configurations of a node.

        Returns:
            Callable[[~.UpdateNodeRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "update_node" not in self._stubs:
            self._stubs["update_node"] = self.grpc_channel.unary_unary(
                "/google.cloud.tpu.v2alpha1.Tpu/UpdateNode",
                request_serializer=cloud_tpu.UpdateNodeRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["update_node"]

    @property
    def generate_service_identity(
        self,
    ) -> Callable[
        [cloud_tpu.GenerateServiceIdentityRequest],
        cloud_tpu.GenerateServiceIdentityResponse,
    ]:
        r"""Return a callable for the generate service identity method over gRPC.

        Generates the Cloud TPU service identity for the
        project.

        Returns:
            Callable[[~.GenerateServiceIdentityRequest],
                    ~.GenerateServiceIdentityResponse]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "generate_service_identity" not in self._stubs:
            self._stubs["generate_service_identity"] = self.grpc_channel.unary_unary(
                "/google.cloud.tpu.v2alpha1.Tpu/GenerateServiceIdentity",
                request_serializer=cloud_tpu.GenerateServiceIdentityRequest.serialize,
                response_deserializer=cloud_tpu.GenerateServiceIdentityResponse.deserialize,
            )
        return self._stubs["generate_service_identity"]

    @property
    def list_accelerator_types(
        self,
    ) -> Callable[
        [cloud_tpu.ListAcceleratorTypesRequest], cloud_tpu.ListAcceleratorTypesResponse
    ]:
        r"""Return a callable for the list accelerator types method over gRPC.

        Lists accelerator types supported by this API.

        Returns:
            Callable[[~.ListAcceleratorTypesRequest],
                    ~.ListAcceleratorTypesResponse]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_accelerator_types" not in self._stubs:
            self._stubs["list_accelerator_types"] = self.grpc_channel.unary_unary(
                "/google.cloud.tpu.v2alpha1.Tpu/ListAcceleratorTypes",
                request_serializer=cloud_tpu.ListAcceleratorTypesRequest.serialize,
                response_deserializer=cloud_tpu.ListAcceleratorTypesResponse.deserialize,
            )
        return self._stubs["list_accelerator_types"]

    @property
    def get_accelerator_type(
        self,
    ) -> Callable[[cloud_tpu.GetAcceleratorTypeRequest], cloud_tpu.AcceleratorType]:
        r"""Return a callable for the get accelerator type method over gRPC.

        Gets AcceleratorType.

        Returns:
            Callable[[~.GetAcceleratorTypeRequest],
                    ~.AcceleratorType]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_accelerator_type" not in self._stubs:
            self._stubs["get_accelerator_type"] = self.grpc_channel.unary_unary(
                "/google.cloud.tpu.v2alpha1.Tpu/GetAcceleratorType",
                request_serializer=cloud_tpu.GetAcceleratorTypeRequest.serialize,
                response_deserializer=cloud_tpu.AcceleratorType.deserialize,
            )
        return self._stubs["get_accelerator_type"]

    @property
    def list_runtime_versions(
        self,
    ) -> Callable[
        [cloud_tpu.ListRuntimeVersionsRequest], cloud_tpu.ListRuntimeVersionsResponse
    ]:
        r"""Return a callable for the list runtime versions method over gRPC.

        Lists runtime versions supported by this API.

        Returns:
            Callable[[~.ListRuntimeVersionsRequest],
                    ~.ListRuntimeVersionsResponse]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_runtime_versions" not in self._stubs:
            self._stubs["list_runtime_versions"] = self.grpc_channel.unary_unary(
                "/google.cloud.tpu.v2alpha1.Tpu/ListRuntimeVersions",
                request_serializer=cloud_tpu.ListRuntimeVersionsRequest.serialize,
                response_deserializer=cloud_tpu.ListRuntimeVersionsResponse.deserialize,
            )
        return self._stubs["list_runtime_versions"]

    @property
    def get_runtime_version(
        self,
    ) -> Callable[[cloud_tpu.GetRuntimeVersionRequest], cloud_tpu.RuntimeVersion]:
        r"""Return a callable for the get runtime version method over gRPC.

        Gets a runtime version.

        Returns:
            Callable[[~.GetRuntimeVersionRequest],
                    ~.RuntimeVersion]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_runtime_version" not in self._stubs:
            self._stubs["get_runtime_version"] = self.grpc_channel.unary_unary(
                "/google.cloud.tpu.v2alpha1.Tpu/GetRuntimeVersion",
                request_serializer=cloud_tpu.GetRuntimeVersionRequest.serialize,
                response_deserializer=cloud_tpu.RuntimeVersion.deserialize,
            )
        return self._stubs["get_runtime_version"]

    @property
    def get_guest_attributes(
        self,
    ) -> Callable[
        [cloud_tpu.GetGuestAttributesRequest], cloud_tpu.GetGuestAttributesResponse
    ]:
        r"""Return a callable for the get guest attributes method over gRPC.

        Retrieves the guest attributes for the node.

        Returns:
            Callable[[~.GetGuestAttributesRequest],
                    ~.GetGuestAttributesResponse]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_guest_attributes" not in self._stubs:
            self._stubs["get_guest_attributes"] = self.grpc_channel.unary_unary(
                "/google.cloud.tpu.v2alpha1.Tpu/GetGuestAttributes",
                request_serializer=cloud_tpu.GetGuestAttributesRequest.serialize,
                response_deserializer=cloud_tpu.GetGuestAttributesResponse.deserialize,
            )
        return self._stubs["get_guest_attributes"]

    def close(self):
        self.grpc_channel.close()

    @property
    def kind(self) -> str:
        return "grpc"


__all__ = ("TpuGrpcTransport",)
