/**
 * Modified from nbdime
 * https://github.com/jupyter/nbdime/blob/master/packages/labextension/src/widget.ts
 */
import { Panel, Widget } from '@lumino/widgets';
import jsonMap from 'json-source-map';
import { NotebookDiffModel } from 'nbdime/lib/diff/model';
import { generateNode, requestAPI } from '../../utils';
import { NotebookCellsDiff } from './NotebookCellsDiff';
/**
 * Class of the outermost widget, the draggable tab
 */
const NBDIME_CLASS = 'nbdime-Widget';
/**
 * Class of the root of the actual diff, the scroller element
 */
const ROOT_CLASS = 'nbdime-root';
export class NotebookDiff extends Panel {
    constructor(props) {
        super();
        this.addClass(NBDIME_CLASS);
        this.scroller = new Panel();
        this.scroller.addClass(ROOT_CLASS);
        this.scroller.node.tabIndex = -1;
        const header = Private.diffHeader(props.diff.base.label, props.diff.head.label);
        this.scroller.addWidget(header);
        this.addWidget(this.scroller);
        this.computeDiff(props.diff.base.content, props.diff.head.content)
            .then(data => {
            this.onData(props.pullRequestId, props.filename, data, props.renderMime, props.threads);
        })
            .catch(error => {
            this.onError(error);
        });
    }
    async computeDiff(previousContent, currentContent) {
        const data = await requestAPI('pullrequests/files/nbdiff', 'POST', {
            previousContent,
            currentContent
        });
        data['baseMapping'] = Private.computeNotebookMapping(previousContent || '{}');
        data['headMapping'] = Private.computeNotebookMapping(currentContent || '{}');
        return data;
    }
    dispose() {
        this.scroller.dispose();
        super.dispose();
    }
    static mapThreadsOnChunks(baseMapping, headMapping, chunks, threads) {
        var _a, _b, _c, _d;
        // Last element will be for the notebook metadata
        const threadsByChunk = new Array(chunks.length + 1);
        for (let index = 0; index < threadsByChunk.length; index++) {
            threadsByChunk[index] = {
                threads: new Array()
            };
        }
        // Sort thread by line and originalLine order
        const sortedThreads = [...threads].sort((a, b) => {
            if (a.line !== null && b.line !== null) {
                return a.line - b.line;
            }
            if (a.originalLine !== null && b.originalLine !== null) {
                return a.originalLine - b.originalLine;
            }
            return 0;
        });
        let lastBaseCell = -1;
        let lastHeadCell = -1;
        let lastThread = -1;
        // Handle thread set before the cells
        let thread;
        do {
            lastThread += 1;
            thread = sortedThreads[lastThread];
        } while (lastThread < sortedThreads.length &&
            thread.line < headMapping.cells[0].start &&
            thread.originalLine < baseMapping.cells[0].start);
        if (lastThread > 0) {
            // There are thread before the cells
            // They will be added on the metadata diff
            threadsByChunk[threadsByChunk.length - 1].threads = sortedThreads.splice(0, lastThread);
        }
        // Handle threads on cells
        for (let chunkIndex = 0; chunkIndex < chunks.length; chunkIndex++) {
            const chunk = chunks[chunkIndex];
            for (const cellDiff of chunk) {
                let inThisBaseChunk = true;
                let inThisHeadChunk = true;
                let currentThread = 0;
                if (cellDiff.source.base !== null) {
                    lastBaseCell += 1;
                    threadsByChunk[chunkIndex].originalRange =
                        baseMapping.cells[lastBaseCell];
                }
                if (cellDiff.source.remote !== null) {
                    lastHeadCell += 1;
                    threadsByChunk[chunkIndex].range = headMapping.cells[lastHeadCell];
                }
                while ((inThisBaseChunk || inThisHeadChunk) &&
                    currentThread < sortedThreads.length) {
                    const thread = sortedThreads[currentThread];
                    if (cellDiff.source.base !== null) {
                        const baseRange = threadsByChunk[chunkIndex].originalRange;
                        if (((_a = baseRange) === null || _a === void 0 ? void 0 : _a.start) <= thread.originalLine - 1 &&
                            thread.originalLine - 1 <= ((_b = baseRange) === null || _b === void 0 ? void 0 : _b.end)) {
                            threadsByChunk[chunkIndex].threads.push(...sortedThreads.splice(currentThread, 1));
                            continue;
                        }
                        else {
                            inThisBaseChunk = false;
                        }
                    }
                    if (cellDiff.source.remote !== null) {
                        const headRange = threadsByChunk[chunkIndex].range;
                        if (((_c = headRange) === null || _c === void 0 ? void 0 : _c.start) <= thread.line - 1 &&
                            thread.line - 1 <= ((_d = headRange) === null || _d === void 0 ? void 0 : _d.end)) {
                            threadsByChunk[chunkIndex].threads.push(...sortedThreads.splice(currentThread, 1));
                            continue;
                        }
                        else {
                            inThisHeadChunk = false;
                        }
                    }
                    currentThread++;
                }
            }
        }
        // Handle remaining threads
        if (lastThread < sortedThreads.length) {
            // There are thread after the cells
            // They will be added on the metadata diff
            threadsByChunk[threadsByChunk.length - 1].threads.push(...sortedThreads.slice(lastThread, sortedThreads.length));
        }
        threadsByChunk[threadsByChunk.length - 1].range = headMapping.metadata;
        threadsByChunk[threadsByChunk.length - 1].originalRange =
            baseMapping.metadata;
        return threadsByChunk;
    }
    /**
     * Handle `'activate-request'` messages.
     */
    onActivateRequest(msg) {
        this.scroller.node.focus();
    }
    /**
     * Callback on diff and discussions requests
     *
     * @param pullRequestId Pull request ID
     * @param filename Notebook filename
     * @param data Notebook diff raw data
     * @param renderMime Rendermime registry
     * @param threads List of discussion on the file
     */
    onData(pullRequestId, filename, data, renderMime, threads) {
        if (this.isDisposed) {
            return;
        }
        const base = data['base'];
        const diff = data['diff'];
        const model = new NotebookDiffModel(base, diff);
        const comments = NotebookDiff.mapThreadsOnChunks(data.baseMapping, data.headMapping, NotebookDiff.reChunkCells(model.chunkedCells), threads);
        const nbdWidget = new NotebookCellsDiff({
            pullRequestId,
            filename,
            model,
            comments,
            renderMime
        });
        this.scroller.addWidget(nbdWidget);
        nbdWidget.init().catch(error => {
            console.error('Failed to mark unchanged ranges', error);
        });
    }
    /**
     * Callback on error when requesting the diff or the discussions
     *
     * @param error Error
     */
    onError(error) {
        if (this.isDisposed) {
            return;
        }
        console.error(error);
        const widget = new Widget();
        widget.node.innerHTML = `<h2 class="jp-PullRequestTabError">
    <span style="color: 'var(--jp-ui-font-color1)';">
      Error Loading File:
    </span> ${error.message}</h2>`;
        this.scroller.addWidget(widget);
    }
    /**
     * Change cell grouping to allow commenting on each cell
     *
     * @param chunks Cell chunks from nbdime
     * @returns New chunks
     */
    static reChunkCells(chunks) {
        const newChunks = [];
        for (const chunk of chunks) {
            // If chunk is unmodified, push it to stack
            if (chunk.length === 1 && !(chunk[0].added || chunk[0].deleted)) {
                newChunks.push([chunk[0]]);
            }
            else {
                let modifiedPair = [null, null];
                for (const cell of chunk) {
                    if (cell.deleted) {
                        // if 'removed' not in chunk, add to chunk
                        if (modifiedPair[0] === null) {
                            modifiedPair[0] = cell;
                        }
                        // if 'removed' already in chunk, push chunk to chunks and start new chunk
                        else {
                            newChunks.push(modifiedPair.filter(item => item !== null));
                            modifiedPair = [cell, null];
                        }
                    }
                    else {
                        // if 'added' not in chunk, add to chunk
                        if (modifiedPair[1] === null) {
                            modifiedPair[1] = cell;
                        }
                        // if 'added' already in chunk, push chunk to chunks and start new chunk
                        else {
                            newChunks.push(modifiedPair.filter(item => item !== null));
                            modifiedPair = [null, cell];
                        }
                    }
                }
                // if nonempty at end, push the remaining pair
                if (modifiedPair[0] !== null || modifiedPair[1] !== null) {
                    newChunks.push(modifiedPair.filter(item => item !== null));
                }
            }
        }
        return newChunks;
    }
}
var Private;
(function (Private) {
    /**
     * Map cell index with their position in the JSON file
     *
     * @param content Notebook file content
     */
    function computeNotebookMapping(content) {
        var _a, _b;
        const parsedContent = jsonMap.parse(content);
        return {
            metadata: {
                start: (_a = parsedContent.pointers['/metadata']) === null || _a === void 0 ? void 0 : _a.key.line,
                end: (_b = parsedContent.pointers['/metadata']) === null || _b === void 0 ? void 0 : _b.valueEnd.line
            },
            cells: (parsedContent.data.cells || []).map((cell, index) => {
                return {
                    start: parsedContent.pointers[`/cells/${index}`].value.line,
                    end: parsedContent.pointers[`/cells/${index}`].valueEnd.line
                };
            })
        };
    }
    Private.computeNotebookMapping = computeNotebookMapping;
    /**
     * Create a header widget for the diff view.
     */
    function diffHeader(baseLabel, remoteLabel) {
        const node = generateNode('div', { class: 'jp-git-diff-banner' });
        node.innerHTML = `<span>${baseLabel}</span>
        <span>${remoteLabel}</span>`;
        return new Widget({ node });
    }
    Private.diffHeader = diffHeader;
})(Private || (Private = {}));
//# sourceMappingURL=NotebookDiff.js.map