import { ReactWidget } from '@jupyterlab/apputils';
import React, { useEffect, useState } from 'react';
import { BeatLoader } from 'react-spinners';
import { requestAPI } from '../utils';
import { Browser } from './browser/Browser';
import { Toolbar } from './Toolbar';
/**
 * React wrapper to mount and umount the React child component
 * when the widget is shown/hidden.
 *
 * In this case this is particularly interesting to trigger the
 * useEffect of the React widget to update the pull requests list
 * each time the user come back to the panel.
 */
export class PullRequestPanelWrapper extends ReactWidget {
    constructor(commands, docRegistry) {
        super();
        this._commands = commands;
        this._docRegistry = docRegistry;
    }
    onBeforeShow(msg) {
        super.onBeforeShow(msg);
        this.update();
    }
    onBeforeHide(msg) {
        super.onBeforeHide(msg);
        this.onBeforeDetach(msg);
    }
    render() {
        return (this.isAttached &&
            this.isVisible && (React.createElement(PullRequestPanel, { commands: this._commands, docRegistry: this._docRegistry })));
    }
}
/**
 * Get a group of pull requests for each filters
 *
 * @param filters Filter types
 * @returns The group of pull requests
 */
async function fetchPullRequests(filters) {
    return Promise.all(filters.map(async (filter) => {
        var _a;
        try {
            const pullRequests = await requestAPI('pullrequests/prs/user?filter=' + filter.filter, 'GET');
            return {
                name: filter.name,
                pullRequests
            };
        }
        catch (err) {
            let error = 'Unknown Error';
            if (((_a = err.response) === null || _a === void 0 ? void 0 : _a.status) && err.message) {
                error = `${err.response.status} (${err.message})`;
            }
            return {
                name: filter.name,
                pullRequests: [],
                error
            };
        }
    }));
}
/**
 * PullRequestPanel component
 *
 * @param props PullRequestPanel properties
 */
export function PullRequestPanel(props) {
    const [pullRequests, setPullRequests] = useState([]);
    const [isLoading, setIsLoading] = useState(true);
    const refreshPullRequests = () => {
        setIsLoading(true);
        fetchPullRequests([
            {
                name: 'Created by Me',
                filter: 'created'
            },
            {
                name: 'Assigned to Me',
                filter: 'assigned'
            }
        ])
            .then(data => {
            setPullRequests(data);
            setIsLoading(false);
        })
            .catch(reason => {
            console.error('Failed to fetch pull requests', reason);
            setPullRequests([]);
            setIsLoading(false);
        });
    };
    useEffect(refreshPullRequests, []);
    return (React.createElement("div", { className: "jp-PullRequestPanel" },
        React.createElement(Toolbar, { onRefresh: refreshPullRequests }),
        isLoading ? (React.createElement(BeatLoader, { sizeUnit: 'px', size: 5, color: 'var(--jp-ui-font-color1)', loading: isLoading })) : (React.createElement(Browser, { docRegistry: props.docRegistry, commands: props.commands, prGroups: pullRequests }))));
}
//# sourceMappingURL=PullRequestPanel.js.map