# --- __init__.py ---

"""RomPy
   =====
   
   Provides
     1. Functionality to build problem-specific greedy algorithms
     2. Reduced basis representations of function spaces
     3. Empirical interpolation methods
     4. Reduced-order quadrature rules
     5. Empirical Fourier transforms
   
   How to use the documentation
   -----------------------------
   Documentation is available via docstrings provided with the code.
   
   We recommend exploring the docstrings using 
   `IPython <http://ipython.scipy.org>`, an advanced Python shell with 
   TAB-completion and introspection capabilities. See below for further 
   instructions.
   
   The docstring examples assume that `rompy` has been imported as `rp`::
   
     >>> import rompy as rp
   
   Code snippets are indicated by three greater-than signs.
   
   Use the built-in ``help`` function to view a function's or class's 
   docstring::
   
     >>> help(rp.greedy)
   
   Available subpackages
   ---------------------
   eim
       Core empirical interpolation tools
   
   fits
   
   greedy
       Core greedy algorithm tools
   
   integrals
       Tools for quadrature rules and integration
       
   lib
       Basic functions used by several subpackages
   
   surrogate
   
   training
       Core training set generation tools
   
   Viewing documentation using IPython
   -----------------------------------
   Start IPython and load RomPy (``import rompy as rp``), which 
   will import `rompy` under the alias `rp`. Then, use the ``cpaste`` 
   command to paste examples into the shell. To see which functions 
   are available in `rompy`, type ``rp.<TAB>`` (where ``<TAB>`` refers 
   to the TAB key), or use ``rp.greedy?<ENTER>`` (where ``<ENTER>``
   refers to the ENTER key) to narrow down the list. To view the 
   docstring for a function or class, use ``rp.greedy?<ENTER>`` (to 
   view the docstring) and ``rp.greedy??<ENTER>`` (to view the source 
   code).
   """

__copyright__ = "Copyright (C) 2013 Chad R. Galley"
__email__ = "crgalley `at` tapir `dot` caltech `dot` edu"
__author__ = "Chad Galley"

# TODO: Add contributors

import numpy as np
import time
import random

from .data import *
from .integrals import *
from .derivatives import D, logD
from .greedy import *
from .eim import *
from .lib import *
from .fits import *
from .fits import EvaluateFits
from .univar import *
from .multivar import *
from .surrogate import *
from .training import *



class _TruthData(object):
  """
  Generate the truth data used as an example for demonstrating 
  the construction of various interpolants in the notebooks/fits.ipynb
  IPython tutorial.
  """
  
  def __init__(self, noise=None, uv=None):
    # Generate test data
    self.x = np.linspace(-1, 1, 4001)
    self.y = self.f(self.x, noise=noise, uv=uv)
  
  def f(self, x, noise=None, uv=None):
    """Univariate function to sample for pedagogical purposes"""
    
    # Validate inputs
    assert None in [noise, uv], "One or both of `noise` and `uv` must be None."
    x = np.asarray(x)
    
    # Return smooth function values
    smooth = 100.*( (x+1.)*np.sin(5.*(x-0.2)**2) + np.exp(-(x-0.5)**2/2./0.01)*np.sin(100*x) )
    if noise is None and uv is None:
      return smooth
    
    # Return smooth function values with high-frequency (UV) features
    elif uv is not None:
      assert type(uv) in [float, int], "Expecting integer or float type."
      return smooth + uv*self.uv(x)
    
    # Return smooth function values with stochastic noise
    elif noise is not None:
      assert type(noise) in [float, int], "Expecting integer or float type."
      return smooth + noise*np.random.randn(len(x))
  
  def uv(self, x, width=20):
    """High-frequency oscillations for adding to smooth data generated by `f` method"""
    X = x[width] - x[0]
    return np.sin(len(x)/X * x)
