/*
 * Decompiled with CFR 0.152.
 */
package androidx.test.tools.crawler.postprocessing.logcat;

import androidx.test.tools.crawler.postprocessing.logcat.LogLine;
import androidx.test.tools.crawler.postprocessing.logcat.LogcatFinding;
import androidx.test.tools.crawler.postprocessing.logcat.LogcatParser;
import androidx.test.tools.crawler.postprocessing.logcat.TestCaseTiming;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import com.google.common.flogger.GoogleLogger;
import java.io.BufferedReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.joda.time.Instant;

public class LogcatAnalyser {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    private static final int MAX_STACK_TRACE_LINES = 10000;
    private static final String DEFAULT_ERROR_MESSAGE = "Fatal AndroidRuntime Exception detected.";
    private static final String ANDROID_RUNTIME = "AndroidRuntime";
    private static final String MONITORING_INSTR = "MonitoringInstr";
    private static final String NON_SDK_API_USAGE_STRICTMODE_MESSSAGE = "StrictMode policy violation:";
    private static final Pattern CALLER_SIGNATURE_EXTRACTION_PATTERN = Pattern.compile("StrictMode policy violation: .*\\n\\tat android.os.StrictMode.*\\n\\tat android.os.*\\n(\\tat java.lang.Class.*\\n)+\\tat (?<caller>.*)\\n");
    private Set<Integer> suppressedCrashPids = new HashSet<Integer>();
    private List<LogLine> logLines = new ArrayList<LogLine>();
    private static final long CONTINUED_LONG_ENTRY_DELTA_IN_MS = 100L;

    private static boolean hasFatalException(LogLine logLine) {
        return logLine.tag().equals(ANDROID_RUNTIME) && logLine.message().trim().startsWith("FATAL EXCEPTION");
    }

    private static boolean hasMonitoringInstrException(LogLine logLine) {
        return logLine.tag().equals(MONITORING_INSTR) && logLine.message().trim().endsWith("Dumping thread state to outputs and pining for the fjords.");
    }

    public ImmutableList<LogcatFinding> processLogcat(BufferedReader logcat) throws IOException {
        return this.processLogcat(logcat, true);
    }

    public ImmutableList<LogcatFinding> processLogcat(BufferedReader logcat, boolean logUnparseable) throws IOException {
        this.suppressedCrashPids.clear();
        LogcatParser parser = new LogcatParser(logcat);
        this.logLines = parser.parse();
        if (logUnparseable) {
            for (String line : parser.getUnparseableLines()) {
                ((GoogleLogger.Api)logger.atWarning()).log("Logcat line cannot be parsed: %s", line);
            }
        }
        List<LogEvent> logEvents = this.formLogEvents(this.logLines);
        ImmutableList.Builder findings = ImmutableList.builder();
        for (LogEvent event : logEvents) {
            com.google.common.base.Optional<LogcatFinding> optionalFinding = this.analyseLogEvent(event);
            if (!optionalFinding.isPresent()) continue;
            findings.add(optionalFinding.get());
        }
        com.google.common.base.Optional<Object> instrumentationTimeline = com.google.common.base.Optional.absent();
        try {
            instrumentationTimeline = this.findInstrumentationTimeline(this.logLines);
        }
        catch (RuntimeException e) {
            ((GoogleLogger.Api)((GoogleLogger.Api)logger.atWarning()).withCause(e)).log("Unable to handle instrumentation time line");
        }
        if (instrumentationTimeline.isPresent()) {
            findings.add((LogcatFinding)instrumentationTimeline.get());
        }
        return findings.build();
    }

    public ImmutableList<LogLine> getLogLines() {
        return ImmutableList.copyOf(this.logLines);
    }

    private com.google.common.base.Optional<LogcatFinding> analyseLogEvent(LogEvent event) {
        LogLine processDied;
        int crashedProcessPid;
        com.google.common.base.Optional processDiedLine;
        LogLine lastLine = event.getLastLine();
        com.google.common.base.Optional startupTimeLine = Iterables.tryFind(event.lines, LogLine::startupDefaultLine);
        if (startupTimeLine.isPresent()) {
            LogLine startupTime = (LogLine)startupTimeLine.get();
            return com.google.common.base.Optional.of(LogcatFinding.createStartupTimeFinding(LogcatFinding.LogcatFindingType.STARTUP_TIME_DEFAULT, lastLine.processId(), startupTime.message(), startupTime.timeStamp(), startupTime.lineNumber(), startupTime.startupTime().get()));
        }
        com.google.common.base.Optional fullyDrawnTimeLine = Iterables.tryFind(event.lines, LogLine::startupFullyDrawnLine);
        if (fullyDrawnTimeLine.isPresent()) {
            LogLine fullyDrawnTime = (LogLine)fullyDrawnTimeLine.get();
            return com.google.common.base.Optional.of(LogcatFinding.createStartupTimeFinding(LogcatFinding.LogcatFindingType.STARTUP_TIME_FULLY_DRAWN, lastLine.processId(), fullyDrawnTime.message(), fullyDrawnTime.timeStamp(), fullyDrawnTime.lineNumber(), fullyDrawnTime.startupTime().get()));
        }
        if (Arrays.asList(LogLine.LogSeverity.ERROR, LogLine.LogSeverity.FATAL).contains((Object)event.getSeverity()) && event.anyMatch(((Predicate<LogLine>)LogcatAnalyser::hasFatalException).or(LogcatAnalyser::hasMonitoringInstrException))) {
            LogcatFinding.LogcatFindingType type;
            Optional<LogLine> processLine = event.lines.stream().filter(LogLine::processLine).findFirst();
            com.google.common.base.Optional<String> packageName = processLine.isPresent() ? processLine.get().packageName() : com.google.common.base.Optional.absent();
            List stackTraceLines = event.lines.stream().filter(LogLine::stackTraceLine).collect(Collectors.toList());
            LogcatFinding.LogcatFindingType logcatFindingType = type = event.getTag().equals(MONITORING_INSTR) ? LogcatFinding.LogcatFindingType.MONITORING_INSTR_EXCEPTION : LogcatFinding.LogcatFindingType.FATAL_EXCEPTION;
            if (!stackTraceLines.isEmpty()) {
                int indexOfStackTrace = event.lines.indexOf(stackTraceLines.get(0));
                int errorMessageIndex = Math.max(0, indexOfStackTrace - 1);
                String message = ((LogLine)event.lines.get(errorMessageIndex)).message();
                String stackTrace = LogcatAnalyser.getStackTrace(event.lines);
                return com.google.common.base.Optional.of(LogcatFinding.createCrashFinding(type, event.getProcessId(), packageName, message, stackTrace, lastLine.timeStamp(), lastLine.lineNumber()));
            }
            String stackTrace = LogcatAnalyser.getStackTrace(event.lines);
            return com.google.common.base.Optional.of(LogcatFinding.createCrashFinding(type, event.getProcessId(), packageName, DEFAULT_ERROR_MESSAGE, stackTrace, lastLine.timeStamp(), lastLine.lineNumber()));
        }
        com.google.common.base.Optional nativeCrashLine = Iterables.tryFind(event.lines, LogLine::nativeCrashLine);
        if (nativeCrashLine.isPresent()) {
            LogLine crash = (LogLine)nativeCrashLine.get();
            String stackTrace = LogcatAnalyser.getStackTrace(event.lines);
            String errorMessage = "Native crash of " + crash.packageName().get();
            return com.google.common.base.Optional.of(LogcatFinding.createCrashFinding(LogcatFinding.LogcatFindingType.NATIVE_CRASH, crash.targetProcessId().get(), crash.packageName(), errorMessage, stackTrace, crash.timeStamp(), crash.lineNumber()));
        }
        com.google.common.base.Optional roboTestCompleteLine = Iterables.tryFind(event.lines, LogLine::roboTestCompleteLine);
        if (roboTestCompleteLine.isPresent()) {
            LogLine roboTestComplete = (LogLine)roboTestCompleteLine.get();
            return com.google.common.base.Optional.of(LogcatFinding.createRoboTestFinding(LogcatFinding.LogcatFindingType.ROBO_TEST_COMPLETED, roboTestComplete.timeStamp(), roboTestComplete.lineNumber()));
        }
        com.google.common.base.Optional suppressedCrashLine = Iterables.tryFind(event.lines, LogLine::suppressedCrashLine);
        if (suppressedCrashLine.isPresent()) {
            LogLine suppressedCrash = (LogLine)suppressedCrashLine.get();
            this.suppressedCrashPids.add(suppressedCrash.processId());
        }
        if ((processDiedLine = Iterables.tryFind(event.lines, LogLine::processDiedLine)).isPresent() && this.suppressedCrashPids.contains(crashedProcessPid = (processDied = (LogLine)processDiedLine.get()).targetProcessId().get().intValue())) {
            return com.google.common.base.Optional.of(LogcatFinding.createCrashFinding(LogcatFinding.LogcatFindingType.NATIVE_CRASH, crashedProcessPid, processDied.packageName(), processDied.message(), "", processDied.timeStamp(), processDied.lineNumber()));
        }
        com.google.common.base.Optional lowMemoryKillerLine = Iterables.tryFind(event.lines, LogLine::lowMemoryKillerLine);
        if (lowMemoryKillerLine.isPresent()) {
            LogLine crash = (LogLine)lowMemoryKillerLine.get();
            String stackTrace = FluentIterable.from(event.lines).transform(LogLine::message).join(Joiner.on('\n'));
            return com.google.common.base.Optional.of(LogcatFinding.createCrashFinding(LogcatFinding.LogcatFindingType.LOW_MEMORY_KILLED, crash.processId(), crash.packageName(), crash.message(), stackTrace, crash.timeStamp(), crash.lineNumber()));
        }
        com.google.common.base.Optional optionalNonSdkApiUsedViolationLine = Iterables.tryFind(event.lines, LogLine::nonSdkApiUsedViolationLine);
        if (optionalNonSdkApiUsedViolationLine.isPresent()) {
            LogLine nonSdkApiUsedViolationLine = (LogLine)optionalNonSdkApiUsedViolationLine.get();
            String stackTrace = LogcatAnalyser.getStackTrace(event.lines);
            Matcher matcher = CALLER_SIGNATURE_EXTRACTION_PATTERN.matcher(stackTrace);
            com.google.common.base.Optional<String> callerName = matcher.find() ? com.google.common.base.Optional.of(matcher.group("caller")) : com.google.common.base.Optional.absent();
            return com.google.common.base.Optional.of(LogcatFinding.createNonSdkApiUsedFinding(nonSdkApiUsedViolationLine.processId(), nonSdkApiUsedViolationLine.message(), stackTrace, nonSdkApiUsedViolationLine.timeStamp(), nonSdkApiUsedViolationLine.lineNumber(), nonSdkApiUsedViolationLine.nonSdkApiSignature().get(), callerName));
        }
        com.google.common.base.Optional optionalStartProcLine = Iterables.tryFind(event.lines, LogLine::startProcLine);
        if (optionalStartProcLine.isPresent()) {
            LogLine startProcLine = (LogLine)optionalStartProcLine.get();
            return com.google.common.base.Optional.of(LogcatFinding.createStartProcFinding(startProcLine.targetProcessId().get(), startProcLine.packageName().get(), startProcLine.timeStamp(), startProcLine.lineNumber()));
        }
        return com.google.common.base.Optional.absent();
    }

    private com.google.common.base.Optional<LogcatFinding> findInstrumentationTimeline(List<LogLine> lines) {
        ImmutableList.Builder instrumentationTimeline = ImmutableList.builder();
        HashMap testStartedLines = new HashMap();
        HashMap testFinishedLines = new HashMap();
        try {
            lines.stream().filter(LogLine::instrumentationTestStartedLine).forEach(line -> testStartedLines.putIfAbsent(LogcatAnalyser.toFullTestCaseName(line), (LogLine)line));
            if (testStartedLines.isEmpty()) {
                return com.google.common.base.Optional.absent();
            }
            lines.stream().filter(LogLine::instrumentationTestFinishedLine).forEach(line -> testFinishedLines.put(LogcatAnalyser.toFullTestCaseName(line), line));
        }
        catch (IllegalArgumentException e) {
            ((GoogleLogger.Api)((GoogleLogger.Api)logger.atInfo()).withCause(e)).log("Have duplicated methods in instrumentation test.");
            return com.google.common.base.Optional.absent();
        }
        ImmutableList<String> allTestCases = ImmutableList.sortedCopyOf(String::compareTo, Sets.union(testStartedLines.keySet(), testFinishedLines.keySet()));
        for (String fullTestCaseName : allTestCases) {
            LogLine testLogLine = testStartedLines.containsKey(fullTestCaseName) ? (LogLine)testStartedLines.get(fullTestCaseName) : (LogLine)testFinishedLines.get(fullTestCaseName);
            com.google.common.base.Optional<Instant> startTime = testStartedLines.containsKey(fullTestCaseName) ? com.google.common.base.Optional.of(((LogLine)testStartedLines.get(fullTestCaseName)).timeStamp()) : com.google.common.base.Optional.absent();
            com.google.common.base.Optional<Instant> endTime = testFinishedLines.containsKey(fullTestCaseName) ? com.google.common.base.Optional.of(((LogLine)testFinishedLines.get(fullTestCaseName)).timeStamp()) : com.google.common.base.Optional.absent();
            instrumentationTimeline.add(TestCaseTiming.create(startTime, endTime, testLogLine.sourceClass().get(), testLogLine.sourceMethod().get()));
        }
        LogLine firstLine = testStartedLines.getOrDefault(allTestCases.get(0), (LogLine)testFinishedLines.get(allTestCases.get(0)));
        return com.google.common.base.Optional.of(LogcatFinding.createInstrumentationTimelineFinding(LogcatFinding.LogcatFindingType.INSTRUMENTATION_TIMELINE, firstLine.processId(), firstLine.timeStamp(), firstLine.lineNumber(), (ImmutableList<TestCaseTiming>)instrumentationTimeline.build()));
    }

    private static String toFullTestCaseName(LogLine logLine) {
        Preconditions.checkArgument(logLine.sourceClass().isPresent());
        Preconditions.checkArgument(logLine.sourceMethod().isPresent());
        return logLine.sourceClass().get() + "#" + logLine.sourceMethod().get();
    }

    private List<LogEvent> formLogEvents(List<LogLine> lines) throws IOException {
        HashMap<Integer, LogEvent> previousEventByLoggingPid = new HashMap<Integer, LogEvent>();
        ArrayList<LogEvent> finishedEvents = new ArrayList<LogEvent>();
        for (LogLine currentLine : lines) {
            LogEvent logEvent = (LogEvent)previousEventByLoggingPid.get(currentLine.processId());
            if (logEvent != null) {
                LogLine previousLine = logEvent.getLastLine();
                if (LogcatAnalyser.isContinuedLineOf(previousLine, currentLine)) {
                    logEvent.appendLine(currentLine);
                    continue;
                }
                finishedEvents.add(logEvent);
                previousEventByLoggingPid.put(currentLine.processId(), new LogEvent(currentLine));
                continue;
            }
            previousEventByLoggingPid.put(currentLine.processId(), new LogEvent(currentLine));
        }
        finishedEvents.addAll(previousEventByLoggingPid.values());
        Collections.sort(finishedEvents, LogEvent.ORDER_BY_LINE_NUMBER);
        return finishedEvents;
    }

    private static boolean isContinuedLineOf(LogLine prevLine, LogLine nextLine) {
        if (nextLine == null) {
            return false;
        }
        if (prevLine.type() == LogLine.LogMessageType.STARTUP_TIME_DEFAULT || prevLine.type() == LogLine.LogMessageType.STARTUP_TIME_FULLY_DRAWN) {
            return false;
        }
        if (nextLine.message().startsWith(NON_SDK_API_USAGE_STRICTMODE_MESSSAGE)) {
            return false;
        }
        if (prevLine.processId() != nextLine.processId()) {
            return false;
        }
        if (prevLine.severity() != nextLine.severity()) {
            return false;
        }
        if (!prevLine.tag().equals(nextLine.tag())) {
            return false;
        }
        long timeStampDelta = nextLine.timeStamp().getMillis() - prevLine.timeStamp().getMillis();
        return timeStampDelta <= 100L;
    }

    private static String getStackTrace(List<LogLine> lines) {
        return FluentIterable.from(lines).limit(10000).transform(LogLine::message).join(Joiner.on('\n'));
    }

    static class LogEvent {
        static final Comparator<LogEvent> ORDER_BY_LINE_NUMBER = new Comparator<LogEvent>(){

            @Override
            public int compare(LogEvent event1, LogEvent event2) {
                return event1.getFirstLine().lineNumber() - event2.getFirstLine().lineNumber();
            }
        };
        private final LinkedList<LogLine> lines = new LinkedList();

        LogEvent(LogLine first) {
            this.lines.add(first);
        }

        LogLine getFirstLine() {
            return this.lines.getFirst();
        }

        LogLine getLastLine() {
            return this.lines.getLast();
        }

        void appendLine(LogLine line) {
            this.lines.add(line);
        }

        int getProcessId() {
            return this.getFirstLine().processId();
        }

        LogLine.LogSeverity getSeverity() {
            return this.getFirstLine().severity();
        }

        String getTag() {
            return this.getFirstLine().tag();
        }

        boolean anyMatch(Predicate<LogLine> predicate) {
            return this.lines.stream().anyMatch(predicate);
        }

        public String toString() {
            return "Event (" + this.lines.size() + "): " + this.lines;
        }
    }
}

