"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UrlShortener = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const logs = require("aws-cdk-lib/aws-logs");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
const redirect_function_1 = require("./redirect-function");
const shortener_function_1 = require("./shortener-function");
/**
 * URL shortener
 */
class UrlShortener extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        const domainName = props.recordName ? `${props.recordName}.${props.hostedZone.zoneName}` : props.hostedZone.zoneName;
        // Table to save a counter
        const table = new dynamodb.Table(this, 'Table', {
            partitionKey: {
                name: 'key',
                type: dynamodb.AttributeType.STRING,
            },
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        // Bucket to save redirects
        const bucket = new s3.Bucket(this, 'Bucket', {
            lifecycleRules: [{
                    expiration: (_b = props.expiration) !== null && _b !== void 0 ? _b : aws_cdk_lib_1.Duration.days(365),
                }],
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            bucketName: (_c = props.bucketName) !== null && _c !== void 0 ? _c : `cloudstructs-url-shortener-${domainName}`,
        });
        // Redirect function
        const redirectFunction = new redirect_function_1.RedirectFunction(this, 'Redirect');
        bucket.grantRead(redirectFunction);
        // CloudFront distribution
        const certificate = new acm.DnsValidatedCertificate(this, 'Certificate', {
            domainName,
            hostedZone: props.hostedZone,
            region: 'us-east-1',
        });
        const distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: {
                origin: new origins.S3Origin(bucket),
                edgeLambdas: [
                    {
                        eventType: cloudfront.LambdaEdgeEventType.ORIGIN_REQUEST,
                        functionVersion: redirectFunction,
                    },
                ],
            },
            certificate,
            domainNames: [domainName],
        });
        // Route53 records
        new route53.ARecord(this, 'ARecord', {
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distribution)),
            recordName: props.recordName,
        });
        new route53.AaaaRecord(this, 'AaaaRecord', {
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distribution)),
            recordName: props.recordName,
        });
        // Lambda function to increment counter and write redirect in bucket
        const shortenerFunction = new shortener_function_1.ShortenerFunction(this, 'Shortener', {
            logRetention: logs.RetentionDays.ONE_MONTH,
            environment: {
                DOMAIN_NAME: domainName,
                BUCKET_NAME: bucket.bucketName,
                TABLE_NAME: table.tableName,
            },
        });
        if (props.corsAllowOrigins) {
            shortenerFunction.addEnvironment('CORS_ALLOW_ORIGINS', props.corsAllowOrigins.join(' '));
        }
        bucket.grantPut(shortenerFunction);
        table.grant(shortenerFunction, 'dynamodb:UpdateItem');
        // API
        this.api = new apigateway.RestApi(this, `UrlShortener${props.hostedZone.zoneName}`, {
            endpointTypes: props.apiGatewayEndpoint ? [apigateway.EndpointType.PRIVATE] : undefined,
            policy: props.apiGatewayEndpoint
                ? new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            effect: iam.Effect.ALLOW,
                            actions: ['execute-api:Invoke'],
                            principals: [new iam.AnyPrincipal()],
                            resources: [aws_cdk_lib_1.Fn.join('', ['execute-api:/', '*'])],
                            conditions: {
                                StringEquals: { 'aws:SourceVpce': props.apiGatewayEndpoint.vpcEndpointId },
                            },
                        }),
                    ],
                })
                : undefined,
            defaultCorsPreflightOptions: props.corsAllowOrigins
                ? { allowOrigins: props.corsAllowOrigins }
                : undefined,
        });
        this.api.root.addMethod('ANY', new apigateway.LambdaIntegration(shortenerFunction), {
            authorizer: props.apiGatewayAuthorizer,
        });
        this.api.root
            .addResource('{proxy+}')
            .addMethod('ANY', new apigateway.LambdaIntegration(shortenerFunction), {
            authorizer: props.apiGatewayAuthorizer,
        });
        this.apiEndpoint = this.api.url;
    }
}
exports.UrlShortener = UrlShortener;
_a = JSII_RTTI_SYMBOL_1;
UrlShortener[_a] = { fqn: "cloudstructs.UrlShortener", version: "0.5.1" };
//# sourceMappingURL=data:application/json;base64,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