"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrowApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const lambda = require("aws-cdk-lib/aws-lambda");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const logs = require("aws-cdk-lib/aws-logs");
/**
 * For copying shared code to all paths
 */
const fse = require("fs-extra");
const DEFAULT_LAMBDA_CODE = `
exports.handler = async function (event, context, callback) {
  try {
    const data = {
      statusCode: 201,
    };
    return data;
  } catch (uncaughtError) {
    console.error(uncaughtError);
    throw uncaughtError;
  }
}
`;
/**
 * @experimental
 */
class CrowApi extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Pulling out props
        const { sourceDirectory = 'src', sharedDirectory = 'shared', useAuthorizerLambda = false, authorizerDirectory = 'authorizer', authorizerLambdaConfiguration = {}, tokenAuthorizerConfiguration = {}, createApiKey = false, logRetention = logs.RetentionDays.ONE_WEEK, apiGatewayConfiguration = {}, lambdaConfigurations = {}, methodConfigurations = {}, } = props;
        // Initializing constants
        const LAMBDA_RUNTIME = lambda.Runtime.NODEJS_14_X;
        const SPECIAL_DIRECTORIES = [
            sharedDirectory,
            authorizerDirectory,
        ];
        // Helpers functions for constructor
        // Prepares default Lambda props and overrides them with user input
        function bundleLambdaProps(codePath, userConfiguration, sharedLayer) {
            let layers;
            if (sharedLayer) {
                const { layers: userLayers = [], } = userConfiguration;
                layers = [sharedLayer, ...userLayers];
            }
            const defaultProps = {
                runtime: LAMBDA_RUNTIME,
                code: lambda.Code.fromAsset(codePath),
                handler: 'index.handler',
                logRetention,
            };
            const lambdaProps = {
                ...defaultProps,
                ...userConfiguration,
                layers,
            };
            return lambdaProps;
        }
        // Returns child directories given the path of a parent
        function getDirectoryChildren(parentDirectory) {
            try {
                const directories = fse.readdirSync(parentDirectory, { withFileTypes: true })
                    .filter((dirent) => dirent.isDirectory())
                    .map((dirent) => dirent.name);
                return directories;
            }
            catch {
                /**
                 * The only time I have run into this was when the src/ directory
                 * was empty.
                 * If it is empty, let CDK tree validation tell user that the
                 * REST API does not have any methods.
                 */
            }
            return [];
        }
        // A default Lambda function is needed for the API Gateway
        const defaultLambda = new lambda.Function(this, 'default-crow-lambda', {
            runtime: LAMBDA_RUNTIME,
            code: new lambda.InlineCode(DEFAULT_LAMBDA_CODE),
            handler: 'index.handler',
            logRetention,
        });
        // API Gateway log group
        const gatewayLogGroup = new logs.LogGroup(this, 'api-access-logs', {
            retention: logs.RetentionDays.ONE_WEEK,
        });
        // The API Gateway itself
        const gateway = new apigateway.LambdaRestApi(this, 'api-gateway', {
            handler: defaultLambda,
            proxy: false,
            deploy: true,
            deployOptions: {
                loggingLevel: apigateway.MethodLoggingLevel.ERROR,
                accessLogDestination: new apigateway.LogGroupLogDestination(gatewayLogGroup),
            },
            apiKeySourceType: createApiKey ? apigateway.ApiKeySourceType.HEADER : undefined,
            ...apiGatewayConfiguration,
        });
        // Create API key if desired
        if (createApiKey) {
            const apiKey = gateway.addApiKey('api-key');
            const usagePlan = new apigateway.UsagePlan(this, 'usage-plan', {
                throttle: {
                    burstLimit: 5000,
                    rateLimit: 10000,
                },
                apiStages: [
                    {
                        api: gateway,
                        stage: gateway.deploymentStage,
                    },
                ],
            });
            usagePlan.addApiKey(apiKey);
        }
        // Create Lambda layer out of shared directory if it exists
        const sourceSharedDirectory = `${sourceDirectory}/${sharedDirectory}`;
        let sharedLayer;
        if (fse.existsSync(sourceSharedDirectory)) {
            sharedLayer = new lambda.LayerVersion(this, 'shared-layer', {
                code: lambda.Code.fromAsset(sourceSharedDirectory),
                compatibleRuntimes: [LAMBDA_RUNTIME],
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            this.lambdaLayer = sharedLayer;
        }
        // Create Lambda authorizer to be used in subsequent Methods
        let tokenAuthorizer;
        if (useAuthorizerLambda) {
            const fullAuthorizerDirectory = `${sourceDirectory}/${authorizerDirectory}`;
            const authorizerLambdaProps = bundleLambdaProps(fullAuthorizerDirectory, authorizerLambdaConfiguration, sharedLayer);
            const authorizerLambda = new lambda.Function(this, 'authorizer-lambda', authorizerLambdaProps);
            this.authorizerLambda = authorizerLambda;
            const bundledTokenAuthConfig = {
                handler: authorizerLambda,
                resultsCacheTtl: cdk.Duration.seconds(3600),
                ...tokenAuthorizerConfiguration,
            };
            tokenAuthorizer = new apigateway.TokenAuthorizer(this, 'token-authorizer', bundledTokenAuthConfig);
        }
        // Time to start walking the directories
        const root = sourceDirectory;
        const verbs = ['get', 'post', 'put', 'delete'];
        const graph = {};
        const lambdasByPath = {};
        // Initialize with root
        graph['/'] = {
            resource: gateway.root,
            path: root,
            paths: [],
            verbs: [],
        };
        // First element in tuple is directory path, second is API path
        const nodes = [[root, '/']];
        // BFS that creates API Gateway structure using addMethod
        while (nodes.length) {
            // The `|| ['type', 'script']` piece is needed or TS throws a fit
            const [directoryPath, apiPath] = nodes.shift() || ['type', 'script'];
            const children = getDirectoryChildren(directoryPath);
            // For debugging purposes
            // console.log(`${apiPath}'s children are: ${children}`);
            // Don't have to worry about previously visited nodes
            // since this is a file structure
            // ...unless there are symlinks? Haven't run into that
            children.forEach((child) => {
                const newDirectoryPath = `${directoryPath}/${child}`;
                // If we're on the root path, don't separate with a slash (/)
                //   because it ends up looking like //child-path
                const newApiPath = apiPath === '/' ? `/${child}` : `${apiPath}/${child}`;
                if (verbs.includes(child)) {
                    // If directory is a verb, we don't traverse it anymore
                    //   and need to create an API Gateway method and Lambda
                    const userLambdaConfiguration = lambdaConfigurations[newApiPath]
                        || {};
                    const lambdaProps = bundleLambdaProps(newDirectoryPath, userLambdaConfiguration, sharedLayer);
                    const newLambda = new lambda.Function(this, newDirectoryPath, lambdaProps);
                    // Pull out useAuthorizerLambda value
                    const { useAuthorizerLambda: authorizerLambdaConfigured, ...userMethodConfiguration } = methodConfigurations[newApiPath] || {};
                    let methodConfiguration = userMethodConfiguration;
                    // If this method should be behind an authorizer Lambda
                    //   construct the methodConfiguration object as such
                    if (authorizerLambdaConfigured && useAuthorizerLambda) {
                        methodConfiguration = {
                            ...userMethodConfiguration,
                            authorizationType: apigateway.AuthorizationType.CUSTOM,
                            authorizer: tokenAuthorizer,
                        };
                    }
                    graph[apiPath].resource.addMethod(child.toUpperCase(), new apigateway.LambdaIntegration(newLambda), methodConfiguration);
                    graph[apiPath].verbs.push(child);
                    lambdasByPath[newApiPath] = newLambda;
                }
                else if (SPECIAL_DIRECTORIES.includes(child)) {
                    // The special directories should not result in an API path
                    // This means the API also cannot have a resource with the
                    //   same name
                }
                else {
                    // If directory is not a verb, create new API Gateway resource
                    //   for use by verb directory later
                    const newResource = graph[apiPath].resource
                        .resourceForPath(child);
                    nodes.push([newDirectoryPath, newApiPath]);
                    // Add child to parent's paths
                    graph[apiPath].paths.push(child);
                    // Initialize graph node to include child
                    graph[newApiPath] = {
                        resource: newResource,
                        path: newDirectoryPath,
                        paths: [],
                        verbs: [],
                    };
                }
            });
        }
        // For debugging purposes
        // console.log(graph);
        // Expose API Gateway
        this.gateway = gateway;
        this.lambdaFunctions = lambdasByPath;
    }
}
exports.CrowApi = CrowApi;
_a = JSII_RTTI_SYMBOL_1;
CrowApi[_a] = { fqn: "crow-api.CrowApi", version: "2.1.1" };
//# sourceMappingURL=data:application/json;base64,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