'''
# Crow API

Crow API lets you build an API intuitively based on the file structure of a project. Provide API Gateway and Lambda function configurations and crow will build out the appropriate paths and methods to the API Gateway. All created resources are available after initialization. `lambdaFunctions` will expose all Lambda functions created for further operations like adding environment variables and providing permissions.

| `crow-api` version | `aws-cdk` version | Notes                   |
| ------------------ | ----------------- | ----------------------- |
| 0                  | 1                 |                         |
| 1                  | 2                 | Not recommended for use |
| 2                  | 2                 |                         |

Contents:

* [Getting Started](#getting-started)
* [Example File Structure](#example-file-structure)
* [Crow API Props](#crow-api-props)

  * [`sourceDirectory`](#sourcedirectory)
  * [`sharedDirectory`](#shareddirectory)
  * [`useAuthorizerLambda`](#useauthorizerlambda)
  * [`authorizerDirectory`](#authorizerdirectory)
  * [`authorizerLambdaConfiguration`](#authorizerlambdaconfiguration)
  * [`tokenAuthorizerConfiguration`](#tokenauthorizerconfiguration)
  * [`createApiKey`](#createapikey)
  * [`logRetention`](#logretention)
  * [`apiGatewayConfiguration`](#apigatewayconfiguration)
  * [`lambdaConfigurations`](#lambdaconfigurations)
  * [`methodConfigurations`](#methodconfigurations)
* [Properties](#properties)

  * [`authorizerLambda`](#authorizerlambda)
  * [`gateway`](#gateway)
  * [`lambdaLayer`](#lambdalayer)
  * [`lambdaFunctions`](#lambdafunctions)

## Getting Started

[Start your application as a normal CDK app](https://docs.aws.amazon.com/cdk/latest/guide/getting_started.html)

```sh
npm install -g aws-cdk
cdk bootstrap # If this is your first cdk app, you will need to bootstrap your AWS account
cdk init app --language typescript
```

Next, install the Crow API package

```sh
npm install --save crow-api
```

In the `lib/` folder generated by the `cdk`, there should be a single file named `<your-app>-stack.js`. Create your Crow API construct inside of that file like so

```python
# Example automatically generated from non-compiling source. May contain errors.
from aws_cdk import Stack, StackProps
from constructs import Construct
from crow_api import CrowApi, ICrowApiProps

class YourAppStack(Stack):
    def __init__(self, scope, id, props):
        super().__init__(scope, id, props)

        {
              crowApiProps,
            } = props

        api = CrowApi(self, "api", {
            (SpreadAssignment ...crowApiProps
              crow_api_props)
        })
```

Your API will start to take shape as you create folders to define paths and methods (see Example File Structure below). To deploy your API, simply run `cdk synth` and `cdk deploy`. Follow the instructions as they are prompted, and you will end up receiving a URL where your API now lives.

## Example File Structure

```
|-- src/
    |-- authorizer/
        |-- index.js
    |-- v1/
        |-- book/
            |-- get/
                |-- index.js
            |-- post/
                |-- index.js
        |-- chapters/
            |-- get/
                |-- index.js
        |-- authors/
            |-- get/
                |-- index.js
            |-- post/
                |-- index.js
```

The preceding file structure will create an API with the following routes:

* GET /v1/book
* POST /v1/book
* GET /v1/book/chapters
* GET /v1/authors
* POST /v1/authors

There needs to be an `index.js` file inside of a folder named after an HTTP method in order for a path to be created. The `index.js` file needs to export a `handler` method that will process the payload and return like the following.

```javascript
exports.handler = async function (event, context, callback) {
  try {
    const data = {
      statusCode: 201,
    };
    return data;
  } catch (uncaughtError) {
    console.error(uncaughtError);
    throw uncaughtError;
  }
}
```

## Crow API Props

Crow API takes in a few props to help you customize away from defaults.

#### `sourceDirectory`

By default, Crow walks through the `src` directory in the root of the repository to determine routes and methods, but you can change the top level directory by passing in the `sourceDirectory` prop. The string passed in should not start with or end with a slash (`/`). For example, `src`, `api/src`, or `source` are all valid options to pass in through that prop.

#### `sharedDirectory`

By default, Crow creates a Lambda layer out of the `shared` directory in the source directory of the repository, but you can change the name of the shared directory by passing in the `sharedDirectory` prop. The string passed in should not start with or end with a slash (`/`) and must be a direct child of the source directory. For example, `common` or `utils` are valid but `shared/utils` is not.

The Lambda layer created will be prepended to any the of the layers passed in through `lambdaConfigurations` and added to all Lambda functions created.

#### `useAuthorizerLambda`

Crow will create and attach an authorizer Lambda to specific methods if requested. The `useAuthorizerLambda` prop tells the `CrowApi` Construct that it should create an authorizer Lambda and accepts a boolean value. This is `false` by default.

#### `authorizerDirectory`

Crow will allow for a Lambda authorizer to be created and used by specific methods if requested. The `authorizerDirectory` prop tells Crow where to find the code for the Lambda authorizer **within the source directory which can be specified in the `sourceDirectory` prop**. It expects to find an `index.js` file that exports a `handler` function within the `authorizerDirectory`.

By default, Crow expects to find a directory called `src/authorizer` containing the authorizer Lambda source if the `useAuthorizerLambda` prop is `true`. If a different directory within the source directory should be looked at for this code, it should be specified by passing in a string to the `authorizerDirectory` prop. The string passed in should not start with nor end with a slash (`/`). For example, `auth` or `authLambdaSrc` are valid.

#### `authorizerLambdaConfiguration`

The `authorizerLambdaConfiguration` prop is passed directly to the Lambda functions which will be in charge of your API's authorization. The configuration allowed is exactly the same as the [Lambda Function props](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-lambda.Function.html).

#### `tokenAuthorizerConfiguration`

The `tokenAuthorizerConfiguration` prop is passed directly to the `APIGateway.TokenAuthorizer` construct which will be in charge of your API's authorization. Anything available in the [class constructor for the `TokenAuthorizer`](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-apigateway.TokenAuthorizer.html) can be overridden.

**Note:**

Be careful with this configuration item as all configuration here takes precedence over Crow defaults. I suggest not using this configuration item unless you are experienced with the AWS CDK, API Gateway, and Lambda.

#### `createApiKey`

By default, Crow does not create an API key associated with the API. If an API key is desired, pass in the `createApiKey` prop as `true`.

#### `logRetention`

By default, Crow creates log groups for resources it creates and sets the log retention to one week. If a different retention is desired pass in the `logRetention` prop of [enum type `RetentionDays`](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-logs.RetentionDays.html).

#### `apiGatewayConfiguration`

This props allows for more complex overrides to the API Gateway that fronts your API. The configuration allowed is exactly the same as the [LambdaRestApi props](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-apigateway.LambdaRestApi.html).

**Note:**

Be careful with this configuration item as all configuration here takes precedence over Crow defaults. I suggest not using this configuration item unless you are experienced with the AWS CDK and API Gateway.

An example of this prop might look like the following:

```python
# Example automatically generated from non-compiling source. May contain errors.
from source_map_support.register import
import aws_cdk as cdk
import aws_cdk.aws_apigateway as apigateway
from ...lib.crow_api_stack import CrowApiStack

dev_environment = {
    "account": "123456789012",
    "region": "us-east-1"
}

app = cdk.App()

CrowApiStack(app, "CrowApiStack",
    env=dev_environment,
    api_gateway_configuration={
        "endpoint_configuration": {
            "types": [apigateway.EndpointType.REGIONAL]
        }
    }
)
```

#### `lambdaConfigurations`

This props allows for more complex overrides to Lambda functions. The prop is an object with keys corresponding to the API path of a Lambda function and a value corresponding to the configuration that should be applied to the Lambda. The configuration allowed is exactly the same as the [Lambda Function props](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-lambda.Function.html).

**Note:**

Be careful with this configuration item as all configuration here takes precedence over Crow defaults. I suggest not using this configuration item unless you are experienced with the AWS CDK and Lambda.

An example of this prop might look like the following:

```python
# Example automatically generated from non-compiling source. May contain errors.
from source_map_support.register import
import aws_cdk as cdk
from ...lib.crow_api_stack import CrowApiStack

dev_environment = {
    "account": "123456789012",
    "region": "us-east-1"
}

app = cdk.App()

CrowApiStack(app, "CrowApiStack",
    env=dev_environment,
    lambda_configurations={
        "/v1/book/get": {
            "timeout": cdk.Duration.seconds(5)
        }
    }
)
```

#### `methodConfigurations`

This props allows for more complex overrides to individual methods. The prop is an object with keys corresponding to the API path of a method and a value corresponding to the configuration that should be applied to the method as well as the key `useAuthorizerLambda` which will invoke the authorizer Lambda whenever the method is called. The configuration allowed is exactly the same as [`MethodOptions`](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_apigateway.MethodOptions.html) plus the `useAuthorizerLambda` boolean.

**Note:**

If `createApiKey` is `true`, then the `apiKeyRequired` parameter will need to be set for the methods needing the API key.

An example of this prop might look like the following:

```python
# Example automatically generated from non-compiling source. May contain errors.
from source_map_support.register import
import aws_cdk as cdk
from ...lib.crow_api_stack import CrowApiStack

dev_environment = {
    "account": "123456789012",
    "region": "us-east-1"
}

app = cdk.App()

CrowApiStack(app, "CrowApiStack",
    env=dev_environment,
    method_configurations={
        "/v1/book/get": {
            "use_authorizer_lambda": True
        },
        "/v1/book/post": {
            "api_key_required": True
        }
    }
)
```

## Properties

A `CrowApi` construct will give full access to all of the resources it created.

#### `authorizerLambda`

This is the `lambda.Function` that authorizes API Gateway requests.

#### `gateway`

This is the `apigateway.RestApi` that all of the created Lambda functions sit behind.

#### `lambdaLayer`

If the `sharedDirectory` is populated, this is the `lambda.LayerVersion` created for that code. If the `sharedDirectory` is not populated, then this is `undefined`.

#### `lambdaFunctions`

This is an object with keys being the API paths and the values being the `lambda.Function`s sitting being them. Continuing off of the example file structure from above, the following would be an example of referencing `GET` `/v1/book/chapters`.

```python
# Example automatically generated from non-compiling source. May contain errors.
from aws_cdk import Stack, StackProps
from constructs import Construct
from crow_api import CrowApi, ICrowApiProps

class YourAppStack(Stack):
    def __init__(self, scope, id, props):
        super().__init__(scope, id, props)

        {
              crowApiProps,
            } = props

        api = CrowApi(self, "api", {
            (SpreadAssignment ...crowApiProps
              crow_api_props)
        })

        lambda_ = api.lambda_functions["/v1/book/chapters/get"]
        lambda_.add_environment("FOO", "bar")
```
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from ._jsii import *

import aws_cdk.aws_apigateway
import aws_cdk.aws_lambda
import aws_cdk.aws_logs
import constructs


class CrowApi(
    constructs.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="crow-api.CrowApi",
):
    '''
    :stability: experimental
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        props: "ICrowApiProps",
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param props: -

        :stability: experimental
        '''
        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authorizerLambda")
    def authorizer_lambda(self) -> aws_cdk.aws_lambda.Function:
        '''
        :stability: experimental
        '''
        return typing.cast(aws_cdk.aws_lambda.Function, jsii.get(self, "authorizerLambda"))

    @authorizer_lambda.setter
    def authorizer_lambda(self, value: aws_cdk.aws_lambda.Function) -> None:
        jsii.set(self, "authorizerLambda", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="gateway")
    def gateway(self) -> aws_cdk.aws_apigateway.RestApi:
        '''
        :stability: experimental
        '''
        return typing.cast(aws_cdk.aws_apigateway.RestApi, jsii.get(self, "gateway"))

    @gateway.setter
    def gateway(self, value: aws_cdk.aws_apigateway.RestApi) -> None:
        jsii.set(self, "gateway", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="lambdaFunctions")
    def lambda_functions(self) -> "LambdasByPath":
        '''
        :stability: experimental
        '''
        return typing.cast("LambdasByPath", jsii.get(self, "lambdaFunctions"))

    @lambda_functions.setter
    def lambda_functions(self, value: "LambdasByPath") -> None:
        jsii.set(self, "lambdaFunctions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="lambdaLayer")
    def lambda_layer(self) -> typing.Optional[aws_cdk.aws_lambda.LayerVersion]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[aws_cdk.aws_lambda.LayerVersion], jsii.get(self, "lambdaLayer"))

    @lambda_layer.setter
    def lambda_layer(
        self,
        value: typing.Optional[aws_cdk.aws_lambda.LayerVersion],
    ) -> None:
        jsii.set(self, "lambdaLayer", value)


@jsii.data_type(
    jsii_type="crow-api.CrowLambdaConfigurations",
    jsii_struct_bases=[],
    name_mapping={},
)
class CrowLambdaConfigurations:
    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CrowLambdaConfigurations(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="crow-api.CrowMethodConfiguration",
    jsii_struct_bases=[aws_cdk.aws_apigateway.MethodOptions],
    name_mapping={
        "api_key_required": "apiKeyRequired",
        "authorization_scopes": "authorizationScopes",
        "authorization_type": "authorizationType",
        "authorizer": "authorizer",
        "method_responses": "methodResponses",
        "operation_name": "operationName",
        "request_models": "requestModels",
        "request_parameters": "requestParameters",
        "request_validator": "requestValidator",
        "request_validator_options": "requestValidatorOptions",
        "use_authorizer_lambda": "useAuthorizerLambda",
    },
)
class CrowMethodConfiguration(aws_cdk.aws_apigateway.MethodOptions):
    def __init__(
        self,
        *,
        api_key_required: typing.Optional[builtins.bool] = None,
        authorization_scopes: typing.Optional[typing.Sequence[builtins.str]] = None,
        authorization_type: typing.Optional[aws_cdk.aws_apigateway.AuthorizationType] = None,
        authorizer: typing.Optional[aws_cdk.aws_apigateway.IAuthorizer] = None,
        method_responses: typing.Optional[typing.Sequence[aws_cdk.aws_apigateway.MethodResponse]] = None,
        operation_name: typing.Optional[builtins.str] = None,
        request_models: typing.Optional[typing.Mapping[builtins.str, aws_cdk.aws_apigateway.IModel]] = None,
        request_parameters: typing.Optional[typing.Mapping[builtins.str, builtins.bool]] = None,
        request_validator: typing.Optional[aws_cdk.aws_apigateway.IRequestValidator] = None,
        request_validator_options: typing.Optional[aws_cdk.aws_apigateway.RequestValidatorOptions] = None,
        use_authorizer_lambda: builtins.bool,
    ) -> None:
        '''
        :param api_key_required: Indicates whether the method requires clients to submit a valid API key. Default: false
        :param authorization_scopes: A list of authorization scopes configured on the method. The scopes are used with a COGNITO_USER_POOLS authorizer to authorize the method invocation. Default: - no authorization scopes
        :param authorization_type: Method authorization. If the value is set of ``Custom``, an ``authorizer`` must also be specified. If you're using one of the authorizers that are available via the {@link Authorizer} class, such as {@link Authorizer#token()}, it is recommended that this option not be specified. The authorizer will take care of setting the correct authorization type. However, specifying an authorization type using this property that conflicts with what is expected by the {@link Authorizer} will result in an error. Default: - open access unless ``authorizer`` is specified
        :param authorizer: If ``authorizationType`` is ``Custom``, this specifies the ID of the method authorizer resource. If specified, the value of ``authorizationType`` must be set to ``Custom``
        :param method_responses: The responses that can be sent to the client who calls the method. Default: None This property is not required, but if these are not supplied for a Lambda proxy integration, the Lambda function must return a value of the correct format, for the integration response to be correctly mapped to a response to the client.
        :param operation_name: A friendly operation name for the method. For example, you can assign the OperationName of ListPets for the GET /pets method.
        :param request_models: The models which describe data structure of request payload. When combined with ``requestValidator`` or ``requestValidatorOptions``, the service will validate the API request payload before it reaches the API's Integration (including proxies). Specify ``requestModels`` as key-value pairs, with a content type (e.g. ``'application/json'``) as the key and an API Gateway Model as the value.
        :param request_parameters: The request parameters that API Gateway accepts. Specify request parameters as key-value pairs (string-to-Boolean mapping), with a source as the key and a Boolean as the value. The Boolean specifies whether a parameter is required. A source must match the format method.request.location.name, where the location is querystring, path, or header, and name is a valid, unique parameter name. Default: None
        :param request_validator: The ID of the associated request validator. Only one of ``requestValidator`` or ``requestValidatorOptions`` must be specified. Works together with ``requestModels`` or ``requestParameters`` to validate the request before it reaches integration like Lambda Proxy Integration. Default: - No default validator
        :param request_validator_options: Request validator options to create new validator Only one of ``requestValidator`` or ``requestValidatorOptions`` must be specified. Works together with ``requestModels`` or ``requestParameters`` to validate the request before it reaches integration like Lambda Proxy Integration. Default: - No default validator
        :param use_authorizer_lambda: 

        :stability: experimental
        '''
        if isinstance(request_validator_options, dict):
            request_validator_options = aws_cdk.aws_apigateway.RequestValidatorOptions(**request_validator_options)
        self._values: typing.Dict[str, typing.Any] = {
            "use_authorizer_lambda": use_authorizer_lambda,
        }
        if api_key_required is not None:
            self._values["api_key_required"] = api_key_required
        if authorization_scopes is not None:
            self._values["authorization_scopes"] = authorization_scopes
        if authorization_type is not None:
            self._values["authorization_type"] = authorization_type
        if authorizer is not None:
            self._values["authorizer"] = authorizer
        if method_responses is not None:
            self._values["method_responses"] = method_responses
        if operation_name is not None:
            self._values["operation_name"] = operation_name
        if request_models is not None:
            self._values["request_models"] = request_models
        if request_parameters is not None:
            self._values["request_parameters"] = request_parameters
        if request_validator is not None:
            self._values["request_validator"] = request_validator
        if request_validator_options is not None:
            self._values["request_validator_options"] = request_validator_options

    @builtins.property
    def api_key_required(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether the method requires clients to submit a valid API key.

        :default: false
        '''
        result = self._values.get("api_key_required")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def authorization_scopes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of authorization scopes configured on the method.

        The scopes are used with
        a COGNITO_USER_POOLS authorizer to authorize the method invocation.

        :default: - no authorization scopes

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-authorizationscopes
        '''
        result = self._values.get("authorization_scopes")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def authorization_type(
        self,
    ) -> typing.Optional[aws_cdk.aws_apigateway.AuthorizationType]:
        '''Method authorization. If the value is set of ``Custom``, an ``authorizer`` must also be specified.

        If you're using one of the authorizers that are available via the {@link Authorizer} class, such as {@link Authorizer#token()},
        it is recommended that this option not be specified. The authorizer will take care of setting the correct authorization type.
        However, specifying an authorization type using this property that conflicts with what is expected by the {@link Authorizer}
        will result in an error.

        :default: - open access unless ``authorizer`` is specified
        '''
        result = self._values.get("authorization_type")
        return typing.cast(typing.Optional[aws_cdk.aws_apigateway.AuthorizationType], result)

    @builtins.property
    def authorizer(self) -> typing.Optional[aws_cdk.aws_apigateway.IAuthorizer]:
        '''If ``authorizationType`` is ``Custom``, this specifies the ID of the method authorizer resource.

        If specified, the value of ``authorizationType`` must be set to ``Custom``
        '''
        result = self._values.get("authorizer")
        return typing.cast(typing.Optional[aws_cdk.aws_apigateway.IAuthorizer], result)

    @builtins.property
    def method_responses(
        self,
    ) -> typing.Optional[typing.List[aws_cdk.aws_apigateway.MethodResponse]]:
        '''The responses that can be sent to the client who calls the method.

        :default:

        None

        This property is not required, but if these are not supplied for a Lambda
        proxy integration, the Lambda function must return a value of the correct format,
        for the integration response to be correctly mapped to a response to the client.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-method-settings-method-response.html
        '''
        result = self._values.get("method_responses")
        return typing.cast(typing.Optional[typing.List[aws_cdk.aws_apigateway.MethodResponse]], result)

    @builtins.property
    def operation_name(self) -> typing.Optional[builtins.str]:
        '''A friendly operation name for the method.

        For example, you can assign the
        OperationName of ListPets for the GET /pets method.
        '''
        result = self._values.get("operation_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def request_models(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, aws_cdk.aws_apigateway.IModel]]:
        '''The models which describe data structure of request payload.

        When
        combined with ``requestValidator`` or ``requestValidatorOptions``, the service
        will validate the API request payload before it reaches the API's Integration (including proxies).
        Specify ``requestModels`` as key-value pairs, with a content type
        (e.g. ``'application/json'``) as the key and an API Gateway Model as the value.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-method-settings-method-request.html#setup-method-request-model

        Example::

            # Example automatically generated from non-compiling source. May contain errors.
            # api is of type apigateway.RestApi
            # user_lambda is of type lambda.Function
            
            user_model = api.add_model("UserModel",
                schema={
                    "type": apigateway.JsonSchemaType.OBJECT,
                    "properties": {
                        "user_id": {
                            "type": apigateway.JsonSchemaType.STRING
                        },
                        "name": {
                            "type": apigateway.JsonSchemaType.STRING
                        }
                    },
                    "required": ["userId"]
                }
            )
            api.root.add_resource("user").add_method("POST",
                apigateway.LambdaIntegration(user_lambda),
                    request_models={
                        "application/json": user_model
                    }
                )
        '''
        result = self._values.get("request_models")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, aws_cdk.aws_apigateway.IModel]], result)

    @builtins.property
    def request_parameters(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.bool]]:
        '''The request parameters that API Gateway accepts.

        Specify request parameters
        as key-value pairs (string-to-Boolean mapping), with a source as the key and
        a Boolean as the value. The Boolean specifies whether a parameter is required.
        A source must match the format method.request.location.name, where the location
        is querystring, path, or header, and name is a valid, unique parameter name.

        :default: None
        '''
        result = self._values.get("request_parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.bool]], result)

    @builtins.property
    def request_validator(
        self,
    ) -> typing.Optional[aws_cdk.aws_apigateway.IRequestValidator]:
        '''The ID of the associated request validator.

        Only one of ``requestValidator`` or ``requestValidatorOptions`` must be specified.
        Works together with ``requestModels`` or ``requestParameters`` to validate
        the request before it reaches integration like Lambda Proxy Integration.

        :default: - No default validator
        '''
        result = self._values.get("request_validator")
        return typing.cast(typing.Optional[aws_cdk.aws_apigateway.IRequestValidator], result)

    @builtins.property
    def request_validator_options(
        self,
    ) -> typing.Optional[aws_cdk.aws_apigateway.RequestValidatorOptions]:
        '''Request validator options to create new validator Only one of ``requestValidator`` or ``requestValidatorOptions`` must be specified.

        Works together with ``requestModels`` or ``requestParameters`` to validate
        the request before it reaches integration like Lambda Proxy Integration.

        :default: - No default validator
        '''
        result = self._values.get("request_validator_options")
        return typing.cast(typing.Optional[aws_cdk.aws_apigateway.RequestValidatorOptions], result)

    @builtins.property
    def use_authorizer_lambda(self) -> builtins.bool:
        '''
        :stability: experimental
        '''
        result = self._values.get("use_authorizer_lambda")
        assert result is not None, "Required property 'use_authorizer_lambda' is missing"
        return typing.cast(builtins.bool, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CrowMethodConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="crow-api.CrowMethodConfigurations",
    jsii_struct_bases=[],
    name_mapping={},
)
class CrowMethodConfigurations:
    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CrowMethodConfigurations(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.interface(jsii_type="crow-api.ICrowApiProps")
class ICrowApiProps(typing_extensions.Protocol):
    '''
    :stability: experimental
    '''

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="apiGatewayConfiguration")
    def api_gateway_configuration(self) -> typing.Any:
        '''
        :stability: experimental
        '''
        ...

    @api_gateway_configuration.setter
    def api_gateway_configuration(self, value: typing.Any) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authorizerDirectory")
    def authorizer_directory(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        ...

    @authorizer_directory.setter
    def authorizer_directory(self, value: typing.Optional[builtins.str]) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authorizerLambdaConfiguration")
    def authorizer_lambda_configuration(self) -> typing.Any:
        '''
        :stability: experimental
        '''
        ...

    @authorizer_lambda_configuration.setter
    def authorizer_lambda_configuration(self, value: typing.Any) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="createApiKey")
    def create_api_key(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        ...

    @create_api_key.setter
    def create_api_key(self, value: typing.Optional[builtins.bool]) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="lambdaConfigurations")
    def lambda_configurations(self) -> typing.Optional[CrowLambdaConfigurations]:
        '''
        :stability: experimental
        '''
        ...

    @lambda_configurations.setter
    def lambda_configurations(
        self,
        value: typing.Optional[CrowLambdaConfigurations],
    ) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="logRetention")
    def log_retention(self) -> typing.Optional[aws_cdk.aws_logs.RetentionDays]:
        '''
        :stability: experimental
        '''
        ...

    @log_retention.setter
    def log_retention(
        self,
        value: typing.Optional[aws_cdk.aws_logs.RetentionDays],
    ) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="methodConfigurations")
    def method_configurations(self) -> typing.Optional[CrowMethodConfigurations]:
        '''
        :stability: experimental
        '''
        ...

    @method_configurations.setter
    def method_configurations(
        self,
        value: typing.Optional[CrowMethodConfigurations],
    ) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sharedDirectory")
    def shared_directory(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        ...

    @shared_directory.setter
    def shared_directory(self, value: typing.Optional[builtins.str]) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceDirectory")
    def source_directory(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        ...

    @source_directory.setter
    def source_directory(self, value: typing.Optional[builtins.str]) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tokenAuthorizerConfiguration")
    def token_authorizer_configuration(self) -> typing.Any:
        '''
        :stability: experimental
        '''
        ...

    @token_authorizer_configuration.setter
    def token_authorizer_configuration(self, value: typing.Any) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="useAuthorizerLambda")
    def use_authorizer_lambda(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        ...

    @use_authorizer_lambda.setter
    def use_authorizer_lambda(self, value: typing.Optional[builtins.bool]) -> None:
        ...


class _ICrowApiPropsProxy:
    '''
    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "crow-api.ICrowApiProps"

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="apiGatewayConfiguration")
    def api_gateway_configuration(self) -> typing.Any:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Any, jsii.get(self, "apiGatewayConfiguration"))

    @api_gateway_configuration.setter
    def api_gateway_configuration(self, value: typing.Any) -> None:
        jsii.set(self, "apiGatewayConfiguration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authorizerDirectory")
    def authorizer_directory(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authorizerDirectory"))

    @authorizer_directory.setter
    def authorizer_directory(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "authorizerDirectory", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authorizerLambdaConfiguration")
    def authorizer_lambda_configuration(self) -> typing.Any:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Any, jsii.get(self, "authorizerLambdaConfiguration"))

    @authorizer_lambda_configuration.setter
    def authorizer_lambda_configuration(self, value: typing.Any) -> None:
        jsii.set(self, "authorizerLambdaConfiguration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="createApiKey")
    def create_api_key(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "createApiKey"))

    @create_api_key.setter
    def create_api_key(self, value: typing.Optional[builtins.bool]) -> None:
        jsii.set(self, "createApiKey", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="lambdaConfigurations")
    def lambda_configurations(self) -> typing.Optional[CrowLambdaConfigurations]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[CrowLambdaConfigurations], jsii.get(self, "lambdaConfigurations"))

    @lambda_configurations.setter
    def lambda_configurations(
        self,
        value: typing.Optional[CrowLambdaConfigurations],
    ) -> None:
        jsii.set(self, "lambdaConfigurations", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="logRetention")
    def log_retention(self) -> typing.Optional[aws_cdk.aws_logs.RetentionDays]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[aws_cdk.aws_logs.RetentionDays], jsii.get(self, "logRetention"))

    @log_retention.setter
    def log_retention(
        self,
        value: typing.Optional[aws_cdk.aws_logs.RetentionDays],
    ) -> None:
        jsii.set(self, "logRetention", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="methodConfigurations")
    def method_configurations(self) -> typing.Optional[CrowMethodConfigurations]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[CrowMethodConfigurations], jsii.get(self, "methodConfigurations"))

    @method_configurations.setter
    def method_configurations(
        self,
        value: typing.Optional[CrowMethodConfigurations],
    ) -> None:
        jsii.set(self, "methodConfigurations", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sharedDirectory")
    def shared_directory(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sharedDirectory"))

    @shared_directory.setter
    def shared_directory(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "sharedDirectory", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceDirectory")
    def source_directory(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceDirectory"))

    @source_directory.setter
    def source_directory(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "sourceDirectory", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tokenAuthorizerConfiguration")
    def token_authorizer_configuration(self) -> typing.Any:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Any, jsii.get(self, "tokenAuthorizerConfiguration"))

    @token_authorizer_configuration.setter
    def token_authorizer_configuration(self, value: typing.Any) -> None:
        jsii.set(self, "tokenAuthorizerConfiguration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="useAuthorizerLambda")
    def use_authorizer_lambda(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "useAuthorizerLambda"))

    @use_authorizer_lambda.setter
    def use_authorizer_lambda(self, value: typing.Optional[builtins.bool]) -> None:
        jsii.set(self, "useAuthorizerLambda", value)

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ICrowApiProps).__jsii_proxy_class__ = lambda : _ICrowApiPropsProxy


@jsii.data_type(
    jsii_type="crow-api.LambdasByPath",
    jsii_struct_bases=[],
    name_mapping={},
)
class LambdasByPath:
    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LambdasByPath(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


__all__ = [
    "CrowApi",
    "CrowLambdaConfigurations",
    "CrowMethodConfiguration",
    "CrowMethodConfigurations",
    "ICrowApiProps",
    "LambdasByPath",
]

publication.publish()
