#!/usr/bin/python
"""
    Copyright (c) 2016-present,
    Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.
   
    This file is part of Catapult.
   
    Catapult is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
   
    Catapult is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU Lesser General Public License for more details.
   
    You should have received a copy of the GNU Lesser General Public License
    along with Catapult. If not, see <http://www.gnu.org/licenses/>.
"""

from __future__ import annotations
from .GeneratorUtils import GeneratorUtils
from .UnresolvedAddressDto import UnresolvedAddressDto
from .UnresolvedMosaicIdDto import UnresolvedMosaicIdDto


class MosaicAddressRestrictionTransactionBodyBuilder:
    """Binary layout for a mosaic address restriction transaction."""

    def __init__(self,  mosaicId: UnresolvedMosaicIdDto,  restrictionKey: int,  previousRestrictionValue: int,  newRestrictionValue: int,  targetAddress: UnresolvedAddressDto):
        # pylint: disable-msg=line-too-long
        """Constructor.
        Args:
            mosaicId: Identifier of the mosaic to which the restriction applies.
            restrictionKey: Restriction key.
            previousRestrictionValue: Previous restriction value.
            newRestrictionValue: New restriction value.
            targetAddress: Address being restricted.
        """
        self.mosaicId = mosaicId
        self.restrictionKey = restrictionKey
        self.previousRestrictionValue = previousRestrictionValue
        self.newRestrictionValue = newRestrictionValue
        self.targetAddress = targetAddress

    @classmethod
    def loadFromBinary(cls, payload: bytes) -> MosaicAddressRestrictionTransactionBodyBuilder:
        """Creates an instance of MosaicAddressRestrictionTransactionBodyBuilder from binary payload.
        Args:
            payload: Byte payload to use to serialize the object.
        Returns:
            Instance of MosaicAddressRestrictionTransactionBodyBuilder.
        """
        bytes_ = bytes(payload)
        # gen: _load_from_binary_custom
        mosaicId = UnresolvedMosaicIdDto.loadFromBinary(bytes_)
        bytes_ = bytes_[mosaicId.getSize():]
        # gen: _load_from_binary_simple
        restrictionKey = GeneratorUtils.bufferToUint(GeneratorUtils.getBytes(bytes_, 8))
        bytes_ = bytes_[8:]
        # gen: _load_from_binary_simple
        previousRestrictionValue = GeneratorUtils.bufferToUint(GeneratorUtils.getBytes(bytes_, 8))
        bytes_ = bytes_[8:]
        # gen: _load_from_binary_simple
        newRestrictionValue = GeneratorUtils.bufferToUint(GeneratorUtils.getBytes(bytes_, 8))
        bytes_ = bytes_[8:]
        # gen: _load_from_binary_custom
        targetAddress = UnresolvedAddressDto.loadFromBinary(bytes_)
        bytes_ = bytes_[targetAddress.getSize():]
        return MosaicAddressRestrictionTransactionBodyBuilder(mosaicId, restrictionKey, previousRestrictionValue, newRestrictionValue, targetAddress)

    def getMosaicId(self) -> UnresolvedMosaicIdDto:
        """Gets identifier of the mosaic to which the restriction applies.
        Returns:
            Identifier of the mosaic to which the restriction applies.
        """
        return self.mosaicId

    def getRestrictionKey(self) -> int:
        """Gets restriction key.
        Returns:
            Restriction key.
        """
        return self.restrictionKey

    def getPreviousRestrictionValue(self) -> int:
        """Gets previous restriction value.
        Returns:
            Previous restriction value.
        """
        return self.previousRestrictionValue

    def getNewRestrictionValue(self) -> int:
        """Gets new restriction value.
        Returns:
            New restriction value.
        """
        return self.newRestrictionValue

    def getTargetAddress(self) -> UnresolvedAddressDto:
        """Gets address being restricted.
        Returns:
            Address being restricted.
        """
        return self.targetAddress

    def getSize(self) -> int:
        """Gets the size of the object.
        Returns:
            Size in bytes.
        """
        size = 0
        size += self.mosaicId.getSize()
        size += 8  # restrictionKey
        size += 8  # previousRestrictionValue
        size += 8  # newRestrictionValue
        size += self.targetAddress.getSize()
        return size

    def serialize(self) -> bytes:
        """Serializes an object to bytes.
        Returns:
            Serialized bytes.
        """
        bytes_ = bytes()
        mosaicIdBytes = self.mosaicId.serialize()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, mosaicIdBytes)
        restrictionKeyBytes = GeneratorUtils.uintToBuffer(self.getRestrictionKey(), 8)
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, restrictionKeyBytes)
        previousRestrictionValueBytes = GeneratorUtils.uintToBuffer(self.getPreviousRestrictionValue(), 8)
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, previousRestrictionValueBytes)
        newRestrictionValueBytes = GeneratorUtils.uintToBuffer(self.getNewRestrictionValue(), 8)
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, newRestrictionValueBytes)
        targetAddressBytes = self.targetAddress.serialize()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, targetAddressBytes)
        return bytes_

