"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscaling = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const cdk = require("aws-cdk-lib");
const asg = require("aws-cdk-lib/aws-autoscaling");
const aws_autoscaling_hooktargets_1 = require("aws-cdk-lib/aws-autoscaling-hooktargets");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const assets = require("aws-cdk-lib/aws-s3-assets");
const sns = require("aws-cdk-lib/aws-sns");
const subscriptions = require("aws-cdk-lib/aws-sns-subscriptions");
const cr = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
/**
 * GitlabRunnerAutoscaling Construct for create Autoscaling Gitlab Runner.
 */
class GitlabRunnerAutoscaling extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f;
        super(scope, id);
        const defaultProps = {
            instanceType: 't3.micro',
            tags: ['gitlab', 'awscdk', 'runner'],
            gitlabUrl: 'https://gitlab.com/',
            gitlabRunnerImage: 'public.ecr.aws/gitlab/gitlab-runner:alpine',
            alarms: [
                {
                    AlarmName: 'GitlabRunnerDiskUsage',
                    MetricName: 'disk_used_percent',
                },
            ],
        };
        const runnerProps = { ...defaultProps, ...props };
        const asset = new assets.Asset(this, 'GitlabRunnerUserDataAsset', {
            path: path.join(__dirname, '../assets/userdata/amazon-cloudwatch-agent.json'),
        });
        const userData = ec2.UserData.forLinux();
        userData.addS3DownloadCommand({
            bucket: asset.bucket,
            bucketKey: asset.s3ObjectKey,
            localFile: '/opt/aws/amazon-cloudwatch-agent/etc/amazon-cloudwatch-agent.json',
        });
        userData.addCommands(...this.createUserData(runnerProps));
        this.instanceRole = (_b = runnerProps.instanceRole) !== null && _b !== void 0 ? _b : new iam.Role(this, 'GitlabRunnerInstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            description: 'For EC2 Instance (Gitlab Runner) Role',
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchAgentServerPolicy'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3ReadOnlyAccess'),
            ],
        });
        this.vpc = (_c = runnerProps.vpc) !== null && _c !== void 0 ? _c : new ec2.Vpc(this, 'VPC');
        this.securityGroup = new ec2.SecurityGroup(this, 'GitlabRunnerSecurityGroup', {
            vpc: this.vpc,
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.instanceRole.roleName],
        });
        const lt = new ec2.CfnLaunchTemplate(this, 'GitlabRunnerLaunchTemplate', {
            launchTemplateData: {
                imageId: ec2.MachineImage.latestAmazonLinux({
                    generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                }).getImage(this).imageId,
                instanceType: runnerProps.instanceType,
                instanceMarketOptions: {
                    marketType: runnerProps.spotInstance ? 'spot' : undefined,
                    spotOptions: runnerProps.spotInstance ? {
                        spotInstanceType: 'one-time',
                    } : undefined,
                },
                userData: cdk.Fn.base64(userData.render()),
                blockDeviceMappings: [
                    {
                        deviceName: '/dev/xvda',
                        ebs: {
                            volumeSize: (_d = runnerProps.ebsSize) !== null && _d !== void 0 ? _d : 60,
                        },
                    },
                ],
                iamInstanceProfile: {
                    arn: instanceProfile.attrArn,
                },
                securityGroupIds: this.securityGroup.connections.securityGroups.map((m) => m.securityGroupId),
            },
        });
        this.autoscalingGroup = new asg.AutoScalingGroup(this, 'GitlabRunnerAutoscalingGroup', {
            instanceType: new ec2.InstanceType(runnerProps.instanceType),
            autoScalingGroupName: `Gitlab Runners (${runnerProps.instanceType})`,
            vpc: this.vpc,
            vpcSubnets: runnerProps.vpcSubnet,
            machineImage: ec2.MachineImage.latestAmazonLinux({
                generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            minCapacity: runnerProps.minCapacity,
            maxCapacity: runnerProps.maxCapacity,
            desiredCapacity: runnerProps.desiredCapacity,
        });
        const cfnAsg = this.autoscalingGroup.node.tryFindChild('ASG');
        cfnAsg.addPropertyDeletionOverride('LaunchConfigurationName');
        cfnAsg.addPropertyOverride('LaunchTemplate', {
            LaunchTemplateId: lt.ref,
            Version: lt.attrLatestVersionNumber,
        });
        this.autoscalingGroup.node.tryRemoveChild('LaunchConfig');
        this.topicAlarm = new sns.Topic(this, 'GitlabRunnerAlarm');
        const alarms = JSON.stringify(runnerProps.alarms);
        // Put alarms at launch
        const registerFunction = new lambda.Function(this, 'GitlabRunnerRegisterFunction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.register',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            logRetention: logs.RetentionDays.ONE_DAY,
            environment: {
                ALARMS: alarms,
                SNS_TOPIC_ARN: this.topicAlarm.topicArn,
            },
        });
        (_e = registerFunction.role) === null || _e === void 0 ? void 0 : _e.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'cloudwatch:PutMetricAlarm',
            ],
        }));
        this.autoscalingGroup.addLifecycleHook('GitlabRunnerLifeCycleHookLaunching', {
            lifecycleTransition: asg.LifecycleTransition.INSTANCE_LAUNCHING,
            notificationTarget: new aws_autoscaling_hooktargets_1.FunctionHook(registerFunction),
            defaultResult: asg.DefaultResult.CONTINUE,
            heartbeatTimeout: cdk.Duration.seconds(60),
        });
        // Add an alarm action to terminate invalid instances
        const alarmAction = new lambda.Function(this, 'GitlabRunnerAlarmAction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.on_alarm',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        (_f = alarmAction.role) === null || _f === void 0 ? void 0 : _f.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'autoscaling:SetInstanceHealth',
            ],
        }));
        const alarmSubscription = new subscriptions.LambdaSubscription(alarmAction);
        this.topicAlarm.addSubscription(alarmSubscription);
        // Unregister gitlab runners and remove alarms on instance termination or CFn stack deletion
        const unregisterRole = new iam.Role(this, 'GitlabRunnerUnregisterRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            description: 'For Gitlab Runner Unregistering Function Role',
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        unregisterRole.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'ssm:SendCommand',
                'autoscaling:DescribeAutoScalingGroups',
                'cloudwatch:DeleteAlarms',
            ],
        }));
        const unregisterFunction = new lambda.Function(this, 'GitlabRunnerUnregisterFunction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.unregister',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            role: unregisterRole,
            logRetention: logs.RetentionDays.ONE_DAY,
            environment: {
                ALARMS: alarms,
            },
        });
        this.autoscalingGroup.addLifecycleHook('GitlabRunnerLifeCycleHookTerminating', {
            lifecycleTransition: asg.LifecycleTransition.INSTANCE_TERMINATING,
            notificationTarget: new aws_autoscaling_hooktargets_1.FunctionHook(unregisterFunction),
            defaultResult: asg.DefaultResult.CONTINUE,
            heartbeatTimeout: cdk.Duration.seconds(60),
        });
        const unregisterCustomResource = new lambda.Function(this, 'GitlabRunnerUnregisterCustomResource', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.on_event',
            runtime: lambda.Runtime.PYTHON_3_8,
            role: unregisterRole,
            logRetention: logs.RetentionDays.ONE_DAY,
            environment: {
                ALARMS: alarms,
            },
        });
        const unregisterProvider = new cr.Provider(this, 'GitlabRunnerUnregisterProvider', {
            onEventHandler: unregisterCustomResource,
        });
        const customResource = new cdk.CustomResource(this, 'GitlabRunnerCustomResource', {
            serviceToken: unregisterProvider.serviceToken,
            properties: {
                AutoScalingGroupNames: [this.autoscalingGroup.autoScalingGroupName],
            },
        });
        customResource.node.addDependency(unregisterProvider);
        new cdk.CfnOutput(this, 'GitlabRunnerAutoScalingGroupArn', {
            value: this.autoscalingGroup.autoScalingGroupArn,
        });
    }
    dockerVolumesList(dockerVolume) {
        let tempString = '--docker-volumes "/var/run/docker.sock:/var/run/docker.sock"';
        if (dockerVolume) {
            let tempList = [];
            dockerVolume.forEach(e => {
                tempList.push(`"${e.hostPath}:${e.containerPath}"`);
            });
            tempList.forEach(e => {
                tempString = `${tempString} --docker-volumes ${e}`;
            });
        }
        return tempString;
    }
    /**
     * @param props
     * @returns Array.
     */
    createUserData(props) {
        var _b;
        return [
            'yum update -y',
            'sleep 15 && amazon-linux-extras install docker && yum install -y amazon-cloudwatch-agent && systemctl start docker && usermod -aG docker ec2-user && chmod 777 /var/run/docker.sock',
            'systemctl restart docker && systemctl enable docker && systemctl start amazon-cloudwatch-agent && systemctl enable amazon-cloudwatch-agent',
            `docker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock \
      --name gitlab-runner-register ${props.gitlabRunnerImage} register --non-interactive --url ${props.gitlabUrl} --registration-token ${props.gitlabToken} \
      --docker-pull-policy if-not-present ${this.dockerVolumesList(props === null || props === void 0 ? void 0 : props.dockerVolumes)} \
      --executor docker --docker-image "alpine:latest" --description "A Runner on EC2 Instance (${props.instanceType})" \
      --tag-list "${(_b = props.tags) === null || _b === void 0 ? void 0 : _b.join(',')}" --docker-privileged`,
            `sleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner ${props.gitlabRunnerImage}`,
        ];
    }
}
exports.GitlabRunnerAutoscaling = GitlabRunnerAutoscaling;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscaling[_a] = { fqn: "cdk-gitlab-runner.GitlabRunnerAutoscaling", version: "2.0.71" };
//# sourceMappingURL=data:application/json;base64,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