"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const log_retention_1 = require("../lib/log-retention");
module.exports = {
    'log retention construct'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new log_retention_1.LogRetention(stack, 'MyLambda', {
            logGroupName: 'group',
            retention: logs.RetentionDays.ONE_MONTH
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "logs:PutRetentionPolicy",
                            "logs:DeleteRetentionPolicy"
                        ],
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aServiceRoleDefaultPolicyADDA7DEB",
            "Roles": [
                {
                    "Ref": "LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aServiceRole9741ECFB"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            "ServiceToken": {
                "Fn::GetAtt": [
                    "LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aFD4BFC8A",
                    "Arn"
                ]
            },
            "LogGroupName": "group",
            "RetentionInDays": 30
        }));
        test.done();
    },
    'with imported role'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const role = iam.Role.fromRoleArn(stack, 'Role', 'arn:aws:iam::123456789012:role/CoolRole');
        // WHEN
        new log_retention_1.LogRetention(stack, 'MyLambda', {
            logGroupName: 'group',
            retention: logs.RetentionDays.ONE_MONTH,
            role
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "logs:PutRetentionPolicy",
                            "logs:DeleteRetentionPolicy"
                        ],
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "RolePolicy72E7D967",
            "Roles": [
                'CoolRole'
            ]
        }));
        assert_1.expect(stack).to(assert_1.countResources('AWS::IAM::Role', 0));
        test.done();
    },
    'with RetentionPeriod set to Infinity'(test) {
        const stack = new cdk.Stack();
        new log_retention_1.LogRetention(stack, 'MyLambda', {
            logGroupName: 'group',
            retention: logs.RetentionDays.INFINITE,
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            RetentionInDays: assert_1.ABSENT
        }));
        test.done();
    },
    'log group ARN is well formed and conforms'(test) {
        const stack = new cdk.Stack();
        const group = new log_retention_1.LogRetention(stack, 'MyLambda', {
            logGroupName: 'group',
            retention: logs.RetentionDays.ONE_MONTH,
        });
        const logGroupArn = group.logGroupArn;
        test.ok(logGroupArn.indexOf('logs') > -1, 'log group ARN is not as expected');
        test.ok(logGroupArn.indexOf('log-group') > -1, 'log group ARN is not as expected');
        test.ok(logGroupArn.endsWith(':*'), 'log group ARN is not as expected');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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