"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const lambda = require("../lib");
module.exports = {
    'can import a Lambda version by ARN'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const version = lambda.Version.fromVersionArn(stack, 'Version', 'arn:aws:lambda:region:account-id:function:function-name:version');
        new cdk.CfnOutput(stack, 'ARN', { value: version.functionArn });
        new cdk.CfnOutput(stack, 'Name', { value: version.functionName });
        // THEN
        assert_1.expect(stack).toMatch({
            Outputs: {
                ARN: {
                    Value: "arn:aws:lambda:region:account-id:function:function-name:version"
                },
                Name: {
                    Value: "function-name:version"
                }
            }
        });
        test.done();
    },
    'create a version with event invoke config'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Fn', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline('foo'),
        });
        // WHEN
        new lambda.Version(stack, 'Version', {
            lambda: fn,
            maxEventAge: cdk.Duration.hours(1),
            retryAttempts: 0
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventInvokeConfig', {
            FunctionName: {
                Ref: 'Fn9270CBC0'
            },
            Qualifier: {
                'Fn::GetAtt': [
                    'Version6A868472',
                    'Version'
                ]
            },
            MaximumEventAgeInSeconds: 3600,
            MaximumRetryAttempts: 0
        }));
        test.done();
    },
    'throws when calling configureAsyncInvoke on already configured version'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Fn', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline('foo'),
        });
        const version = new lambda.Version(stack, 'Version', {
            lambda: fn,
            maxEventAge: cdk.Duration.hours(1),
            retryAttempts: 0
        });
        // THEN
        test.throws(() => version.configureAsyncInvoke({ retryAttempts: 1 }), /An EventInvokeConfig has already been configured/);
        test.done();
    },
    'event invoke config on imported versions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const version1 = lambda.Version.fromVersionArn(stack, 'Version1', 'arn:aws:lambda:region:account-id:function:function-name:version1');
        const version2 = lambda.Version.fromVersionArn(stack, 'Version2', 'arn:aws:lambda:region:account-id:function:function-name:version2');
        // WHEN
        version1.configureAsyncInvoke({
            retryAttempts: 1
        });
        version2.configureAsyncInvoke({
            retryAttempts: 0
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventInvokeConfig', {
            FunctionName: 'function-name',
            Qualifier: 'version1',
            MaximumRetryAttempts: 1
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventInvokeConfig', {
            FunctionName: 'function-name',
            Qualifier: 'version2',
            MaximumRetryAttempts: 0
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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