# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs-dev/1_02_ml.ipynb.

# %% ../nbs-dev/1_02_ml.ipynb 4
from __future__ import annotations
from IPython.display import Image, display
from pathlib import Path
from pprint import pprint
from scipy import stats
from scipy.cluster import hierarchy as hc
from typing import Any
from zipfile import ZipFile

import configparser
import datetime as dt
import json
import matplotlib.pyplot as plt
import numpy as np
import os
import pandas as pd
import shutil
import subprocess

# %% auto 0
__all__ = ['are_features_consistent', 'cluster_columns', 'run_cli', 'get_config_value', 'fastbook_on_colab', 'kaggle_setup_colab',
           'kaggle_list_files', 'kaggle_download_competition_files']

# %% ../nbs-dev/1_02_ml.ipynb 5
def are_features_consistent(
    train_df:pd.DataFrame,               # Training dataset DataFrame 
    test_df:pd.DataFrame,                # Testing dataset DataFrame 
    dependent_variables:list(str) = None # List of column name(s) for dependent variables 
)-> bool :                               # True if features in train and test datasets are consistent, False otherwise
    """Verify that features/columns in training and test sets are consistent"""
    if dependent_variables is None:
        features_training_set = train_df.columns
    else:
        features_training_set = train_df.drop(dependent_variables, axis=1).columns
    features_test_set = test_df.columns
    features_diff = set(features_training_set).symmetric_difference(features_test_set)
    if not features_diff == set():
        raise ValueError(f"Discrepancy between training and test feature set: {features_diff}")

    return True

# %% ../nbs-dev/1_02_ml.ipynb 14
def cluster_columns(df:pd.DataFrame,  # Multi-feature dataset with column names
                    figsize:tuple(int, int) = (10,6), # Size of the figure
                    font_size:int = 12    # Font size for the chart
                   ):
    """Plot dendogram based on Dataframe's columns' spearman correlation coefficients"""
    corr = np.round(stats.spearmanr(df).correlation, 4)
    corr_condensed = hc.distance.squareform(1-corr)
    z = hc.linkage(corr_condensed, method='average')
    fig = plt.figure(figsize=figsize)
    hc.dendrogram(z, labels=df.columns, orientation='left', leaf_font_size=font_size)
    plt.show()

# %% ../nbs-dev/1_02_ml.ipynb 17
def run_cli(cmd:str = 'ls -l'   # command to execute in the cli
           ):
    """Wrapper to use subprocess.run with passed command, and print the shell messages"""
    p = subprocess.run(cmd, stdout=subprocess.PIPE, shell=True)
    print(str(p.stdout, 'utf-8'))

# %% ../nbs-dev/1_02_ml.ipynb 19
def get_config_value(section:str,                        # section in the configparser cfg file
                     key:str,                            # key in the selected section
                     path_to_config_file:Path|str=None   # path to the cfg file
                    )-> Any :                            # the value corresponding to section - key - value 
    """Returns the value corresponding to the key-value pair in the configuration file (configparser format)"""
    if path_to_config_file is None:
        path_to_config_file = Path(f"/content/gdrive/My Drive/config-api-keys.cfg")
    elif isinstance(path_to_config_file, str):
        path_to_config_file = Path(f"/content/gdrive/My Drive/{path_to_config_file}")

    msg = f"Cannot find file {path_to_config_file}. Please check the path or add the config file at that location"
    assert path_to_config_file.is_file(), msg

    configuration = configparser.ConfigParser()
    configuration.read(path_to_config_file)
    return configuration[section][key]

# %% ../nbs-dev/1_02_ml.ipynb 23
def fastbook_on_colab():
    """Set up environment to run fastbook notebooks for colab"""
    instructions = ['pip install -Uqq fastbook',
                    'wget -O utils.py https://raw.githubusercontent.com/vtecftwy/fastbook/walk-thru/utils.py',
                    'wget -O fastbook_utils.py https://raw.githubusercontent.com/vtecftwy/fastbook/walk-thru/fastbook_utils.py'
                    ]

# %% ../nbs-dev/1_02_ml.ipynb 25
def kaggle_setup_colab(path_to_config_file:Path|str = None      # path to the configuration file (e.g. config.cfg)
                      ):
    """Update kaggle API and create security key json file from config file on Google Drive"""
    # Create API security key file
    path_to_kaggle = Path('/root/.kaggle')
    os.makedirs(path_to_kaggle, exist_ok=True)

    username = get_config_value('kaggle', 'kaggle_username', path_to_config_file=path_to_config_file)
    key = get_config_value('kaggle', 'kaggle_key', path_to_config_file=path_to_config_file)

    api_token = {"username": username, "key": key}
    with open(path_to_kaggle / 'kaggle.json', 'w') as file:
        json.dump(api_token, file)
        os.fchmod(file.fileno(), 600)

    # Update kaggle API software
    run_cli('pip install -Uqq kaggle --upgrade')

# %% ../nbs-dev/1_02_ml.ipynb 27
def kaggle_list_files(code:str = None,          # code for the kaggle competition or dataset
                      mode:str ='competitions'  # mode: `competitions` or `datasets`
                     ):
    """List all files available in the competition or dataset for the passed code"""
    if code is None:
        print(f"code is None, please provide the code of the kaggle competition or dataset")
        return 'Failed'
    elif mode not in ['competitions', 'datasets']:
        print(f"mode must be either 'competitions' or 'datasets', not {mode}")
        return 'Failed'
    else:
        print(f"Listing the files available for {mode}: <{code}>")
        run_cli(f"kaggle {mode} files {code}")

    if mode == 'competitions':
        print(f"{'=' * 140}")
        print(f"Make sure to set the parameters for <{code}> in next cell:")
        print(f" - kaggle_project_folder_name: string with name of the project folder")
        print(f" - train_files: list of files to place into the <train> folder")
        print(f" - test_files: list of files to place into the <test> folder")
        print(f" - submit_files: list of files to place into the <submit> folder")
        print(f"{'=' * 140}")

# %% ../nbs-dev/1_02_ml.ipynb 28
def kaggle_download_competition_files(
    competition_code:str = None, 
    train_files:list() = [], 
    test_files:list = [], 
    submit_files:list = [], 
    project_folder:str = 'ds'
    ):
    """download all files for passed competition, unzip them if required, move them to train, test and submit folders

    competition_code: str       code of the kaggle competition
    train_files: list of str    names of files to be moved into train folder
    test_files: list of str     names of files to be moved into test folder
    submit_files: list of str   names of files to be moved into submit folder
    """
    if competition_code is None:
        print(f"competition_code is None, please provide the code of the kaggle competition")
        return 'Failed'
    else:
        list_of_datasets = {'train': train_files,
                            'test': test_files,
                            'submit': submit_files}

        # creating a project directory and set paths
        if not os.path.exists(project_folder):
            os.makedirs(project_folder)

        path2datasets = Path(f"/content/{project_folder}")
        path2datasets_str = str(path2datasets.absolute())

        # download all files from kaggle
        run_cli(f"kaggle competitions download -c {competition_code} -p {path2datasets}")

        print(f"{'=' * 140}")
        print('Downloaded files:')
        for f in [item for item in path2datasets.iterdir() if item.is_file()]:
            print(f" - {f}")
        print(f"{'=' * 140}")

        # Unzip all zipped files
        for f in path2datasets.glob('*.zip'):
            print(f"Unzipping {f.name}")
            zip_f = ZipFile(f)
            zip_f.extractall(path=path2datasets)
            os.remove(f)
        print(f"{'=' * 140}")

        # Move all data files to the correct data folder
        for dataset_folder, files in list_of_datasets.items():
            if not os.path.exists(f'{project_folder}/{dataset_folder}'):
                os.makedirs(f'{project_folder}/{dataset_folder}')

            for f in files:
                print(f"Moving {f} to {dataset_folder}")
                p2f = path2datasets / f
                if p2f.suffix == '.csv':
                    shutil.move(path2datasets / f, path2datasets / dataset_folder / f)
                else:
                    msg = f"Does not support {p2f.name}'s extension {p2f.suffix}"
                    raise RuntimeError(msg)

        print(f"{'=' * 140}")
        print('Done loading Kaggle files and moving them to corresponding folders')
