"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongControlPlaneEKS = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const defaultProps = {
    HelmOptions: {
        repository: 'https://charts.konghq.com',
        chart: 'kong',
        release: 'kong',
        timeout: aws_cdk_lib_1.Duration.minutes(15),
        wait: true,
        createNamespace: true,
    },
};
class KongControlPlaneEKS extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.options = { ...defaultProps, ...props };
        const rds_secrets = 'rds-secrets';
        const rds_secrets_provider_class = 'rds-secrets-provider-class';
        // const route_53_zone = aws_route53.HostedZone.fromLookup(this, 'MyZone', {
        //   domainName: props.endpoints.hostedZoneName,
        // });
        // const admin_cert = new aws_certificatemanager.Certificate(this, 'adminCert', {
        //   domainName: props.endpoints.adminDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        // const cluster_cert = new aws_certificatemanager.Certificate(this, 'clusterCert', {
        //   domainName: props.endpoints.clusterDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        // const telemetry_cert = new aws_certificatemanager.Certificate(this, 'telemetryCert', {
        //   domainName: props.endpoints.telemetryDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        // const manager_cert = new aws_certificatemanager.Certificate(this, 'managerCert', {
        //   domainName: props.endpoints.managerDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        const kong_namespace = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongNamespace', {
            cluster: this.options.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: this.options.namespace },
                },
            ],
        });
        const kong_cp_service_account = new aws_cdk_lib_1.aws_eks.ServiceAccount(this, 'KongCpSA', {
            cluster: this.options.cluster,
            name: 'kong-cp-service-account',
            namespace: this.options.namespace,
        });
        const aws_region = aws_cdk_lib_1.Stack.of(this).region;
        const aws_account = aws_cdk_lib_1.Stack.of(this).account;
        kong_cp_service_account.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: [
                `arn:aws:secretsmanager:${aws_region}:${aws_account}:secret:${this.options.license_secret_name}-??????`,
                `${this.options.rds.secret?.secretFullArn}`,
            ],
            actions: [
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
            ],
        }));
        kong_cp_service_account.node.addDependency(kong_namespace);
        const kong_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongCert', {
            cluster: this.options.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: this.options.cacertname,
                        namespace: this.options.namespace,
                    },
                    spec: {
                        commonName: this.options.endpoints.hostedZoneName,
                        dnsNames: [
                            this.options.endpoints.adminDns,
                            this.options.endpoints.clusterDns,
                            this.options.endpoints.managerDns,
                            this.options.endpoints.telemetryDns,
                        ],
                        duration: '2160h0m0s',
                        issuerRef: {
                            group: 'awspca.cert-manager.io',
                            kind: 'AWSPCAClusterIssuer',
                            name: this.options.clusterIssuerName,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${this.options.clusterIssuerName}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        kong_certificate.node.addDependency(kong_namespace);
        // const secretNameParts = Fn.split('-', props.rds.secret?.secretName!);
        // const secretNameWithoutSuffix = Fn.join('-', [Fn.select(0, secretNameParts), Fn.select(1, secretNameParts)]);
        const secrets_crd_mount = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'SecretsMount', {
            cluster: this.options.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: rds_secrets_provider_class,
                        namespace: this.options.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        parameters: {
                            objects: `- objectName: "${this.options.rds.secret?.secretFullArn}"\n  jmesPath:\n    - path: username\n      objectAlias: dbusername\n    - path: password\n      objectAlias: dbpassword\n    - path: dbname\n      objectAlias: dbname\n    - path: host\n      objectAlias: dbhost\n`,
                        },
                        secretObjects: [
                            {
                                secretName: rds_secrets,
                                type: 'Opaque',
                                data: [
                                    {
                                        objectName: 'dbusername',
                                        key: 'dbusername',
                                    },
                                    {
                                        objectName: 'dbpassword',
                                        key: 'dbpassword',
                                    },
                                    {
                                        objectName: 'dbname',
                                        key: 'dbname',
                                    },
                                    {
                                        objectName: 'dbhost',
                                        key: 'dbhost',
                                    },
                                ],
                            },
                        ],
                    },
                },
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: this.options.license_secret_name,
                        namespace: this.options.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        secretObjects: [
                            {
                                secretName: this.options.license_secret_name,
                                type: 'opaque',
                                data: [
                                    {
                                        objectName: this.options.license_secret_name,
                                        key: 'license',
                                    },
                                ],
                            },
                        ],
                        parameters: {
                            objects: `- objectName: "${this.options.license_secret_name}"\n  objectType: "secretsmanager"\n`,
                        },
                    },
                },
            ],
        });
        secrets_crd_mount.node.addDependency(kong_namespace);
        const deploy_kong_cp_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'KongCpHelm', {
            ...this.options.HelmOptions,
            cluster: this.options.cluster,
            namespace: this.options.namespace,
            values: {
                ingressController: {
                    enabled: false,
                },
                // image: {
                //   repository: 'kong/kong-gateway',
                //   tag: '2.5.0.0-alpine',
                // },
                env: {
                    cluster_data_plane_purge_delay: 86400,
                    database: 'postgres',
                    role: 'control_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: `/etc/secrets/${this.options.clusterIssuerName}-secret/tls.crt`,
                    cluster_cert_key: `/etc/secrets/${this.options.clusterIssuerName}-secret/tls.key`,
                    cluster_ca_cert: `/etc/secrets/${this.options.clusterIssuerName}-secret/ca.crt`,
                    admin_ssl_cert: `/etc/secrets/${this.options.clusterIssuerName}-secret/tls.crt`,
                    admin_ssl_cert_key: `/etc/secrets/${this.options.clusterIssuerName}-secret/tls.key`,
                    admin_gui_listen: '0.0.0.0:8002, 0.0.0.0:8445 ssl',
                    admin_gui_url: `https://${this.options.endpoints.managerDns}`,
                    admin_gui_ssl_cert: `/etc/secrets/${this.options.clusterIssuerName}-secret/tls.crt`,
                    admin_gui_ssl_cert_key: `/etc/secrets/${this.options.clusterIssuerName}-secret/tls.key`,
                    cluster_telemetry_listen: '0.0.0.0:8006',
                    cluster_telemetry_endpoint: `${this.options.endpoints.telemetryDns}:443`,
                    pg_user: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbusername',
                            },
                        },
                    },
                    pg_password: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbpassword',
                            },
                        },
                    },
                    pg_database: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbname',
                            },
                        },
                    },
                    pg_host: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbhost',
                            },
                        },
                    },
                },
                cluster: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: true,
                        servicePort: 443,
                    },
                    annotations: {
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': cluster_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internet-facing',
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${this.options.cluster.clusterName}-eks-cluster`,
                        'external-dns.alpha.kubernetes.io/hostname': this.options.endpoints.clusterDns,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                clustertelemetry: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: true,
                        servicePort: 443,
                    },
                    annotations: {
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': telemetry_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internet-facing',
                        'external-dns.alpha.kubernetes.io/hostname': this.options.endpoints.telemetryDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${this.options.cluster.clusterName}-eks-telemetry`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                proxy: {
                    enabled: false,
                },
                admin: {
                    enabled: true,
                    type: 'LoadBalancer',
                    labels: {
                        'enable-metrics': true,
                    },
                    tls: {
                        servicePort: 443,
                    },
                    annotations: {
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': admin_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internal',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=admin',
                        'external-dns.alpha.kubernetes.io/hostname': this.options.endpoints.adminDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${this.options.cluster.clusterName}-eks-admin`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                deployment: {
                    serviceAccount: {
                        create: false,
                        name: kong_cp_service_account.serviceAccountName,
                    },
                    userDefinedVolumes: [
                        {
                            name: this.options.license_secret_name,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: this.options.license_secret_name,
                                },
                            },
                        },
                        {
                            name: rds_secrets,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: rds_secrets_provider_class,
                                },
                            },
                        },
                    ],
                    userDefinedVolumeMounts: [
                        {
                            name: this.options.license_secret_name,
                            mountPath: '/mnt/license_secrets',
                            readOnly: true,
                        },
                        {
                            name: rds_secrets,
                            mountPath: '/mnt/rds_secrets',
                            readOnly: true,
                        },
                    ],
                },
                enterprise: {
                    enabled: true,
                    license_secret: this.options.license_secret_name,
                },
                serviceMonitor: {
                    enabled: true,
                },
                manager: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: true,
                        servicePort: 443,
                    },
                    // http: {
                    //   enabled: false,
                    // },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': manager_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internal',
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${this.options.cluster.clusterName}-eks-manager`,
                        'external-dns.alpha.kubernetes.io/hostname': this.options.endpoints.managerDns,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                postgresql: {
                    enabled: false,
                },
                secretVolumes: [
                    `${this.options.clusterIssuerName}-secret`,
                ],
            },
        });
        deploy_kong_cp_helm.node.addDependency(secrets_crd_mount, kong_namespace, kong_cp_service_account);
    }
}
exports.KongControlPlaneEKS = KongControlPlaneEKS;
//# sourceMappingURL=data:application/json;base64,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