"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DualAlbFargateService = exports.LoadBalancerAccessibility = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const route53 = require("@aws-cdk/aws-route53");
const targets = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
/**
 * The load balancer accessibility.
 *
 * @stability stable
 */
var LoadBalancerAccessibility;
(function (LoadBalancerAccessibility) {
    LoadBalancerAccessibility["EXTERNAL_ONLY"] = "EXTERNAL_ONLY";
    LoadBalancerAccessibility["INTERNAL_ONLY"] = "INTERNAL_ONLY";
})(LoadBalancerAccessibility = exports.LoadBalancerAccessibility || (exports.LoadBalancerAccessibility = {}));
/**
 * @stability stable
 */
class DualAlbFargateService extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(scope, id);
        this.hasExternalLoadBalancer = false;
        this.hasInternalLoadBalancer = false;
        this.vpcSubnets = { subnetType: ec2.SubnetType.PRIVATE };
        /**
         * determine if vpcSubnets are all public ones
         */
        this.isPublicSubnets = false;
        this.vpc = (_b = props.vpc) !== null && _b !== void 0 ? _b : getOrCreateVpc(this),
            this.service = [];
        if (props.vpcSubnets) {
            this.vpcSubnets = props.vpcSubnets;
            this.validateSubnets(this.vpc, this.vpcSubnets);
        }
        // determine whether we need the external LB
        props.tasks.forEach(t => {
            // determine the accessibility
            if (t.accessibility != LoadBalancerAccessibility.INTERNAL_ONLY) {
                this.hasExternalLoadBalancer = true;
            }
            if (t.accessibility != LoadBalancerAccessibility.EXTERNAL_ONLY) {
                this.hasInternalLoadBalancer = true;
            }
        });
        if (this.hasExternalLoadBalancer) {
            this.externalAlb = new elbv2.ApplicationLoadBalancer(this, 'ExternalAlb', {
                vpc: this.vpc,
                internetFacing: true,
            });
        }
        if (this.hasInternalLoadBalancer) {
            this.internalAlb = new elbv2.ApplicationLoadBalancer(this, 'InternalAlb', {
                vpc: this.vpc,
                internetFacing: false,
            });
        }
        const cluster = new ecs.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: true,
        });
        const spotOnlyStrategy = [
            {
                capacityProvider: 'FARGATE_SPOT',
                base: 0,
                weight: 1,
            },
            {
                capacityProvider: 'FARGATE',
                base: 0,
                weight: 0,
            },
        ];
        props.tasks.forEach(t => {
            var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
            const defaultContainerName = (_b = t.task.defaultContainer) === null || _b === void 0 ? void 0 : _b.containerName;
            const svc = new ecs.FargateService(this, `${defaultContainerName}Service`, {
                taskDefinition: t.task,
                cluster,
                capacityProviderStrategies: (_c = t.capacityProviderStrategy) !== null && _c !== void 0 ? _c : (props.spot ? spotOnlyStrategy : undefined),
                desiredCount: t.desiredCount,
                enableExecuteCommand: (_d = props.enableExecuteCommand) !== null && _d !== void 0 ? _d : false,
                vpcSubnets: this.vpcSubnets,
                assignPublicIp: this.isPublicSubnets,
            });
            this.service.push(svc);
            // default scaling policy
            const scaling = svc.autoScaleTaskCount({ maxCapacity: (_f = (_e = t.scalingPolicy) === null || _e === void 0 ? void 0 : _e.maxCapacity) !== null && _f !== void 0 ? _f : 10 });
            scaling.scaleOnCpuUtilization('CpuScaling', {
                targetUtilizationPercent: (_h = (_g = t.scalingPolicy) === null || _g === void 0 ? void 0 : _g.targetCpuUtilization) !== null && _h !== void 0 ? _h : 50,
            });
            if (t.accessibility != LoadBalancerAccessibility.INTERNAL_ONLY) {
                const exttg = new elbv2.ApplicationTargetGroup(this, `${defaultContainerName}ExtTG`, {
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    vpc: this.vpc,
                });
                // listener for the external ALB
                new elbv2.ApplicationListener(this, `ExtAlbListener${t.listenerPort}`, {
                    loadBalancer: this.externalAlb,
                    open: true,
                    port: t.listenerPort,
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    defaultTargetGroups: [exttg],
                });
                scaling.scaleOnRequestCount('RequestScaling', {
                    requestsPerTarget: (_k = (_j = t.scalingPolicy) === null || _j === void 0 ? void 0 : _j.requestPerTarget) !== null && _k !== void 0 ? _k : 1000,
                    targetGroup: exttg,
                });
                exttg.addTarget(svc);
            }
            if (t.accessibility != LoadBalancerAccessibility.EXTERNAL_ONLY) {
                const inttg = new elbv2.ApplicationTargetGroup(this, `${defaultContainerName}IntTG`, {
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    vpc: this.vpc,
                });
                // listener for the internal ALB
                new elbv2.ApplicationListener(this, `IntAlbListener${t.listenerPort}`, {
                    loadBalancer: this.internalAlb,
                    open: true,
                    port: t.listenerPort,
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    defaultTargetGroups: [inttg],
                });
                // extra scaling policy
                scaling.scaleOnRequestCount('RequestScaling2', {
                    requestsPerTarget: (_m = (_l = t.scalingPolicy) === null || _l === void 0 ? void 0 : _l.requestPerTarget) !== null && _m !== void 0 ? _m : 1000,
                    targetGroup: inttg,
                });
                inttg.addTarget(svc);
            }
        });
        // Route53
        const zoneName = (_d = (_c = props.route53Ops) === null || _c === void 0 ? void 0 : _c.zoneName) !== null && _d !== void 0 ? _d : 'svc.local';
        const externalAlbRecordName = (_f = (_e = props.route53Ops) === null || _e === void 0 ? void 0 : _e.externalAlbRecordName) !== null && _f !== void 0 ? _f : 'external';
        const internalAlbRecordName = (_h = (_g = props.route53Ops) === null || _g === void 0 ? void 0 : _g.internalAlbRecordName) !== null && _h !== void 0 ? _h : 'internal';
        const zone = new route53.PrivateHostedZone(this, 'HostedZone', {
            zoneName,
            vpc: this.vpc,
        });
        if (this.hasInternalLoadBalancer) {
            new route53.ARecord(this, 'InternalAlbAlias', {
                zone,
                recordName: internalAlbRecordName,
                target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(this.internalAlb)),
            });
        }
        if (this.hasExternalLoadBalancer) {
            new route53.ARecord(this, 'ExternalAlbAlias', {
                zone,
                recordName: externalAlbRecordName,
                target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(this.externalAlb)),
            });
        }
        if (this.hasExternalLoadBalancer) {
            new cdk.CfnOutput(this, 'ExternalEndpoint', { value: `http://${this.externalAlb.loadBalancerDnsName}` });
            new cdk.CfnOutput(this, 'ExternalEndpointPrivate', { value: `http://${externalAlbRecordName}.${zoneName}` });
        }
        if (this.hasInternalLoadBalancer) {
            new cdk.CfnOutput(this, 'InternalEndpoint', { value: `http://${this.internalAlb.loadBalancerDnsName}` });
            new cdk.CfnOutput(this, 'InternalEndpointPrivate', { value: `http://${internalAlbRecordName}.${zoneName}` });
        }
    }
    validateSubnets(vpc, vpcSubnets) {
        const subnets = vpc.selectSubnets(vpcSubnets);
        // get all subnets in the VPC
        const allsubnetIds = vpc.publicSubnets.concat(vpc.privateSubnets).concat(vpc.isolatedSubnets).map(x => x.subnetId);
        // validate the given subnets
        subnets.subnetIds.forEach(s => {
            if (!allsubnetIds.includes(s)) {
                throw new Error(`${s} does not exist in the VPC`);
            }
            if (vpc.isolatedSubnets.map(i => i.subnetId).includes(s)) {
                throw new Error(`Isolated subnet ${s} is not allowed`);
            }
        });
        const hasPublic = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
        const hasPrivate = subnets.subnetIds.some(s => new Set(vpc.privateSubnets.map(x => x.subnetId)).has(s));
        if (hasPublic && hasPrivate) {
            throw new Error('You should provide either all public or all private subnets, not both.');
        }
        this.isPublicSubnets = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
    }
}
exports.DualAlbFargateService = DualAlbFargateService;
_a = JSII_RTTI_SYMBOL_1;
DualAlbFargateService[_a] = { fqn: "cdk-fargate-patterns.DualAlbFargateService", version: "0.1.2" };
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1'
        || process.env.CDK_USE_DEFAULT_VPC === '1' ? ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWFpbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9tYWluLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4Qyw2REFBNkQ7QUFDN0QsZ0RBQWdEO0FBQ2hELHdEQUF3RDtBQUN4RCxxQ0FBcUM7Ozs7OztBQWdCckMsSUFBWSx5QkFLWDtBQUxELFdBQVkseUJBQXlCO0lBRW5DLDREQUErQixDQUFBO0lBRS9CLDREQUErQixDQUFBO0FBQ2pDLENBQUMsRUFMVyx5QkFBeUIsR0FBekIsaUNBQXlCLEtBQXpCLGlDQUF5QixRQUtwQzs7OztBQWlDRCxNQUFhLHFCQUFzQixTQUFRLEdBQUcsQ0FBQyxTQUFTOzs7O0lBYXRELFlBQVksS0FBb0IsRUFBRSxFQUFVLEVBQUUsS0FBaUM7O1FBQzdFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFSWCw0QkFBdUIsR0FBWSxLQUFLLENBQUM7UUFDekMsNEJBQXVCLEdBQVksS0FBSyxDQUFDO1FBQ3pDLGVBQVUsR0FBd0IsRUFBRSxVQUFVLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUNqRjs7V0FFRztRQUNLLG9CQUFlLEdBQVksS0FBSyxDQUFDO1FBSXZDLElBQUksQ0FBQyxHQUFHLFNBQUcsS0FBSyxDQUFDLEdBQUcsbUNBQUksY0FBYyxDQUFDLElBQUksQ0FBQztZQUM1QyxJQUFJLENBQUMsT0FBTyxHQUFHLEVBQUUsQ0FBQztRQUNsQixJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7WUFDcEIsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDO1lBQ25DLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDakQ7UUFHRCw0Q0FBNEM7UUFDNUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDdEIsOEJBQThCO1lBQzlCLElBQUksQ0FBQyxDQUFDLGFBQWEsSUFBSSx5QkFBeUIsQ0FBQyxhQUFhLEVBQUc7Z0JBQy9ELElBQUksQ0FBQyx1QkFBdUIsR0FBRyxJQUFJLENBQUM7YUFDckM7WUFDRCxJQUFJLENBQUMsQ0FBQyxhQUFhLElBQUkseUJBQXlCLENBQUMsYUFBYSxFQUFFO2dCQUM5RCxJQUFJLENBQUMsdUJBQXVCLEdBQUcsSUFBSSxDQUFDO2FBQ3JDO1FBQ0gsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLElBQUksQ0FBQyx1QkFBdUIsRUFBRTtZQUNoQyxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksS0FBSyxDQUFDLHVCQUF1QixDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7Z0JBQ3hFLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztnQkFDYixjQUFjLEVBQUUsSUFBSTthQUNyQixDQUFDLENBQUM7U0FDSjtRQUVELElBQUksSUFBSSxDQUFDLHVCQUF1QixFQUFFO1lBQ2hDLElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxLQUFLLENBQUMsdUJBQXVCLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtnQkFDeEUsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO2dCQUNiLGNBQWMsRUFBRSxLQUFLO2FBQ3RCLENBQUMsQ0FBQztTQUNKO1FBRUQsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDL0MsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO1lBQ2IsOEJBQThCLEVBQUUsSUFBSTtTQUNyQyxDQUFDLENBQUM7UUFFSCxNQUFNLGdCQUFnQixHQUFHO1lBQ3ZCO2dCQUNFLGdCQUFnQixFQUFFLGNBQWM7Z0JBQ2hDLElBQUksRUFBRSxDQUFDO2dCQUNQLE1BQU0sRUFBRSxDQUFDO2FBQ1Y7WUFDRDtnQkFDRSxnQkFBZ0IsRUFBRSxTQUFTO2dCQUMzQixJQUFJLEVBQUUsQ0FBQztnQkFDUCxNQUFNLEVBQUUsQ0FBQzthQUNWO1NBQ0YsQ0FBQztRQUVGLEtBQUssQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFOztZQUN0QixNQUFNLG9CQUFvQixTQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLDBDQUFFLGFBQWEsQ0FBQztZQUNwRSxNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLEdBQUcsb0JBQW9CLFNBQVMsRUFBRTtnQkFDekUsY0FBYyxFQUFFLENBQUMsQ0FBQyxJQUFJO2dCQUN0QixPQUFPO2dCQUNQLDBCQUEwQixRQUFFLENBQUMsQ0FBQyx3QkFBd0IsbUNBQUksQ0FBRSxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFFO2dCQUN2RyxZQUFZLEVBQUUsQ0FBQyxDQUFDLFlBQVk7Z0JBQzVCLG9CQUFvQixRQUFFLEtBQUssQ0FBQyxvQkFBb0IsbUNBQUksS0FBSztnQkFDekQsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVO2dCQUMzQixjQUFjLEVBQUUsSUFBSSxDQUFDLGVBQWU7YUFDckMsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7WUFFdkIseUJBQXlCO1lBQ3pCLE1BQU0sT0FBTyxHQUFHLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFLFdBQVcsY0FBRSxDQUFDLENBQUMsYUFBYSwwQ0FBRSxXQUFXLG1DQUFJLEVBQUUsRUFBRSxDQUFDLENBQUM7WUFDNUYsT0FBTyxDQUFDLHFCQUFxQixDQUFDLFlBQVksRUFBRTtnQkFDMUMsd0JBQXdCLGNBQUUsQ0FBQyxDQUFDLGFBQWEsMENBQUUsb0JBQW9CLG1DQUFJLEVBQUU7YUFDdEUsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLENBQUMsYUFBYSxJQUFJLHlCQUF5QixDQUFDLGFBQWEsRUFBRTtnQkFDOUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUMsSUFBSSxFQUFFLEdBQUcsb0JBQW9CLE9BQU8sRUFBRTtvQkFDbkYsUUFBUSxFQUFFLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJO29CQUN4QyxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7aUJBQ2QsQ0FBQyxDQUFDO2dCQUNILGdDQUFnQztnQkFDaEMsSUFBSSxLQUFLLENBQUMsbUJBQW1CLENBQUMsSUFBSSxFQUFFLGlCQUFpQixDQUFDLENBQUMsWUFBWSxFQUFFLEVBQUU7b0JBQ3JFLFlBQVksRUFBRSxJQUFJLENBQUMsV0FBWTtvQkFDL0IsSUFBSSxFQUFFLElBQUk7b0JBQ1YsSUFBSSxFQUFFLENBQUMsQ0FBQyxZQUFZO29CQUNwQixRQUFRLEVBQUUsS0FBSyxDQUFDLG1CQUFtQixDQUFDLElBQUk7b0JBQ3hDLG1CQUFtQixFQUFFLENBQUMsS0FBSyxDQUFDO2lCQUM3QixDQUFDLENBQUM7Z0JBQ0gsT0FBTyxDQUFDLG1CQUFtQixDQUFDLGdCQUFnQixFQUFFO29CQUM1QyxpQkFBaUIsY0FBRSxDQUFDLENBQUMsYUFBYSwwQ0FBRSxnQkFBZ0IsbUNBQUksSUFBSTtvQkFDNUQsV0FBVyxFQUFFLEtBQUs7aUJBQ25CLENBQUMsQ0FBQztnQkFDSCxLQUFLLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDO2FBQ3RCO1lBRUQsSUFBSSxDQUFDLENBQUMsYUFBYSxJQUFJLHlCQUF5QixDQUFDLGFBQWEsRUFBRTtnQkFDOUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUMsSUFBSSxFQUFFLEdBQUcsb0JBQW9CLE9BQU8sRUFBRTtvQkFDbkYsUUFBUSxFQUFFLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJO29CQUN4QyxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7aUJBQ2QsQ0FBQyxDQUFDO2dCQUVILGdDQUFnQztnQkFDaEMsSUFBSSxLQUFLLENBQUMsbUJBQW1CLENBQUMsSUFBSSxFQUFFLGlCQUFpQixDQUFDLENBQUMsWUFBWSxFQUFFLEVBQUU7b0JBQ3JFLFlBQVksRUFBRSxJQUFJLENBQUMsV0FBWTtvQkFDL0IsSUFBSSxFQUFFLElBQUk7b0JBQ1YsSUFBSSxFQUFFLENBQUMsQ0FBQyxZQUFZO29CQUNwQixRQUFRLEVBQUUsS0FBSyxDQUFDLG1CQUFtQixDQUFDLElBQUk7b0JBQ3hDLG1CQUFtQixFQUFFLENBQUMsS0FBSyxDQUFDO2lCQUM3QixDQUFDLENBQUM7Z0JBRUgsdUJBQXVCO2dCQUN2QixPQUFPLENBQUMsbUJBQW1CLENBQUMsaUJBQWlCLEVBQUU7b0JBQzdDLGlCQUFpQixjQUFFLENBQUMsQ0FBQyxhQUFhLDBDQUFFLGdCQUFnQixtQ0FBSSxJQUFJO29CQUM1RCxXQUFXLEVBQUUsS0FBSztpQkFDbkIsQ0FBQyxDQUFDO2dCQUNILEtBQUssQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUM7YUFDdEI7UUFFSCxDQUFDLENBQUMsQ0FBQztRQUVILFVBQVU7UUFDVixNQUFNLFFBQVEsZUFBRyxLQUFLLENBQUMsVUFBVSwwQ0FBRSxRQUFRLG1DQUFJLFdBQVcsQ0FBQztRQUMzRCxNQUFNLHFCQUFxQixlQUFHLEtBQUssQ0FBQyxVQUFVLDBDQUFFLHFCQUFxQixtQ0FBSSxVQUFVLENBQUM7UUFDcEYsTUFBTSxxQkFBcUIsZUFBRyxLQUFLLENBQUMsVUFBVSwwQ0FBRSxxQkFBcUIsbUNBQUksVUFBVSxDQUFDO1FBQ3BGLE1BQU0sSUFBSSxHQUFHLElBQUksT0FBTyxDQUFDLGlCQUFpQixDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7WUFDN0QsUUFBUTtZQUNSLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztTQUNkLENBQUMsQ0FBQztRQUVILElBQUksSUFBSSxDQUFDLHVCQUF1QixFQUFFO1lBQ2hDLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7Z0JBQzVDLElBQUk7Z0JBQ0osVUFBVSxFQUFFLHFCQUFxQjtnQkFDakMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLElBQUksT0FBTyxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxXQUFZLENBQUMsQ0FBQzthQUMxRixDQUFDLENBQUM7U0FDSjtRQUdELElBQUksSUFBSSxDQUFDLHVCQUF1QixFQUFFO1lBQ2hDLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7Z0JBQzVDLElBQUk7Z0JBQ0osVUFBVSxFQUFFLHFCQUFxQjtnQkFDakMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLElBQUksT0FBTyxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxXQUFZLENBQUMsQ0FBQzthQUMxRixDQUFDLENBQUM7U0FDSjtRQUVELElBQUksSUFBSSxDQUFDLHVCQUF1QixFQUFFO1lBQ2hDLElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUUsRUFBRSxLQUFLLEVBQUUsVUFBVSxJQUFJLENBQUMsV0FBWSxDQUFDLG1CQUFtQixFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQzFHLElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUseUJBQXlCLEVBQUUsRUFBRSxLQUFLLEVBQUUsVUFBVSxxQkFBcUIsSUFBSSxRQUFRLEVBQUUsRUFBRSxDQUFDLENBQUM7U0FDOUc7UUFDRCxJQUFJLElBQUksQ0FBQyx1QkFBdUIsRUFBRTtZQUNoQyxJQUFJLEdBQUcsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFLEVBQUUsS0FBSyxFQUFFLFVBQVUsSUFBSSxDQUFDLFdBQVksQ0FBQyxtQkFBbUIsRUFBRSxFQUFFLENBQUMsQ0FBQztZQUMxRyxJQUFJLEdBQUcsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLHlCQUF5QixFQUFFLEVBQUUsS0FBSyxFQUFFLFVBQVUscUJBQXFCLElBQUksUUFBUSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1NBQzlHO0lBRUgsQ0FBQztJQUVPLGVBQWUsQ0FBQyxHQUFhLEVBQUUsVUFBK0I7UUFDcEUsTUFBTSxPQUFPLEdBQUcsR0FBRyxDQUFDLGFBQWEsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUM5Qyw2QkFBNkI7UUFDN0IsTUFBTSxZQUFZLEdBQUcsR0FBRyxDQUFDLGFBQWEsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsZUFBZSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ25ILDZCQUE2QjtRQUM3QixPQUFPLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUM1QixJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDN0IsTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQzthQUNuRDtZQUNELElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxFQUFFO2dCQUN4RCxNQUFNLElBQUksS0FBSyxDQUFDLG1CQUFtQixDQUFDLGlCQUFpQixDQUFDLENBQUM7YUFDeEQ7UUFDSCxDQUFDLENBQUMsQ0FBQztRQUNILE1BQU0sU0FBUyxHQUFHLE9BQU8sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN0RyxNQUFNLFVBQVUsR0FBRyxPQUFPLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDeEcsSUFBSSxTQUFTLElBQUksVUFBVSxFQUFFO1lBQzNCLE1BQU0sSUFBSSxLQUFLLENBQUMsd0VBQXdFLENBQUMsQ0FBQztTQUMzRjtRQUNELElBQUksQ0FBQyxlQUFlLEdBQUcsT0FBTyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQzdHLENBQUM7O0FBM0xILHNEQTRMQzs7O0FBRUQsU0FBUyxjQUFjLENBQUMsS0FBb0I7SUFDMUMsMENBQTBDO0lBQzFDLE9BQU8sS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsaUJBQWlCLENBQUMsS0FBSyxHQUFHO1dBQ3JELE9BQU8sQ0FBQyxHQUFHLENBQUMsbUJBQW1CLEtBQUssR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQztRQUNwRyxLQUFLLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDO1lBQ3RDLEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUUsRUFBRSxLQUFLLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsWUFBWSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDckYsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUUsRUFBRSxNQUFNLEVBQUUsQ0FBQyxFQUFFLFdBQVcsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0FBQy9ELENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBlYzIgZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBlY3MgZnJvbSAnQGF3cy1jZGsvYXdzLWVjcyc7XG5pbXBvcnQgKiBhcyBlbGJ2MiBmcm9tICdAYXdzLWNkay9hd3MtZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mic7XG5pbXBvcnQgKiBhcyByb3V0ZTUzIGZyb20gJ0Bhd3MtY2RrL2F3cy1yb3V0ZTUzJztcbmltcG9ydCAqIGFzIHRhcmdldHMgZnJvbSAnQGF3cy1jZGsvYXdzLXJvdXRlNTMtdGFyZ2V0cyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cblxuZXhwb3J0IGludGVyZmFjZSBEdWFsQWxiRmFyZ2F0ZVNlcnZpY2VQcm9wcyB7XG4gIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuICByZWFkb25seSB0YXNrczogRmFyZ2F0ZVRhc2tQcm9wc1tdO1xuICByZWFkb25seSByb3V0ZTUzT3BzPzogUm91dGU1M09wdGlvbnM7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzcG90PzogYm9vbGVhbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVuYWJsZUV4ZWN1dGVDb21tYW5kPzogYm9vbGVhbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdnBjU3VibmV0cz86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIExvYWRCYWxhbmNlckFjY2Vzc2liaWxpdHkge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgRVhURVJOQUxfT05MWSA9ICdFWFRFUk5BTF9PTkxZJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIElOVEVSTkFMX09OTFkgPSAnSU5URVJOQUxfT05MWScsXG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgRmFyZ2F0ZVRhc2tQcm9wcyB7XG4gIHJlYWRvbmx5IHRhc2s6IGVjcy5GYXJnYXRlVGFza0RlZmluaXRpb247XG4gIHJlYWRvbmx5IGxpc3RlbmVyUG9ydDogbnVtYmVyO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRlc2lyZWRDb3VudD86IG51bWJlcjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2NhbGluZ1BvbGljeT86IFNlcnZpY2VTY2FsaW5nUG9saWN5O1xuICByZWFkb25seSBjYXBhY2l0eVByb3ZpZGVyU3RyYXRlZ3k/OiBlY3MuQ2FwYWNpdHlQcm92aWRlclN0cmF0ZWd5W107XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhY2Nlc3NpYmlsaXR5PzogTG9hZEJhbGFuY2VyQWNjZXNzaWJpbGl0eTtcbn1cblxuXG5leHBvcnQgaW50ZXJmYWNlIFNlcnZpY2VTY2FsaW5nUG9saWN5IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG1heENhcGFjaXR5PzogbnVtYmVyO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdGFyZ2V0Q3B1VXRpbGl6YXRpb24/OiBudW1iZXI7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVxdWVzdFBlclRhcmdldD86IG51bWJlcjtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBSb3V0ZTUzT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHpvbmVOYW1lPzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZXh0ZXJuYWxBbGJSZWNvcmROYW1lPzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW50ZXJuYWxBbGJSZWNvcmROYW1lPzogc3RyaW5nO1xufVxuXG5leHBvcnQgY2xhc3MgRHVhbEFsYkZhcmdhdGVTZXJ2aWNlIGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG4gIHJlYWRvbmx5IGV4dGVybmFsQWxiPzogZWxidjIuQXBwbGljYXRpb25Mb2FkQmFsYW5jZXJcbiAgcmVhZG9ubHkgaW50ZXJuYWxBbGI/OiBlbGJ2Mi5BcHBsaWNhdGlvbkxvYWRCYWxhbmNlclxuICByZWFkb25seSB2cGM6IGVjMi5JVnBjO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2VydmljZTogZWNzLkZhcmdhdGVTZXJ2aWNlW107XG4gIHByaXZhdGUgaGFzRXh0ZXJuYWxMb2FkQmFsYW5jZXI6IGJvb2xlYW4gPSBmYWxzZTtcbiAgcHJpdmF0ZSBoYXNJbnRlcm5hbExvYWRCYWxhbmNlcjogYm9vbGVhbiA9IGZhbHNlO1xuICBwcml2YXRlIHZwY1N1Ym5ldHM6IGVjMi5TdWJuZXRTZWxlY3Rpb24gPSB7IHN1Ym5ldFR5cGU6IGVjMi5TdWJuZXRUeXBlLlBSSVZBVEUgfTtcbiAgLyoqXG4gICAqIGRldGVybWluZSBpZiB2cGNTdWJuZXRzIGFyZSBhbGwgcHVibGljIG9uZXNcbiAgICovXG4gIHByaXZhdGUgaXNQdWJsaWNTdWJuZXRzOiBib29sZWFuID0gZmFsc2U7XG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRHVhbEFsYkZhcmdhdGVTZXJ2aWNlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy52cGMgPSBwcm9wcy52cGMgPz8gZ2V0T3JDcmVhdGVWcGModGhpcyksXG4gICAgdGhpcy5zZXJ2aWNlID0gW107XG4gICAgaWYgKHByb3BzLnZwY1N1Ym5ldHMpIHtcbiAgICAgIHRoaXMudnBjU3VibmV0cyA9IHByb3BzLnZwY1N1Ym5ldHM7XG4gICAgICB0aGlzLnZhbGlkYXRlU3VibmV0cyh0aGlzLnZwYywgdGhpcy52cGNTdWJuZXRzKTtcbiAgICB9XG5cblxuICAgIC8vIGRldGVybWluZSB3aGV0aGVyIHdlIG5lZWQgdGhlIGV4dGVybmFsIExCXG4gICAgcHJvcHMudGFza3MuZm9yRWFjaCh0ID0+IHtcbiAgICAgIC8vIGRldGVybWluZSB0aGUgYWNjZXNzaWJpbGl0eVxuICAgICAgaWYgKHQuYWNjZXNzaWJpbGl0eSAhPSBMb2FkQmFsYW5jZXJBY2Nlc3NpYmlsaXR5LklOVEVSTkFMX09OTFkgKSB7XG4gICAgICAgIHRoaXMuaGFzRXh0ZXJuYWxMb2FkQmFsYW5jZXIgPSB0cnVlO1xuICAgICAgfVxuICAgICAgaWYgKHQuYWNjZXNzaWJpbGl0eSAhPSBMb2FkQmFsYW5jZXJBY2Nlc3NpYmlsaXR5LkVYVEVSTkFMX09OTFkpIHtcbiAgICAgICAgdGhpcy5oYXNJbnRlcm5hbExvYWRCYWxhbmNlciA9IHRydWU7XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICBpZiAodGhpcy5oYXNFeHRlcm5hbExvYWRCYWxhbmNlcikge1xuICAgICAgdGhpcy5leHRlcm5hbEFsYiA9IG5ldyBlbGJ2Mi5BcHBsaWNhdGlvbkxvYWRCYWxhbmNlcih0aGlzLCAnRXh0ZXJuYWxBbGInLCB7XG4gICAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICAgIGludGVybmV0RmFjaW5nOiB0cnVlLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMuaGFzSW50ZXJuYWxMb2FkQmFsYW5jZXIpIHtcbiAgICAgIHRoaXMuaW50ZXJuYWxBbGIgPSBuZXcgZWxidjIuQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIodGhpcywgJ0ludGVybmFsQWxiJywge1xuICAgICAgICB2cGM6IHRoaXMudnBjLFxuICAgICAgICBpbnRlcm5ldEZhY2luZzogZmFsc2UsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBjb25zdCBjbHVzdGVyID0gbmV3IGVjcy5DbHVzdGVyKHRoaXMsICdDbHVzdGVyJywge1xuICAgICAgdnBjOiB0aGlzLnZwYyxcbiAgICAgIGVuYWJsZUZhcmdhdGVDYXBhY2l0eVByb3ZpZGVyczogdHJ1ZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHNwb3RPbmx5U3RyYXRlZ3kgPSBbXG4gICAgICB7XG4gICAgICAgIGNhcGFjaXR5UHJvdmlkZXI6ICdGQVJHQVRFX1NQT1QnLFxuICAgICAgICBiYXNlOiAwLFxuICAgICAgICB3ZWlnaHQ6IDEsXG4gICAgICB9LFxuICAgICAge1xuICAgICAgICBjYXBhY2l0eVByb3ZpZGVyOiAnRkFSR0FURScsXG4gICAgICAgIGJhc2U6IDAsXG4gICAgICAgIHdlaWdodDogMCxcbiAgICAgIH0sXG4gICAgXTtcblxuICAgIHByb3BzLnRhc2tzLmZvckVhY2godCA9PiB7XG4gICAgICBjb25zdCBkZWZhdWx0Q29udGFpbmVyTmFtZSA9IHQudGFzay5kZWZhdWx0Q29udGFpbmVyPy5jb250YWluZXJOYW1lO1xuICAgICAgY29uc3Qgc3ZjID0gbmV3IGVjcy5GYXJnYXRlU2VydmljZSh0aGlzLCBgJHtkZWZhdWx0Q29udGFpbmVyTmFtZX1TZXJ2aWNlYCwge1xuICAgICAgICB0YXNrRGVmaW5pdGlvbjogdC50YXNrLFxuICAgICAgICBjbHVzdGVyLFxuICAgICAgICBjYXBhY2l0eVByb3ZpZGVyU3RyYXRlZ2llczogdC5jYXBhY2l0eVByb3ZpZGVyU3RyYXRlZ3kgPz8gKCBwcm9wcy5zcG90ID8gc3BvdE9ubHlTdHJhdGVneSA6IHVuZGVmaW5lZCApLFxuICAgICAgICBkZXNpcmVkQ291bnQ6IHQuZGVzaXJlZENvdW50LFxuICAgICAgICBlbmFibGVFeGVjdXRlQ29tbWFuZDogcHJvcHMuZW5hYmxlRXhlY3V0ZUNvbW1hbmQgPz8gZmFsc2UsXG4gICAgICAgIHZwY1N1Ym5ldHM6IHRoaXMudnBjU3VibmV0cyxcbiAgICAgICAgYXNzaWduUHVibGljSXA6IHRoaXMuaXNQdWJsaWNTdWJuZXRzLFxuICAgICAgfSk7XG4gICAgICB0aGlzLnNlcnZpY2UucHVzaChzdmMpO1xuXG4gICAgICAvLyBkZWZhdWx0IHNjYWxpbmcgcG9saWN5XG4gICAgICBjb25zdCBzY2FsaW5nID0gc3ZjLmF1dG9TY2FsZVRhc2tDb3VudCh7IG1heENhcGFjaXR5OiB0LnNjYWxpbmdQb2xpY3k/Lm1heENhcGFjaXR5ID8/IDEwIH0pO1xuICAgICAgc2NhbGluZy5zY2FsZU9uQ3B1VXRpbGl6YXRpb24oJ0NwdVNjYWxpbmcnLCB7XG4gICAgICAgIHRhcmdldFV0aWxpemF0aW9uUGVyY2VudDogdC5zY2FsaW5nUG9saWN5Py50YXJnZXRDcHVVdGlsaXphdGlvbiA/PyA1MCxcbiAgICAgIH0pO1xuXG4gICAgICBpZiAodC5hY2Nlc3NpYmlsaXR5ICE9IExvYWRCYWxhbmNlckFjY2Vzc2liaWxpdHkuSU5URVJOQUxfT05MWSkge1xuICAgICAgICBjb25zdCBleHR0ZyA9IG5ldyBlbGJ2Mi5BcHBsaWNhdGlvblRhcmdldEdyb3VwKHRoaXMsIGAke2RlZmF1bHRDb250YWluZXJOYW1lfUV4dFRHYCwge1xuICAgICAgICAgIHByb3RvY29sOiBlbGJ2Mi5BcHBsaWNhdGlvblByb3RvY29sLkhUVFAsXG4gICAgICAgICAgdnBjOiB0aGlzLnZwYyxcbiAgICAgICAgfSk7XG4gICAgICAgIC8vIGxpc3RlbmVyIGZvciB0aGUgZXh0ZXJuYWwgQUxCXG4gICAgICAgIG5ldyBlbGJ2Mi5BcHBsaWNhdGlvbkxpc3RlbmVyKHRoaXMsIGBFeHRBbGJMaXN0ZW5lciR7dC5saXN0ZW5lclBvcnR9YCwge1xuICAgICAgICAgIGxvYWRCYWxhbmNlcjogdGhpcy5leHRlcm5hbEFsYiEsXG4gICAgICAgICAgb3BlbjogdHJ1ZSxcbiAgICAgICAgICBwb3J0OiB0Lmxpc3RlbmVyUG9ydCxcbiAgICAgICAgICBwcm90b2NvbDogZWxidjIuQXBwbGljYXRpb25Qcm90b2NvbC5IVFRQLFxuICAgICAgICAgIGRlZmF1bHRUYXJnZXRHcm91cHM6IFtleHR0Z10sXG4gICAgICAgIH0pO1xuICAgICAgICBzY2FsaW5nLnNjYWxlT25SZXF1ZXN0Q291bnQoJ1JlcXVlc3RTY2FsaW5nJywge1xuICAgICAgICAgIHJlcXVlc3RzUGVyVGFyZ2V0OiB0LnNjYWxpbmdQb2xpY3k/LnJlcXVlc3RQZXJUYXJnZXQgPz8gMTAwMCxcbiAgICAgICAgICB0YXJnZXRHcm91cDogZXh0dGcsXG4gICAgICAgIH0pO1xuICAgICAgICBleHR0Zy5hZGRUYXJnZXQoc3ZjKTtcbiAgICAgIH1cblxuICAgICAgaWYgKHQuYWNjZXNzaWJpbGl0eSAhPSBMb2FkQmFsYW5jZXJBY2Nlc3NpYmlsaXR5LkVYVEVSTkFMX09OTFkpIHtcbiAgICAgICAgY29uc3QgaW50dGcgPSBuZXcgZWxidjIuQXBwbGljYXRpb25UYXJnZXRHcm91cCh0aGlzLCBgJHtkZWZhdWx0Q29udGFpbmVyTmFtZX1JbnRUR2AsIHtcbiAgICAgICAgICBwcm90b2NvbDogZWxidjIuQXBwbGljYXRpb25Qcm90b2NvbC5IVFRQLFxuICAgICAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIGxpc3RlbmVyIGZvciB0aGUgaW50ZXJuYWwgQUxCXG4gICAgICAgIG5ldyBlbGJ2Mi5BcHBsaWNhdGlvbkxpc3RlbmVyKHRoaXMsIGBJbnRBbGJMaXN0ZW5lciR7dC5saXN0ZW5lclBvcnR9YCwge1xuICAgICAgICAgIGxvYWRCYWxhbmNlcjogdGhpcy5pbnRlcm5hbEFsYiEsXG4gICAgICAgICAgb3BlbjogdHJ1ZSxcbiAgICAgICAgICBwb3J0OiB0Lmxpc3RlbmVyUG9ydCxcbiAgICAgICAgICBwcm90b2NvbDogZWxidjIuQXBwbGljYXRpb25Qcm90b2NvbC5IVFRQLFxuICAgICAgICAgIGRlZmF1bHRUYXJnZXRHcm91cHM6IFtpbnR0Z10sXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIGV4dHJhIHNjYWxpbmcgcG9saWN5XG4gICAgICAgIHNjYWxpbmcuc2NhbGVPblJlcXVlc3RDb3VudCgnUmVxdWVzdFNjYWxpbmcyJywge1xuICAgICAgICAgIHJlcXVlc3RzUGVyVGFyZ2V0OiB0LnNjYWxpbmdQb2xpY3k/LnJlcXVlc3RQZXJUYXJnZXQgPz8gMTAwMCxcbiAgICAgICAgICB0YXJnZXRHcm91cDogaW50dGcsXG4gICAgICAgIH0pO1xuICAgICAgICBpbnR0Zy5hZGRUYXJnZXQoc3ZjKTtcbiAgICAgIH1cblxuICAgIH0pO1xuXG4gICAgLy8gUm91dGU1M1xuICAgIGNvbnN0IHpvbmVOYW1lID0gcHJvcHMucm91dGU1M09wcz8uem9uZU5hbWUgPz8gJ3N2Yy5sb2NhbCc7XG4gICAgY29uc3QgZXh0ZXJuYWxBbGJSZWNvcmROYW1lID0gcHJvcHMucm91dGU1M09wcz8uZXh0ZXJuYWxBbGJSZWNvcmROYW1lID8/ICdleHRlcm5hbCc7XG4gICAgY29uc3QgaW50ZXJuYWxBbGJSZWNvcmROYW1lID0gcHJvcHMucm91dGU1M09wcz8uaW50ZXJuYWxBbGJSZWNvcmROYW1lID8/ICdpbnRlcm5hbCc7XG4gICAgY29uc3Qgem9uZSA9IG5ldyByb3V0ZTUzLlByaXZhdGVIb3N0ZWRab25lKHRoaXMsICdIb3N0ZWRab25lJywge1xuICAgICAgem9uZU5hbWUsXG4gICAgICB2cGM6IHRoaXMudnBjLFxuICAgIH0pO1xuXG4gICAgaWYgKHRoaXMuaGFzSW50ZXJuYWxMb2FkQmFsYW5jZXIpIHtcbiAgICAgIG5ldyByb3V0ZTUzLkFSZWNvcmQodGhpcywgJ0ludGVybmFsQWxiQWxpYXMnLCB7XG4gICAgICAgIHpvbmUsXG4gICAgICAgIHJlY29yZE5hbWU6IGludGVybmFsQWxiUmVjb3JkTmFtZSxcbiAgICAgICAgdGFyZ2V0OiByb3V0ZTUzLlJlY29yZFRhcmdldC5mcm9tQWxpYXMobmV3IHRhcmdldHMuTG9hZEJhbGFuY2VyVGFyZ2V0KHRoaXMuaW50ZXJuYWxBbGIhKSksXG4gICAgICB9KTtcbiAgICB9XG5cblxuICAgIGlmICh0aGlzLmhhc0V4dGVybmFsTG9hZEJhbGFuY2VyKSB7XG4gICAgICBuZXcgcm91dGU1My5BUmVjb3JkKHRoaXMsICdFeHRlcm5hbEFsYkFsaWFzJywge1xuICAgICAgICB6b25lLFxuICAgICAgICByZWNvcmROYW1lOiBleHRlcm5hbEFsYlJlY29yZE5hbWUsXG4gICAgICAgIHRhcmdldDogcm91dGU1My5SZWNvcmRUYXJnZXQuZnJvbUFsaWFzKG5ldyB0YXJnZXRzLkxvYWRCYWxhbmNlclRhcmdldCh0aGlzLmV4dGVybmFsQWxiISkpLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMuaGFzRXh0ZXJuYWxMb2FkQmFsYW5jZXIpIHtcbiAgICAgIG5ldyBjZGsuQ2ZuT3V0cHV0KHRoaXMsICdFeHRlcm5hbEVuZHBvaW50JywgeyB2YWx1ZTogYGh0dHA6Ly8ke3RoaXMuZXh0ZXJuYWxBbGIhLmxvYWRCYWxhbmNlckRuc05hbWV9YCB9KTtcbiAgICAgIG5ldyBjZGsuQ2ZuT3V0cHV0KHRoaXMsICdFeHRlcm5hbEVuZHBvaW50UHJpdmF0ZScsIHsgdmFsdWU6IGBodHRwOi8vJHtleHRlcm5hbEFsYlJlY29yZE5hbWV9LiR7em9uZU5hbWV9YCB9KTtcbiAgICB9XG4gICAgaWYgKHRoaXMuaGFzSW50ZXJuYWxMb2FkQmFsYW5jZXIpIHtcbiAgICAgIG5ldyBjZGsuQ2ZuT3V0cHV0KHRoaXMsICdJbnRlcm5hbEVuZHBvaW50JywgeyB2YWx1ZTogYGh0dHA6Ly8ke3RoaXMuaW50ZXJuYWxBbGIhLmxvYWRCYWxhbmNlckRuc05hbWV9YCB9KTtcbiAgICAgIG5ldyBjZGsuQ2ZuT3V0cHV0KHRoaXMsICdJbnRlcm5hbEVuZHBvaW50UHJpdmF0ZScsIHsgdmFsdWU6IGBodHRwOi8vJHtpbnRlcm5hbEFsYlJlY29yZE5hbWV9LiR7em9uZU5hbWV9YCB9KTtcbiAgICB9XG5cbiAgfVxuXG4gIHByaXZhdGUgdmFsaWRhdGVTdWJuZXRzKHZwYzogZWMyLklWcGMsIHZwY1N1Ym5ldHM6IGVjMi5TdWJuZXRTZWxlY3Rpb24pIHtcbiAgICBjb25zdCBzdWJuZXRzID0gdnBjLnNlbGVjdFN1Ym5ldHModnBjU3VibmV0cyk7XG4gICAgLy8gZ2V0IGFsbCBzdWJuZXRzIGluIHRoZSBWUENcbiAgICBjb25zdCBhbGxzdWJuZXRJZHMgPSB2cGMucHVibGljU3VibmV0cy5jb25jYXQodnBjLnByaXZhdGVTdWJuZXRzKS5jb25jYXQodnBjLmlzb2xhdGVkU3VibmV0cykubWFwKHggPT4geC5zdWJuZXRJZCk7XG4gICAgLy8gdmFsaWRhdGUgdGhlIGdpdmVuIHN1Ym5ldHNcbiAgICBzdWJuZXRzLnN1Ym5ldElkcy5mb3JFYWNoKHMgPT4ge1xuICAgICAgaWYgKCFhbGxzdWJuZXRJZHMuaW5jbHVkZXMocykpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGAke3N9IGRvZXMgbm90IGV4aXN0IGluIHRoZSBWUENgKTtcbiAgICAgIH1cbiAgICAgIGlmICh2cGMuaXNvbGF0ZWRTdWJuZXRzLm1hcChpID0+IGkuc3VibmV0SWQpLmluY2x1ZGVzKHMpKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgSXNvbGF0ZWQgc3VibmV0ICR7c30gaXMgbm90IGFsbG93ZWRgKTtcbiAgICAgIH1cbiAgICB9KTtcbiAgICBjb25zdCBoYXNQdWJsaWMgPSBzdWJuZXRzLnN1Ym5ldElkcy5zb21lKHMgPT4gbmV3IFNldCh2cGMucHVibGljU3VibmV0cy5tYXAoeCA9PiB4LnN1Ym5ldElkKSkuaGFzKHMpKTtcbiAgICBjb25zdCBoYXNQcml2YXRlID0gc3VibmV0cy5zdWJuZXRJZHMuc29tZShzID0+IG5ldyBTZXQodnBjLnByaXZhdGVTdWJuZXRzLm1hcCh4ID0+IHguc3VibmV0SWQpKS5oYXMocykpO1xuICAgIGlmIChoYXNQdWJsaWMgJiYgaGFzUHJpdmF0ZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3Ugc2hvdWxkIHByb3ZpZGUgZWl0aGVyIGFsbCBwdWJsaWMgb3IgYWxsIHByaXZhdGUgc3VibmV0cywgbm90IGJvdGguJyk7XG4gICAgfVxuICAgIHRoaXMuaXNQdWJsaWNTdWJuZXRzID0gc3VibmV0cy5zdWJuZXRJZHMuc29tZShzID0+IG5ldyBTZXQodnBjLnB1YmxpY1N1Ym5ldHMubWFwKHggPT4geC5zdWJuZXRJZCkpLmhhcyhzKSk7XG4gIH1cbn1cblxuZnVuY3Rpb24gZ2V0T3JDcmVhdGVWcGMoc2NvcGU6IGNkay5Db25zdHJ1Y3QpOiBlYzIuSVZwYyB7XG4gIC8vIHVzZSBhbiBleGlzdGluZyB2cGMgb3IgY3JlYXRlIGEgbmV3IG9uZVxuICByZXR1cm4gc2NvcGUubm9kZS50cnlHZXRDb250ZXh0KCd1c2VfZGVmYXVsdF92cGMnKSA9PT0gJzEnXG4gICAgfHwgcHJvY2Vzcy5lbnYuQ0RLX1VTRV9ERUZBVUxUX1ZQQyA9PT0gJzEnID8gZWMyLlZwYy5mcm9tTG9va3VwKHNjb3BlLCAnVnBjJywgeyBpc0RlZmF1bHQ6IHRydWUgfSkgOlxuICAgIHNjb3BlLm5vZGUudHJ5R2V0Q29udGV4dCgndXNlX3ZwY19pZCcpID9cbiAgICAgIGVjMi5WcGMuZnJvbUxvb2t1cChzY29wZSwgJ1ZwYycsIHsgdnBjSWQ6IHNjb3BlLm5vZGUudHJ5R2V0Q29udGV4dCgndXNlX3ZwY19pZCcpIH0pIDpcbiAgICAgIG5ldyBlYzIuVnBjKHNjb3BlLCAnVnBjJywgeyBtYXhBenM6IDMsIG5hdEdhdGV3YXlzOiAxIH0pO1xufVxuIl19