"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmithyBuild = void 0;
const component_1 = require("../component");
const json_1 = require("../json");
/**
 * Smithy build configuration options
 */
class SmithyBuild extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        this.version = options.version ?? "1.0";
        this.outputDirectory = options.outputDirectory;
        this._imports = options.imports;
        this._projections = options.projections;
        this._plugins = options.plugins;
        this.ignoreMissingPlugins = options.ignoreMissingPlugins;
        this._maven = options.maven;
        this.manifest = {
            version: this.version,
            outputDirectory: this.outputDirectory,
            imports: () => this._imports,
            projections: () => this._projections,
            plugins: () => this._plugins,
            ignoreMissingPlugins: this.ignoreMissingPlugins,
            maven: () => this._maven,
        };
        new json_1.JsonFile(this.project, "smithy-build.json", {
            obj: this.manifest,
            readonly: true,
            newline: false,
        });
    }
    /**
     * Get configured projections
     */
    get projections() {
        return this._projections ? { ...this._projections } : undefined;
    }
    /**
     * Get configured imports
     */
    get imports() {
        return this._imports ? [...this._imports] : undefined;
    }
    /**
     * Get configured plugins
     */
    get plugins() {
        return this._plugins ? { ...this._plugins } : undefined;
    }
    /**
     * Add a smithy build import
     */
    addImport(imp) {
        this._imports ? this._imports.push(imp) : (this._imports = [`${imp}`]);
    }
    /**
     * Add smithy build projections
     */
    addProjections(projections) {
        for (const [k, v] of Object.entries(projections)) {
            this._projections
                ? (this._projections[k] = v)
                : (this._projections = { [k]: v });
        }
    }
    /**
     * Add smithy build plugins
     */
    addPlugins(plugins) {
        for (const [k, v] of Object.entries(plugins)) {
            this._plugins ? (this._plugins[k] = v) : (this._plugins = { [k]: v });
        }
    }
    /**
     * Add maven dependencies to the smithy build for the vs-code plugin
     */
    addMavenDependencies(...dependencies) {
        this._maven = {
            dependencies: [...(this._maven?.dependencies ?? []), ...dependencies],
            repositories: this._maven?.repositories ?? [],
        };
    }
    /**
     * Add maven repositories to the smithy build for the vs-code plugin
     */
    addMavenRepositories(...repositories) {
        this._maven = {
            dependencies: this._maven?.dependencies ?? [],
            repositories: [...(this._maven?.repositories ?? []), ...repositories],
        };
    }
}
exports.SmithyBuild = SmithyBuild;
//# sourceMappingURL=data:application/json;base64,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