"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsiiProject = exports.Stability = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const consts_1 = require("./consts");
const jsii_docgen_1 = require("./jsii-docgen");
const javascript_1 = require("../javascript");
const typescript_1 = require("../typescript");
const util_1 = require("../util");
const EMAIL_REGEX = /(?:[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*|"(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21\x23-\x5b\x5d-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])*")@(?:(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?|\[(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?|[a-z0-9-]*[a-z0-9]:(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21-\x5a\x53-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])+)\])/;
const URL_REGEX = /((([A-Za-z]{3,9}:(?:\/\/)?)(?:[\-;:&=\+\$,\w]+@)?[A-Za-z0-9\.\-]+|(?:www\.|[\-;:&=\+\$,\w]+@)[A-Za-z0-9\.\-]+)((?:\/[\+~%\/\.\w\-_]*)?\??(?:[\-\+=&;%@\.\w_]*)#?(?:[\.\!\/\\\w]*))?)/;
const REPO_TEMP_DIRECTORY = ".repo";
var Stability;
(function (Stability) {
    Stability["EXPERIMENTAL"] = "experimental";
    Stability["STABLE"] = "stable";
    Stability["DEPRECATED"] = "deprecated";
})(Stability = exports.Stability || (exports.Stability = {}));
/**
 * Multi-language jsii library project
 *
 * @pjid jsii
 */
class JsiiProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        const { authorEmail, authorUrl } = parseAuthorAddress(options);
        const defaultOptions = {
            repository: options.repositoryUrl,
            authorName: options.author,
            authorEmail,
            authorUrl,
            jestOptions: {
                jestVersion: "^27",
            },
        };
        const forcedOptions = {
            releaseToNpm: false,
            disableTsconfig: true,
            docgen: false,
        };
        const mergedOptions = util_1.deepMerge([
            defaultOptions,
            options,
            forcedOptions,
        ]);
        super(mergedOptions);
        const srcdir = this.srcdir;
        const libdir = this.libdir;
        this.addFields({ types: `${libdir}/index.d.ts` });
        const compressAssembly = options.compressAssembly ?? false;
        // this is an unhelpful warning
        const jsiiFlags = ["--silence-warnings=reserved-word"];
        if (compressAssembly) {
            jsiiFlags.push("--compress-assembly");
        }
        const compatIgnore = options.compatIgnore ?? ".compatignore";
        this.addFields({ stability: options.stability ?? Stability.STABLE });
        if (options.stability === Stability.DEPRECATED) {
            this.addFields({ deprecated: true });
        }
        const compatTask = this.addTask("compat", {
            description: "Perform API compatibility check against latest version",
            exec: `jsii-diff npm:$(node -p "require(\'./package.json\').name") -k --ignore-file ${compatIgnore} || (echo "\nUNEXPECTED BREAKING CHANGES: add keys such as \'removed:constructs.Node.of\' to ${compatIgnore} to skip.\n" && exit 1)`,
        });
        const compat = options.compat ?? false;
        if (compat) {
            this.compileTask.spawn(compatTask);
        }
        this.compileTask.reset(["jsii", ...jsiiFlags].join(" "));
        this.watchTask.reset(["jsii", "-w", ...jsiiFlags].join(" "));
        this.packageAllTask = this.addTask("package-all", {
            description: "Packages artifacts for all target languages",
        });
        // in jsii we consider the entire repo (post build) as the build artifact
        // which is then used to create the language bindings in separate jobs.
        const prepareRepoForCI = [
            `rsync -a . .repo --exclude .git --exclude node_modules`,
            `rm -rf ${this.artifactsDirectory}`,
            `mv .repo ${this.artifactsDirectory}`,
        ].join(" && ");
        // when running inside CI we just prepare the repo for packaging, which
        // takes place in separate tasks.
        // outside of CI (i.e locally) we simply package all targets.
        this.packageTask.reset(`if [ ! -z \${CI} ]; then ${prepareRepoForCI}; else ${this.runTaskCommand(this.packageAllTask)}; fi`);
        const targets = {};
        const jsii = {
            outdir: this.artifactsDirectory,
            targets,
            tsc: {
                outDir: libdir,
                rootDir: srcdir,
            },
        };
        if (options.excludeTypescript) {
            jsii.excludeTypescript = options.excludeTypescript;
        }
        this.addFields({ jsii });
        this.release?.publisher.addGitHubPrePublishingSteps({
            name: "Prepare Repository",
            run: `mv ${this.artifactsDirectory} ${REPO_TEMP_DIRECTORY}`,
        }, {
            name: "Collect GitHub Metadata",
            run: `mv ${REPO_TEMP_DIRECTORY}/${this.artifactsDirectory} ${this.artifactsDirectory}`,
        });
        const extraJobOptions = options.workflowRunsOn
            ? { runsOn: options.workflowRunsOn }
            : {};
        if (options.releaseToNpm != false) {
            const task = this.addPackagingTask("js");
            this.release?.publisher.publishToNpm({
                ...this.pacmakForLanguage("js", task),
                registry: this.package.npmRegistry,
                npmTokenSecret: this.package.npmTokenSecret,
                codeArtifactOptions: options.codeArtifactOptions,
            });
            this.addPackagingTarget("js", task, extraJobOptions);
        }
        // we cannot call an option `java` because the java code generated by jsii
        // does not compile due to a conflict between this option name and the `java`
        // package (e.g. when `java.util.Objects` is referenced).
        if ("java" in options) {
            throw new Error('the "java" option is now called "publishToMaven"');
        }
        if (options.publishToMaven) {
            targets.java = {
                package: options.publishToMaven.javaPackage,
                maven: {
                    groupId: options.publishToMaven.mavenGroupId,
                    artifactId: options.publishToMaven.mavenArtifactId,
                },
            };
            const task = this.addPackagingTask("java");
            this.release?.publisher.publishToMaven({
                ...this.pacmakForLanguage("java", task),
                ...options.publishToMaven,
            });
            this.addPackagingTarget("java", task, extraJobOptions);
        }
        const pypi = options.publishToPypi ?? options.python;
        if (pypi) {
            targets.python = {
                distName: pypi.distName,
                module: pypi.module,
            };
            const task = this.addPackagingTask("python");
            this.release?.publisher.publishToPyPi({
                ...this.pacmakForLanguage("python", task),
                ...pypi,
            });
            this.addPackagingTarget("python", task, extraJobOptions);
        }
        const nuget = options.publishToNuget ?? options.dotnet;
        if (nuget) {
            targets.dotnet = {
                namespace: nuget.dotNetNamespace,
                packageId: nuget.packageId,
                iconUrl: nuget.iconUrl,
            };
            const task = this.addPackagingTask("dotnet");
            this.release?.publisher.publishToNuget({
                ...this.pacmakForLanguage("dotnet", task),
                ...nuget,
            });
            this.addPackagingTarget("dotnet", task, extraJobOptions);
        }
        const golang = options.publishToGo;
        if (golang) {
            targets.go = {
                moduleName: golang.moduleName,
                packageName: golang.packageName,
            };
            const task = this.addPackagingTask("go");
            this.release?.publisher.publishToGo({
                ...this.pacmakForLanguage("go", task),
                ...golang,
            });
            this.addPackagingTarget("go", task, extraJobOptions);
        }
        this.addDevDeps("jsii", "jsii-diff", "jsii-pacmak");
        this.gitignore.exclude(".jsii", "tsconfig.json");
        this.npmignore?.include(".jsii");
        if (options.docgen ?? true) {
            new jsii_docgen_1.JsiiDocgen(this, { filePath: options.docgenFilePath });
        }
        // jsii updates .npmignore, so we make it writable
        if (this.npmignore) {
            this.npmignore.readonly = false;
        }
        // https://github.com/projen/projen/issues/2165
        this.package.addPackageResolutions("@types/prettier@2.6.0");
        // https://github.com/projen/projen/issues/2264
        this.package.addPackageResolutions("@types/babel__traverse@7.18.2");
    }
    /**
     * Adds a target language to the build workflow and creates a package task.
     * @param language
     * @returns
     */
    addPackagingTarget(language, packTask, extraJobOptions) {
        if (!this.buildWorkflow) {
            return;
        }
        const pacmak = this.pacmakForLanguage(language, packTask);
        this.buildWorkflow.addPostBuildJob(`package-${language}`, {
            runsOn: ["ubuntu-latest"],
            permissions: {},
            tools: {
                node: { version: this.nodeVersion ?? "14.x" },
                ...pacmak.publishTools,
            },
            steps: pacmak.prePublishSteps ?? [],
            ...extraJobOptions,
        });
    }
    addPackagingTask(language) {
        const packageTask = this.tasks.addTask(`package:${language}`, {
            description: `Create ${language} language bindings`,
        });
        packageTask.exec(`jsii-pacmak -v --target ${language}`);
        this.packageAllTask.spawn(packageTask);
        return packageTask;
    }
    pacmakForLanguage(target, packTask) {
        // at this stage, `artifactsDirectory` contains the prebuilt repository.
        // for the publishing to work seamlessely, that directory needs to contain the actual artifact.
        // so we move the repo, create the artifact, and put it in the expected place.
        const prePublishSteps = [];
        prePublishSteps.push(...this.workflowBootstrapSteps);
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            prePublishSteps.push({
                name: "Setup pnpm",
                uses: "pnpm/action-setup@v2.2.4",
                with: { version: this.package.pnpmVersion },
            });
        }
        prePublishSteps.push({
            name: "Prepare Repository",
            run: `mv ${this.artifactsDirectory} ${REPO_TEMP_DIRECTORY}`,
        }, {
            name: "Install Dependencies",
            run: `cd ${REPO_TEMP_DIRECTORY} && ${this.package.installCommand}`,
        }, {
            name: `Create ${target} artifact`,
            run: `cd ${REPO_TEMP_DIRECTORY} && npx projen ${packTask.name}`,
        }, {
            name: `Collect ${target} Artifact`,
            run: `mv ${REPO_TEMP_DIRECTORY}/${this.artifactsDirectory} ${this.artifactsDirectory}`,
        });
        return {
            publishTools: consts_1.JSII_TOOLCHAIN[target],
            prePublishSteps,
        };
    }
}
exports.JsiiProject = JsiiProject;
_a = JSII_RTTI_SYMBOL_1;
JsiiProject[_a] = { fqn: "projen.cdk.JsiiProject", version: "0.67.87" };
function parseAuthorAddress(options) {
    let authorEmail = options.authorEmail;
    let authorUrl = options.authorUrl;
    if (options.authorAddress) {
        if (options.authorEmail && options.authorEmail !== options.authorAddress) {
            throw new Error("authorEmail is deprecated and cannot be used in conjunction with authorAddress");
        }
        if (options.authorUrl && options.authorUrl !== options.authorAddress) {
            throw new Error("authorUrl is deprecated and cannot be used in conjunction with authorAddress.");
        }
        if (EMAIL_REGEX.test(options.authorAddress)) {
            authorEmail = options.authorAddress;
        }
        else if (URL_REGEX.test(options.authorAddress)) {
            authorUrl = options.authorAddress;
        }
        else {
            throw new Error(`authorAddress must be either an email address or a URL: ${options.authorAddress}`);
        }
    }
    return { authorEmail, authorUrl };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoianNpaS1wcm9qZWN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2Nkay9qc2lpLXByb2plY3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxxQ0FBNEQ7QUFDNUQsK0NBQTJDO0FBRzNDLDhDQUEyRDtBQVEzRCw4Q0FBNEU7QUFDNUUsa0NBQW9DO0FBRXBDLE1BQU0sV0FBVyxHQUNmLDRhQUE0YSxDQUFDO0FBQy9hLE1BQU0sU0FBUyxHQUNiLHNMQUFzTCxDQUFDO0FBQ3pMLE1BQU0sbUJBQW1CLEdBQUcsT0FBTyxDQUFDO0FBbUdwQyxJQUFZLFNBSVg7QUFKRCxXQUFZLFNBQVM7SUFDbkIsMENBQTZCLENBQUE7SUFDN0IsOEJBQWlCLENBQUE7SUFDakIsc0NBQXlCLENBQUE7QUFDM0IsQ0FBQyxFQUpXLFNBQVMsR0FBVCxpQkFBUyxLQUFULGlCQUFTLFFBSXBCO0FBdUNEOzs7O0dBSUc7QUFDSCxNQUFhLFdBQVksU0FBUSw4QkFBaUI7SUFLaEQsWUFBWSxPQUEyQjtRQUNyQyxNQUFNLEVBQUUsV0FBVyxFQUFFLFNBQVMsRUFBRSxHQUFHLGtCQUFrQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRS9ELE1BQU0sY0FBYyxHQUFHO1lBQ3JCLFVBQVUsRUFBRSxPQUFPLENBQUMsYUFBYTtZQUNqQyxVQUFVLEVBQUUsT0FBTyxDQUFDLE1BQU07WUFDMUIsV0FBVztZQUNYLFNBQVM7WUFDVCxXQUFXLEVBQUU7Z0JBQ1gsV0FBVyxFQUFFLEtBQUs7YUFDbkI7U0FDRixDQUFDO1FBRUYsTUFBTSxhQUFhLEdBQUc7WUFDcEIsWUFBWSxFQUFFLEtBQUs7WUFDbkIsZUFBZSxFQUFFLElBQUk7WUFDckIsTUFBTSxFQUFFLEtBQUs7U0FDZCxDQUFDO1FBRUYsTUFBTSxhQUFhLEdBQUcsZ0JBQVMsQ0FBQztZQUM5QixjQUFjO1lBQ2QsT0FBTztZQUNQLGFBQWE7U0FDZCxDQUE2QixDQUFDO1FBRS9CLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUVyQixNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDO1FBQzNCLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUM7UUFFM0IsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLEtBQUssRUFBRSxHQUFHLE1BQU0sYUFBYSxFQUFFLENBQUMsQ0FBQztRQUVsRCxNQUFNLGdCQUFnQixHQUFHLE9BQU8sQ0FBQyxnQkFBZ0IsSUFBSSxLQUFLLENBQUM7UUFFM0QsK0JBQStCO1FBQy9CLE1BQU0sU0FBUyxHQUFHLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUN2RCxJQUFJLGdCQUFnQixFQUFFO1lBQ3BCLFNBQVMsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsQ0FBQztTQUN2QztRQUVELE1BQU0sWUFBWSxHQUFHLE9BQU8sQ0FBQyxZQUFZLElBQUksZUFBZSxDQUFDO1FBRTdELElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxTQUFTLEVBQUUsT0FBTyxDQUFDLFNBQVMsSUFBSSxTQUFTLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQztRQUVyRSxJQUFJLE9BQU8sQ0FBQyxTQUFTLEtBQUssU0FBUyxDQUFDLFVBQVUsRUFBRTtZQUM5QyxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsVUFBVSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7U0FDdEM7UUFFRCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRTtZQUN4QyxXQUFXLEVBQUUsd0RBQXdEO1lBQ3JFLElBQUksRUFBRSxnRkFBZ0YsWUFBWSxnR0FBZ0csWUFBWSx5QkFBeUI7U0FDeE8sQ0FBQyxDQUFDO1FBRUgsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUM7UUFDdkMsSUFBSSxNQUFNLEVBQUU7WUFDVixJQUFJLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUNwQztRQUVELElBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLENBQUMsTUFBTSxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDekQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEdBQUcsU0FBUyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDN0QsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsRUFBRTtZQUNoRCxXQUFXLEVBQUUsNkNBQTZDO1NBQzNELENBQUMsQ0FBQztRQUVILHlFQUF5RTtRQUN6RSx1RUFBdUU7UUFDdkUsTUFBTSxnQkFBZ0IsR0FBRztZQUN2Qix3REFBd0Q7WUFDeEQsVUFBVSxJQUFJLENBQUMsa0JBQWtCLEVBQUU7WUFDbkMsWUFBWSxJQUFJLENBQUMsa0JBQWtCLEVBQUU7U0FDdEMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFZix1RUFBdUU7UUFDdkUsaUNBQWlDO1FBQ2pDLDZEQUE2RDtRQUM3RCxJQUFJLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FDcEIsNEJBQTRCLGdCQUFnQixVQUFVLElBQUksQ0FBQyxjQUFjLENBQ3ZFLElBQUksQ0FBQyxjQUFjLENBQ3BCLE1BQU0sQ0FDUixDQUFDO1FBRUYsTUFBTSxPQUFPLEdBQXdCLEVBQUUsQ0FBQztRQUV4QyxNQUFNLElBQUksR0FBUTtZQUNoQixNQUFNLEVBQUUsSUFBSSxDQUFDLGtCQUFrQjtZQUMvQixPQUFPO1lBQ1AsR0FBRyxFQUFFO2dCQUNILE1BQU0sRUFBRSxNQUFNO2dCQUNkLE9BQU8sRUFBRSxNQUFNO2FBQ2hCO1NBQ0YsQ0FBQztRQUVGLElBQUksT0FBTyxDQUFDLGlCQUFpQixFQUFFO1lBQzdCLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxPQUFPLENBQUMsaUJBQWlCLENBQUM7U0FDcEQ7UUFFRCxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUV6QixJQUFJLENBQUMsT0FBTyxFQUFFLFNBQVMsQ0FBQywyQkFBMkIsQ0FDakQ7WUFDRSxJQUFJLEVBQUUsb0JBQW9CO1lBQzFCLEdBQUcsRUFBRSxNQUFNLElBQUksQ0FBQyxrQkFBa0IsSUFBSSxtQkFBbUIsRUFBRTtTQUM1RCxFQUNEO1lBQ0UsSUFBSSxFQUFFLHlCQUF5QjtZQUMvQixHQUFHLEVBQUUsTUFBTSxtQkFBbUIsSUFBSSxJQUFJLENBQUMsa0JBQWtCLElBQUksSUFBSSxDQUFDLGtCQUFrQixFQUFFO1NBQ3ZGLENBQ0YsQ0FBQztRQUVGLE1BQU0sZUFBZSxHQUFpQixPQUFPLENBQUMsY0FBYztZQUMxRCxDQUFDLENBQUMsRUFBRSxNQUFNLEVBQUUsT0FBTyxDQUFDLGNBQWMsRUFBRTtZQUNwQyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBRVAsSUFBSSxPQUFPLENBQUMsWUFBWSxJQUFJLEtBQUssRUFBRTtZQUNqQyxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDekMsSUFBSSxDQUFDLE9BQU8sRUFBRSxTQUFTLENBQUMsWUFBWSxDQUFDO2dCQUNuQyxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDO2dCQUNyQyxRQUFRLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXO2dCQUNsQyxjQUFjLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjO2dCQUMzQyxtQkFBbUIsRUFBRSxPQUFPLENBQUMsbUJBQW1CO2FBQ2pELENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsSUFBSSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1NBQ3REO1FBRUQsMEVBQTBFO1FBQzFFLDZFQUE2RTtRQUM3RSx5REFBeUQ7UUFDekQsSUFBSSxNQUFNLElBQUksT0FBTyxFQUFFO1lBQ3JCLE1BQU0sSUFBSSxLQUFLLENBQUMsa0RBQWtELENBQUMsQ0FBQztTQUNyRTtRQUVELElBQUksT0FBTyxDQUFDLGNBQWMsRUFBRTtZQUMxQixPQUFPLENBQUMsSUFBSSxHQUFHO2dCQUNiLE9BQU8sRUFBRSxPQUFPLENBQUMsY0FBYyxDQUFDLFdBQVc7Z0JBQzNDLEtBQUssRUFBRTtvQkFDTCxPQUFPLEVBQUUsT0FBTyxDQUFDLGNBQWMsQ0FBQyxZQUFZO29CQUM1QyxVQUFVLEVBQUUsT0FBTyxDQUFDLGNBQWMsQ0FBQyxlQUFlO2lCQUNuRDthQUNGLENBQUM7WUFFRixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLENBQUM7WUFFM0MsSUFBSSxDQUFDLE9BQU8sRUFBRSxTQUFTLENBQUMsY0FBYyxDQUFDO2dCQUNyQyxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDO2dCQUN2QyxHQUFHLE9BQU8sQ0FBQyxjQUFjO2FBQzFCLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1NBQ3hEO1FBRUQsTUFBTSxJQUFJLEdBQUcsT0FBTyxDQUFDLGFBQWEsSUFBSSxPQUFPLENBQUMsTUFBTSxDQUFDO1FBQ3JELElBQUksSUFBSSxFQUFFO1lBQ1IsT0FBTyxDQUFDLE1BQU0sR0FBRztnQkFDZixRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7Z0JBQ3ZCLE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTTthQUNwQixDQUFDO1lBRUYsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQzdDLElBQUksQ0FBQyxPQUFPLEVBQUUsU0FBUyxDQUFDLGFBQWEsQ0FBQztnQkFDcEMsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQztnQkFDekMsR0FBRyxJQUFJO2FBQ1IsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLGtCQUFrQixDQUFDLFFBQVEsRUFBRSxJQUFJLEVBQUUsZUFBZSxDQUFDLENBQUM7U0FDMUQ7UUFFRCxNQUFNLEtBQUssR0FBRyxPQUFPLENBQUMsY0FBYyxJQUFJLE9BQU8sQ0FBQyxNQUFNLENBQUM7UUFDdkQsSUFBSSxLQUFLLEVBQUU7WUFDVCxPQUFPLENBQUMsTUFBTSxHQUFHO2dCQUNmLFNBQVMsRUFBRSxLQUFLLENBQUMsZUFBZTtnQkFDaEMsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO2dCQUMxQixPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87YUFDdkIsQ0FBQztZQUVGLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUM3QyxJQUFJLENBQUMsT0FBTyxFQUFFLFNBQVMsQ0FBQyxjQUFjLENBQUM7Z0JBQ3JDLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUM7Z0JBQ3pDLEdBQUcsS0FBSzthQUNULENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLEVBQUUsSUFBSSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1NBQzFEO1FBRUQsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQztRQUNuQyxJQUFJLE1BQU0sRUFBRTtZQUNWLE9BQU8sQ0FBQyxFQUFFLEdBQUc7Z0JBQ1gsVUFBVSxFQUFFLE1BQU0sQ0FBQyxVQUFVO2dCQUM3QixXQUFXLEVBQUUsTUFBTSxDQUFDLFdBQVc7YUFDaEMsQ0FBQztZQUVGLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUN6QyxJQUFJLENBQUMsT0FBTyxFQUFFLFNBQVMsQ0FBQyxXQUFXLENBQUM7Z0JBQ2xDLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksRUFBRSxJQUFJLENBQUM7Z0JBQ3JDLEdBQUcsTUFBTTthQUNWLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsSUFBSSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1NBQ3REO1FBRUQsSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLEVBQUUsV0FBVyxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBRXBELElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxlQUFlLENBQUMsQ0FBQztRQUNqRCxJQUFJLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUVqQyxJQUFJLE9BQU8sQ0FBQyxNQUFNLElBQUksSUFBSSxFQUFFO1lBQzFCLElBQUksd0JBQVUsQ0FBQyxJQUFJLEVBQUUsRUFBRSxRQUFRLEVBQUUsT0FBTyxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUM7U0FDNUQ7UUFFRCxrREFBa0Q7UUFDbEQsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFO1lBQ2xCLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQztTQUNqQztRQUVELCtDQUErQztRQUMvQyxJQUFJLENBQUMsT0FBTyxDQUFDLHFCQUFxQixDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFFNUQsK0NBQStDO1FBQy9DLElBQUksQ0FBQyxPQUFPLENBQUMscUJBQXFCLENBQUMsK0JBQStCLENBQUMsQ0FBQztJQUN0RSxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGtCQUFrQixDQUN4QixRQUEwQixFQUMxQixRQUFjLEVBQ2QsZUFBNkI7UUFFN0IsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUU7WUFDdkIsT0FBTztTQUNSO1FBQ0QsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFFBQVEsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUUxRCxJQUFJLENBQUMsYUFBYSxDQUFDLGVBQWUsQ0FBQyxXQUFXLFFBQVEsRUFBRSxFQUFFO1lBQ3hELE1BQU0sRUFBRSxDQUFDLGVBQWUsQ0FBQztZQUN6QixXQUFXLEVBQUUsRUFBRTtZQUNmLEtBQUssRUFBRTtnQkFDTCxJQUFJLEVBQUUsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLFdBQVcsSUFBSSxNQUFNLEVBQUU7Z0JBQzdDLEdBQUcsTUFBTSxDQUFDLFlBQVk7YUFDdkI7WUFDRCxLQUFLLEVBQUUsTUFBTSxDQUFDLGVBQWUsSUFBSSxFQUFFO1lBQ25DLEdBQUcsZUFBZTtTQUNuQixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sZ0JBQWdCLENBQUMsUUFBMEI7UUFDakQsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsV0FBVyxRQUFRLEVBQUUsRUFBRTtZQUM1RCxXQUFXLEVBQUUsVUFBVSxRQUFRLG9CQUFvQjtTQUNwRCxDQUFDLENBQUM7UUFDSCxXQUFXLENBQUMsSUFBSSxDQUFDLDJCQUEyQixRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQ3hELElBQUksQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sV0FBVyxDQUFDO0lBQ3JCLENBQUM7SUFFTyxpQkFBaUIsQ0FDdkIsTUFBd0IsRUFDeEIsUUFBYztRQUVkLHdFQUF3RTtRQUN4RSwrRkFBK0Y7UUFDL0YsOEVBQThFO1FBQzlFLE1BQU0sZUFBZSxHQUFnQixFQUFFLENBQUM7UUFFeEMsZUFBZSxDQUFDLElBQUksQ0FBQyxHQUFHLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBRXJELElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEtBQUssK0JBQWtCLENBQUMsSUFBSSxFQUFFO1lBQzNELGVBQWUsQ0FBQyxJQUFJLENBQUM7Z0JBQ25CLElBQUksRUFBRSxZQUFZO2dCQUNsQixJQUFJLEVBQUUsMEJBQTBCO2dCQUNoQyxJQUFJLEVBQUUsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUU7YUFDNUMsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxlQUFlLENBQUMsSUFBSSxDQUNsQjtZQUNFLElBQUksRUFBRSxvQkFBb0I7WUFDMUIsR0FBRyxFQUFFLE1BQU0sSUFBSSxDQUFDLGtCQUFrQixJQUFJLG1CQUFtQixFQUFFO1NBQzVELEVBQ0Q7WUFDRSxJQUFJLEVBQUUsc0JBQXNCO1lBQzVCLEdBQUcsRUFBRSxNQUFNLG1CQUFtQixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFO1NBQ25FLEVBQ0Q7WUFDRSxJQUFJLEVBQUUsVUFBVSxNQUFNLFdBQVc7WUFDakMsR0FBRyxFQUFFLE1BQU0sbUJBQW1CLGtCQUFrQixRQUFRLENBQUMsSUFBSSxFQUFFO1NBQ2hFLEVBQ0Q7WUFDRSxJQUFJLEVBQUUsV0FBVyxNQUFNLFdBQVc7WUFDbEMsR0FBRyxFQUFFLE1BQU0sbUJBQW1CLElBQUksSUFBSSxDQUFDLGtCQUFrQixJQUFJLElBQUksQ0FBQyxrQkFBa0IsRUFBRTtTQUN2RixDQUNGLENBQUM7UUFDRixPQUFPO1lBQ0wsWUFBWSxFQUFFLHVCQUFjLENBQUMsTUFBTSxDQUFDO1lBQ3BDLGVBQWU7U0FDaEIsQ0FBQztJQUNKLENBQUM7O0FBOVNILGtDQStTQzs7O0FBRUQsU0FBUyxrQkFBa0IsQ0FBQyxPQUEyQjtJQUNyRCxJQUFJLFdBQVcsR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO0lBQ3RDLElBQUksU0FBUyxHQUFHLE9BQU8sQ0FBQyxTQUFTLENBQUM7SUFDbEMsSUFBSSxPQUFPLENBQUMsYUFBYSxFQUFFO1FBQ3pCLElBQUksT0FBTyxDQUFDLFdBQVcsSUFBSSxPQUFPLENBQUMsV0FBVyxLQUFLLE9BQU8sQ0FBQyxhQUFhLEVBQUU7WUFDeEUsTUFBTSxJQUFJLEtBQUssQ0FDYixnRkFBZ0YsQ0FDakYsQ0FBQztTQUNIO1FBRUQsSUFBSSxPQUFPLENBQUMsU0FBUyxJQUFJLE9BQU8sQ0FBQyxTQUFTLEtBQUssT0FBTyxDQUFDLGFBQWEsRUFBRTtZQUNwRSxNQUFNLElBQUksS0FBSyxDQUNiLCtFQUErRSxDQUNoRixDQUFDO1NBQ0g7UUFFRCxJQUFJLFdBQVcsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxFQUFFO1lBQzNDLFdBQVcsR0FBRyxPQUFPLENBQUMsYUFBYSxDQUFDO1NBQ3JDO2FBQU0sSUFBSSxTQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsRUFBRTtZQUNoRCxTQUFTLEdBQUcsT0FBTyxDQUFDLGFBQWEsQ0FBQztTQUNuQzthQUFNO1lBQ0wsTUFBTSxJQUFJLEtBQUssQ0FDYiwyREFBMkQsT0FBTyxDQUFDLGFBQWEsRUFBRSxDQUNuRixDQUFDO1NBQ0g7S0FDRjtJQUNELE9BQU8sRUFBRSxXQUFXLEVBQUUsU0FBUyxFQUFFLENBQUM7QUFDcEMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEpzaWlQYWNtYWtUYXJnZXQsIEpTSUlfVE9PTENIQUlOIH0gZnJvbSBcIi4vY29uc3RzXCI7XG5pbXBvcnQgeyBKc2lpRG9jZ2VuIH0gZnJvbSBcIi4vanNpaS1kb2NnZW5cIjtcbmltcG9ydCB7IFRhc2sgfSBmcm9tIFwiLi5cIjtcbmltcG9ydCB7IEpvYiwgU3RlcCB9IGZyb20gXCIuLi9naXRodWIvd29ya2Zsb3dzLW1vZGVsXCI7XG5pbXBvcnQgeyBFc2xpbnQsIE5vZGVQYWNrYWdlTWFuYWdlciB9IGZyb20gXCIuLi9qYXZhc2NyaXB0XCI7XG5pbXBvcnQge1xuICBDb21tb25QdWJsaXNoT3B0aW9ucyxcbiAgR29QdWJsaXNoT3B0aW9ucyxcbiAgTWF2ZW5QdWJsaXNoT3B0aW9ucyxcbiAgTnVnZXRQdWJsaXNoT3B0aW9ucyxcbiAgUHlQaVB1Ymxpc2hPcHRpb25zLFxufSBmcm9tIFwiLi4vcmVsZWFzZVwiO1xuaW1wb3J0IHsgVHlwZVNjcmlwdFByb2plY3QsIFR5cGVTY3JpcHRQcm9qZWN0T3B0aW9ucyB9IGZyb20gXCIuLi90eXBlc2NyaXB0XCI7XG5pbXBvcnQgeyBkZWVwTWVyZ2UgfSBmcm9tIFwiLi4vdXRpbFwiO1xuXG5jb25zdCBFTUFJTF9SRUdFWCA9XG4gIC8oPzpbYS16MC05ISMkJSYnKisvPT9eX2B7fH1+LV0rKD86XFwuW2EtejAtOSEjJCUmJyorLz0/Xl9ge3x9fi1dKykqfFwiKD86W1xceDAxLVxceDA4XFx4MGJcXHgwY1xceDBlLVxceDFmXFx4MjFcXHgyMy1cXHg1YlxceDVkLVxceDdmXXxcXFxcW1xceDAxLVxceDA5XFx4MGJcXHgwY1xceDBlLVxceDdmXSkqXCIpQCg/Oig/OlthLXowLTldKD86W2EtejAtOS1dKlthLXowLTldKT9cXC4pK1thLXowLTldKD86W2EtejAtOS1dKlthLXowLTldKT98XFxbKD86KD86MjVbMC01XXwyWzAtNF1bMC05XXxbMDFdP1swLTldWzAtOV0/KVxcLil7M30oPzoyNVswLTVdfDJbMC00XVswLTldfFswMV0/WzAtOV1bMC05XT98W2EtejAtOS1dKlthLXowLTldOig/OltcXHgwMS1cXHgwOFxceDBiXFx4MGNcXHgwZS1cXHgxZlxceDIxLVxceDVhXFx4NTMtXFx4N2ZdfFxcXFxbXFx4MDEtXFx4MDlcXHgwYlxceDBjXFx4MGUtXFx4N2ZdKSspXFxdKS87XG5jb25zdCBVUkxfUkVHRVggPVxuICAvKCgoW0EtWmEtel17Myw5fTooPzpcXC9cXC8pPykoPzpbXFwtOzomPVxcK1xcJCxcXHddK0ApP1tBLVphLXowLTlcXC5cXC1dK3woPzp3d3dcXC58W1xcLTs6Jj1cXCtcXCQsXFx3XStAKVtBLVphLXowLTlcXC5cXC1dKykoKD86XFwvW1xcK34lXFwvXFwuXFx3XFwtX10qKT9cXD8/KD86W1xcLVxcKz0mOyVAXFwuXFx3X10qKSM/KD86W1xcLlxcIVxcL1xcXFxcXHddKikpPykvO1xuY29uc3QgUkVQT19URU1QX0RJUkVDVE9SWSA9IFwiLnJlcG9cIjtcblxuZXhwb3J0IGludGVyZmFjZSBKc2lpUHJvamVjdE9wdGlvbnMgZXh0ZW5kcyBUeXBlU2NyaXB0UHJvamVjdE9wdGlvbnMge1xuICAvKipcbiAgICogQGRlZmF1bHQgXCIuXCJcbiAgICovXG4gIHJlYWRvbmx5IHJvb3RkaXI/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEdpdCByZXBvc2l0b3J5IFVSTC5cbiAgICogQGRlZmF1bHQgJEdJVF9SRU1PVEVcbiAgICovXG4gIHJlYWRvbmx5IHJlcG9zaXRvcnlVcmw6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGxpYnJhcnkgYXV0aG9yLlxuICAgKiBAZGVmYXVsdCAkR0lUX1VTRVJfTkFNRVxuICAgKi9cbiAgcmVhZG9ubHkgYXV0aG9yOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEVtYWlsIG9yIFVSTCBvZiB0aGUgbGlicmFyeSBhdXRob3IuXG4gICAqIEBkZWZhdWx0ICRHSVRfVVNFUl9FTUFJTFxuICAgKi9cbiAgcmVhZG9ubHkgYXV0aG9yQWRkcmVzczogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBQdWJsaXNoIHRvIG1hdmVuXG4gICAqIEBkZWZhdWx0IC0gbm8gcHVibGlzaGluZ1xuICAgKi9cbiAgcmVhZG9ubHkgcHVibGlzaFRvTWF2ZW4/OiBKc2lpSmF2YVRhcmdldDtcblxuICAvKipcbiAgICogUHVibGlzaCB0byBweXBpXG4gICAqIEBkZWZhdWx0IC0gbm8gcHVibGlzaGluZ1xuICAgKi9cbiAgcmVhZG9ubHkgcHVibGlzaFRvUHlwaT86IEpzaWlQeXRob25UYXJnZXQ7XG5cbiAgLyoqXG4gICAqIFB1Ymxpc2ggR28gYmluZGluZ3MgdG8gYSBnaXQgcmVwb3NpdG9yeS5cbiAgICogQGRlZmF1bHQgLSBubyBwdWJsaXNoaW5nXG4gICAqL1xuICByZWFkb25seSBwdWJsaXNoVG9Hbz86IEpzaWlHb1RhcmdldDtcblxuICAvKipcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBwdWJsaXNoVG9QeVBpYFxuICAgKi9cbiAgcmVhZG9ubHkgcHl0aG9uPzogSnNpaVB5dGhvblRhcmdldDtcblxuICAvKipcbiAgICogUHVibGlzaCB0byBOdUdldFxuICAgKiBAZGVmYXVsdCAtIG5vIHB1Ymxpc2hpbmdcbiAgICovXG4gIHJlYWRvbmx5IHB1Ymxpc2hUb051Z2V0PzogSnNpaURvdE5ldFRhcmdldDtcblxuICAvKipcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBwdWJsaXNoVG9OdWdldGBcbiAgICovXG4gIHJlYWRvbmx5IGRvdG5ldD86IEpzaWlEb3ROZXRUYXJnZXQ7XG5cbiAgLyoqXG4gICAqIEF1dG9tYXRpY2FsbHkgcnVuIEFQSSBjb21wYXRpYmlsaXR5IHRlc3QgYWdhaW5zdCB0aGUgbGF0ZXN0IHZlcnNpb24gcHVibGlzaGVkIHRvIG5wbSBhZnRlciBjb21waWxhdGlvbi5cbiAgICpcbiAgICogLSBZb3UgY2FuIG1hbnVhbGx5IHJ1biBjb21wYXRpYmlsaXR5IHRlc3RzIHVzaW5nIGB5YXJuIGNvbXBhdGAgaWYgdGhpcyBmZWF0dXJlIGlzIGRpc2FibGVkLlxuICAgKiAtIFlvdSBjYW4gaWdub3JlIGNvbXBhdGliaWxpdHkgZmFpbHVyZXMgYnkgYWRkaW5nIGxpbmVzIHRvIGEgXCIuY29tcGF0aWdub3JlXCIgZmlsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGNvbXBhdD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIGlnbm9yZSBmaWxlIGZvciBBUEkgY29tcGF0aWJpbGl0eSB0ZXN0cy5cbiAgICpcbiAgICogQGRlZmF1bHQgXCIuY29tcGF0aWdub3JlXCJcbiAgICovXG4gIHJlYWRvbmx5IGNvbXBhdElnbm9yZT86IHN0cmluZztcblxuICAvKipcbiAgICogQWNjZXB0cyBhIGxpc3Qgb2YgZ2xvYiBwYXR0ZXJucy4gRmlsZXMgbWF0Y2hpbmcgYW55IG9mIHRob3NlIHBhdHRlcm5zIHdpbGwgYmUgZXhjbHVkZWQgZnJvbSB0aGUgVHlwZVNjcmlwdCBjb21waWxlciBpbnB1dC5cbiAgICpcbiAgICogQnkgZGVmYXVsdCwganNpaSB3aWxsIGluY2x1ZGUgYWxsICoudHMgZmlsZXMgKGV4Y2VwdCAuZC50cyBmaWxlcykgaW4gdGhlIFR5cGVTY3JpcHQgY29tcGlsZXIgaW5wdXQuXG4gICAqIFRoaXMgY2FuIGJlIHByb2JsZW1hdGljIGZvciBleGFtcGxlIHdoZW4gdGhlIHBhY2thZ2UncyBidWlsZCBvciB0ZXN0IHByb2NlZHVyZSBnZW5lcmF0ZXMgLnRzIGZpbGVzXG4gICAqIHRoYXQgY2Fubm90IGJlIGNvbXBpbGVkIHdpdGgganNpaSdzIGNvbXBpbGVyIHNldHRpbmdzLlxuICAgKi9cbiAgcmVhZG9ubHkgZXhjbHVkZVR5cGVzY3JpcHQ/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogRmlsZSBwYXRoIGZvciBnZW5lcmF0ZWQgZG9jcy5cbiAgICogQGRlZmF1bHQgXCJBUEkubWRcIlxuICAgKi9cbiAgcmVhZG9ubHkgZG9jZ2VuRmlsZVBhdGg/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEVtaXQgYSBjb21wcmVzc2VkIHZlcnNpb24gb2YgdGhlIGFzc2VtYmx5XG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBjb21wcmVzc0Fzc2VtYmx5PzogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGVudW0gU3RhYmlsaXR5IHtcbiAgRVhQRVJJTUVOVEFMID0gXCJleHBlcmltZW50YWxcIixcbiAgU1RBQkxFID0gXCJzdGFibGVcIixcbiAgREVQUkVDQVRFRCA9IFwiZGVwcmVjYXRlZFwiLFxufVxuXG5leHBvcnQgaW50ZXJmYWNlIEpzaWlKYXZhVGFyZ2V0IGV4dGVuZHMgTWF2ZW5QdWJsaXNoT3B0aW9ucyB7XG4gIHJlYWRvbmx5IGphdmFQYWNrYWdlOiBzdHJpbmc7XG4gIHJlYWRvbmx5IG1hdmVuR3JvdXBJZDogc3RyaW5nO1xuICByZWFkb25seSBtYXZlbkFydGlmYWN0SWQ6IHN0cmluZztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBKc2lpUHl0aG9uVGFyZ2V0IGV4dGVuZHMgUHlQaVB1Ymxpc2hPcHRpb25zIHtcbiAgcmVhZG9ubHkgZGlzdE5hbWU6IHN0cmluZztcbiAgcmVhZG9ubHkgbW9kdWxlOiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgSnNpaURvdE5ldFRhcmdldCBleHRlbmRzIE51Z2V0UHVibGlzaE9wdGlvbnMge1xuICByZWFkb25seSBkb3ROZXROYW1lc3BhY2U6IHN0cmluZztcbiAgcmVhZG9ubHkgcGFja2FnZUlkOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGljb25Vcmw/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogR28gdGFyZ2V0IGNvbmZpZ3VyYXRpb25cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBKc2lpR29UYXJnZXQgZXh0ZW5kcyBHb1B1Ymxpc2hPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSB0YXJnZXQgZ28gbW9kdWxlLlxuICAgKlxuICAgKiBAZXhhbXBsZSBnaXRodWIuY29tL293bmVyL3JlcG9cbiAgICogQGV4YW1wbGUgZ2l0aHViLmNvbS9vd25lci9yZXBvL3N1YmRpclxuICAgKi9cbiAgcmVhZG9ubHkgbW9kdWxlTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgZ28gcGFja2FnZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZXJpdmVkIGZyb20gdGhlIG1vZHVsZSBuYW1lXG4gICAqL1xuICByZWFkb25seSBwYWNrYWdlTmFtZT86IHN0cmluZztcbn1cblxuLyoqXG4gKiBNdWx0aS1sYW5ndWFnZSBqc2lpIGxpYnJhcnkgcHJvamVjdFxuICpcbiAqIEBwamlkIGpzaWlcbiAqL1xuZXhwb3J0IGNsYXNzIEpzaWlQcm9qZWN0IGV4dGVuZHMgVHlwZVNjcmlwdFByb2plY3Qge1xuICBwdWJsaWMgcmVhZG9ubHkgZXNsaW50PzogRXNsaW50O1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgcGFja2FnZUFsbFRhc2s6IFRhc2s7XG5cbiAgY29uc3RydWN0b3Iob3B0aW9uczogSnNpaVByb2plY3RPcHRpb25zKSB7XG4gICAgY29uc3QgeyBhdXRob3JFbWFpbCwgYXV0aG9yVXJsIH0gPSBwYXJzZUF1dGhvckFkZHJlc3Mob3B0aW9ucyk7XG5cbiAgICBjb25zdCBkZWZhdWx0T3B0aW9ucyA9IHtcbiAgICAgIHJlcG9zaXRvcnk6IG9wdGlvbnMucmVwb3NpdG9yeVVybCxcbiAgICAgIGF1dGhvck5hbWU6IG9wdGlvbnMuYXV0aG9yLFxuICAgICAgYXV0aG9yRW1haWwsXG4gICAgICBhdXRob3JVcmwsXG4gICAgICBqZXN0T3B0aW9uczoge1xuICAgICAgICBqZXN0VmVyc2lvbjogXCJeMjdcIixcbiAgICAgIH0sXG4gICAgfTtcblxuICAgIGNvbnN0IGZvcmNlZE9wdGlvbnMgPSB7XG4gICAgICByZWxlYXNlVG9OcG06IGZhbHNlLCAvLyB3ZSBoYXZlIGEganNpaSByZWxlYXNlIHdvcmtmbG93XG4gICAgICBkaXNhYmxlVHNjb25maWc6IHRydWUsIC8vIGpzaWkgZ2VuZXJhdGVzIGl0cyBvd24gdHNjb25maWcuanNvblxuICAgICAgZG9jZ2VuOiBmYWxzZSwgLy8gd2UgdXNlIGpzaWktZG9jZ2VuIGhlcmUgc28gZGlzYWJsZSB0eXBlc2NyaXB0IGRvY2dlblxuICAgIH07XG5cbiAgICBjb25zdCBtZXJnZWRPcHRpb25zID0gZGVlcE1lcmdlKFtcbiAgICAgIGRlZmF1bHRPcHRpb25zLFxuICAgICAgb3B0aW9ucyxcbiAgICAgIGZvcmNlZE9wdGlvbnMsXG4gICAgXSkgYXMgVHlwZVNjcmlwdFByb2plY3RPcHRpb25zO1xuXG4gICAgc3VwZXIobWVyZ2VkT3B0aW9ucyk7XG5cbiAgICBjb25zdCBzcmNkaXIgPSB0aGlzLnNyY2RpcjtcbiAgICBjb25zdCBsaWJkaXIgPSB0aGlzLmxpYmRpcjtcblxuICAgIHRoaXMuYWRkRmllbGRzKHsgdHlwZXM6IGAke2xpYmRpcn0vaW5kZXguZC50c2AgfSk7XG5cbiAgICBjb25zdCBjb21wcmVzc0Fzc2VtYmx5ID0gb3B0aW9ucy5jb21wcmVzc0Fzc2VtYmx5ID8/IGZhbHNlO1xuXG4gICAgLy8gdGhpcyBpcyBhbiB1bmhlbHBmdWwgd2FybmluZ1xuICAgIGNvbnN0IGpzaWlGbGFncyA9IFtcIi0tc2lsZW5jZS13YXJuaW5ncz1yZXNlcnZlZC13b3JkXCJdO1xuICAgIGlmIChjb21wcmVzc0Fzc2VtYmx5KSB7XG4gICAgICBqc2lpRmxhZ3MucHVzaChcIi0tY29tcHJlc3MtYXNzZW1ibHlcIik7XG4gICAgfVxuXG4gICAgY29uc3QgY29tcGF0SWdub3JlID0gb3B0aW9ucy5jb21wYXRJZ25vcmUgPz8gXCIuY29tcGF0aWdub3JlXCI7XG5cbiAgICB0aGlzLmFkZEZpZWxkcyh7IHN0YWJpbGl0eTogb3B0aW9ucy5zdGFiaWxpdHkgPz8gU3RhYmlsaXR5LlNUQUJMRSB9KTtcblxuICAgIGlmIChvcHRpb25zLnN0YWJpbGl0eSA9PT0gU3RhYmlsaXR5LkRFUFJFQ0FURUQpIHtcbiAgICAgIHRoaXMuYWRkRmllbGRzKHsgZGVwcmVjYXRlZDogdHJ1ZSB9KTtcbiAgICB9XG5cbiAgICBjb25zdCBjb21wYXRUYXNrID0gdGhpcy5hZGRUYXNrKFwiY29tcGF0XCIsIHtcbiAgICAgIGRlc2NyaXB0aW9uOiBcIlBlcmZvcm0gQVBJIGNvbXBhdGliaWxpdHkgY2hlY2sgYWdhaW5zdCBsYXRlc3QgdmVyc2lvblwiLFxuICAgICAgZXhlYzogYGpzaWktZGlmZiBucG06JChub2RlIC1wIFwicmVxdWlyZShcXCcuL3BhY2thZ2UuanNvblxcJykubmFtZVwiKSAtayAtLWlnbm9yZS1maWxlICR7Y29tcGF0SWdub3JlfSB8fCAoZWNobyBcIlxcblVORVhQRUNURUQgQlJFQUtJTkcgQ0hBTkdFUzogYWRkIGtleXMgc3VjaCBhcyBcXCdyZW1vdmVkOmNvbnN0cnVjdHMuTm9kZS5vZlxcJyB0byAke2NvbXBhdElnbm9yZX0gdG8gc2tpcC5cXG5cIiAmJiBleGl0IDEpYCxcbiAgICB9KTtcblxuICAgIGNvbnN0IGNvbXBhdCA9IG9wdGlvbnMuY29tcGF0ID8/IGZhbHNlO1xuICAgIGlmIChjb21wYXQpIHtcbiAgICAgIHRoaXMuY29tcGlsZVRhc2suc3Bhd24oY29tcGF0VGFzayk7XG4gICAgfVxuXG4gICAgdGhpcy5jb21waWxlVGFzay5yZXNldChbXCJqc2lpXCIsIC4uLmpzaWlGbGFnc10uam9pbihcIiBcIikpO1xuICAgIHRoaXMud2F0Y2hUYXNrLnJlc2V0KFtcImpzaWlcIiwgXCItd1wiLCAuLi5qc2lpRmxhZ3NdLmpvaW4oXCIgXCIpKTtcbiAgICB0aGlzLnBhY2thZ2VBbGxUYXNrID0gdGhpcy5hZGRUYXNrKFwicGFja2FnZS1hbGxcIiwge1xuICAgICAgZGVzY3JpcHRpb246IFwiUGFja2FnZXMgYXJ0aWZhY3RzIGZvciBhbGwgdGFyZ2V0IGxhbmd1YWdlc1wiLFxuICAgIH0pO1xuXG4gICAgLy8gaW4ganNpaSB3ZSBjb25zaWRlciB0aGUgZW50aXJlIHJlcG8gKHBvc3QgYnVpbGQpIGFzIHRoZSBidWlsZCBhcnRpZmFjdFxuICAgIC8vIHdoaWNoIGlzIHRoZW4gdXNlZCB0byBjcmVhdGUgdGhlIGxhbmd1YWdlIGJpbmRpbmdzIGluIHNlcGFyYXRlIGpvYnMuXG4gICAgY29uc3QgcHJlcGFyZVJlcG9Gb3JDSSA9IFtcbiAgICAgIGByc3luYyAtYSAuIC5yZXBvIC0tZXhjbHVkZSAuZ2l0IC0tZXhjbHVkZSBub2RlX21vZHVsZXNgLFxuICAgICAgYHJtIC1yZiAke3RoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5fWAsXG4gICAgICBgbXYgLnJlcG8gJHt0aGlzLmFydGlmYWN0c0RpcmVjdG9yeX1gLFxuICAgIF0uam9pbihcIiAmJiBcIik7XG5cbiAgICAvLyB3aGVuIHJ1bm5pbmcgaW5zaWRlIENJIHdlIGp1c3QgcHJlcGFyZSB0aGUgcmVwbyBmb3IgcGFja2FnaW5nLCB3aGljaFxuICAgIC8vIHRha2VzIHBsYWNlIGluIHNlcGFyYXRlIHRhc2tzLlxuICAgIC8vIG91dHNpZGUgb2YgQ0kgKGkuZSBsb2NhbGx5KSB3ZSBzaW1wbHkgcGFja2FnZSBhbGwgdGFyZ2V0cy5cbiAgICB0aGlzLnBhY2thZ2VUYXNrLnJlc2V0KFxuICAgICAgYGlmIFsgISAteiBcXCR7Q0l9IF07IHRoZW4gJHtwcmVwYXJlUmVwb0ZvckNJfTsgZWxzZSAke3RoaXMucnVuVGFza0NvbW1hbmQoXG4gICAgICAgIHRoaXMucGFja2FnZUFsbFRhc2tcbiAgICAgICl9OyBmaWBcbiAgICApO1xuXG4gICAgY29uc3QgdGFyZ2V0czogUmVjb3JkPHN0cmluZywgYW55PiA9IHt9O1xuXG4gICAgY29uc3QganNpaTogYW55ID0ge1xuICAgICAgb3V0ZGlyOiB0aGlzLmFydGlmYWN0c0RpcmVjdG9yeSxcbiAgICAgIHRhcmdldHMsXG4gICAgICB0c2M6IHtcbiAgICAgICAgb3V0RGlyOiBsaWJkaXIsXG4gICAgICAgIHJvb3REaXI6IHNyY2RpcixcbiAgICAgIH0sXG4gICAgfTtcblxuICAgIGlmIChvcHRpb25zLmV4Y2x1ZGVUeXBlc2NyaXB0KSB7XG4gICAgICBqc2lpLmV4Y2x1ZGVUeXBlc2NyaXB0ID0gb3B0aW9ucy5leGNsdWRlVHlwZXNjcmlwdDtcbiAgICB9XG5cbiAgICB0aGlzLmFkZEZpZWxkcyh7IGpzaWkgfSk7XG5cbiAgICB0aGlzLnJlbGVhc2U/LnB1Ymxpc2hlci5hZGRHaXRIdWJQcmVQdWJsaXNoaW5nU3RlcHMoXG4gICAgICB7XG4gICAgICAgIG5hbWU6IFwiUHJlcGFyZSBSZXBvc2l0b3J5XCIsXG4gICAgICAgIHJ1bjogYG12ICR7dGhpcy5hcnRpZmFjdHNEaXJlY3Rvcnl9ICR7UkVQT19URU1QX0RJUkVDVE9SWX1gLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgbmFtZTogXCJDb2xsZWN0IEdpdEh1YiBNZXRhZGF0YVwiLFxuICAgICAgICBydW46IGBtdiAke1JFUE9fVEVNUF9ESVJFQ1RPUll9LyR7dGhpcy5hcnRpZmFjdHNEaXJlY3Rvcnl9ICR7dGhpcy5hcnRpZmFjdHNEaXJlY3Rvcnl9YCxcbiAgICAgIH1cbiAgICApO1xuXG4gICAgY29uc3QgZXh0cmFKb2JPcHRpb25zOiBQYXJ0aWFsPEpvYj4gPSBvcHRpb25zLndvcmtmbG93UnVuc09uXG4gICAgICA/IHsgcnVuc09uOiBvcHRpb25zLndvcmtmbG93UnVuc09uIH1cbiAgICAgIDoge307XG5cbiAgICBpZiAob3B0aW9ucy5yZWxlYXNlVG9OcG0gIT0gZmFsc2UpIHtcbiAgICAgIGNvbnN0IHRhc2sgPSB0aGlzLmFkZFBhY2thZ2luZ1Rhc2soXCJqc1wiKTtcbiAgICAgIHRoaXMucmVsZWFzZT8ucHVibGlzaGVyLnB1Ymxpc2hUb05wbSh7XG4gICAgICAgIC4uLnRoaXMucGFjbWFrRm9yTGFuZ3VhZ2UoXCJqc1wiLCB0YXNrKSxcbiAgICAgICAgcmVnaXN0cnk6IHRoaXMucGFja2FnZS5ucG1SZWdpc3RyeSxcbiAgICAgICAgbnBtVG9rZW5TZWNyZXQ6IHRoaXMucGFja2FnZS5ucG1Ub2tlblNlY3JldCxcbiAgICAgICAgY29kZUFydGlmYWN0T3B0aW9uczogb3B0aW9ucy5jb2RlQXJ0aWZhY3RPcHRpb25zLFxuICAgICAgfSk7XG4gICAgICB0aGlzLmFkZFBhY2thZ2luZ1RhcmdldChcImpzXCIsIHRhc2ssIGV4dHJhSm9iT3B0aW9ucyk7XG4gICAgfVxuXG4gICAgLy8gd2UgY2Fubm90IGNhbGwgYW4gb3B0aW9uIGBqYXZhYCBiZWNhdXNlIHRoZSBqYXZhIGNvZGUgZ2VuZXJhdGVkIGJ5IGpzaWlcbiAgICAvLyBkb2VzIG5vdCBjb21waWxlIGR1ZSB0byBhIGNvbmZsaWN0IGJldHdlZW4gdGhpcyBvcHRpb24gbmFtZSBhbmQgdGhlIGBqYXZhYFxuICAgIC8vIHBhY2thZ2UgKGUuZy4gd2hlbiBgamF2YS51dGlsLk9iamVjdHNgIGlzIHJlZmVyZW5jZWQpLlxuICAgIGlmIChcImphdmFcIiBpbiBvcHRpb25zKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3RoZSBcImphdmFcIiBvcHRpb24gaXMgbm93IGNhbGxlZCBcInB1Ymxpc2hUb01hdmVuXCInKTtcbiAgICB9XG5cbiAgICBpZiAob3B0aW9ucy5wdWJsaXNoVG9NYXZlbikge1xuICAgICAgdGFyZ2V0cy5qYXZhID0ge1xuICAgICAgICBwYWNrYWdlOiBvcHRpb25zLnB1Ymxpc2hUb01hdmVuLmphdmFQYWNrYWdlLFxuICAgICAgICBtYXZlbjoge1xuICAgICAgICAgIGdyb3VwSWQ6IG9wdGlvbnMucHVibGlzaFRvTWF2ZW4ubWF2ZW5Hcm91cElkLFxuICAgICAgICAgIGFydGlmYWN0SWQ6IG9wdGlvbnMucHVibGlzaFRvTWF2ZW4ubWF2ZW5BcnRpZmFjdElkLFxuICAgICAgICB9LFxuICAgICAgfTtcblxuICAgICAgY29uc3QgdGFzayA9IHRoaXMuYWRkUGFja2FnaW5nVGFzayhcImphdmFcIik7XG5cbiAgICAgIHRoaXMucmVsZWFzZT8ucHVibGlzaGVyLnB1Ymxpc2hUb01hdmVuKHtcbiAgICAgICAgLi4udGhpcy5wYWNtYWtGb3JMYW5ndWFnZShcImphdmFcIiwgdGFzayksXG4gICAgICAgIC4uLm9wdGlvbnMucHVibGlzaFRvTWF2ZW4sXG4gICAgICB9KTtcblxuICAgICAgdGhpcy5hZGRQYWNrYWdpbmdUYXJnZXQoXCJqYXZhXCIsIHRhc2ssIGV4dHJhSm9iT3B0aW9ucyk7XG4gICAgfVxuXG4gICAgY29uc3QgcHlwaSA9IG9wdGlvbnMucHVibGlzaFRvUHlwaSA/PyBvcHRpb25zLnB5dGhvbjtcbiAgICBpZiAocHlwaSkge1xuICAgICAgdGFyZ2V0cy5weXRob24gPSB7XG4gICAgICAgIGRpc3ROYW1lOiBweXBpLmRpc3ROYW1lLFxuICAgICAgICBtb2R1bGU6IHB5cGkubW9kdWxlLFxuICAgICAgfTtcblxuICAgICAgY29uc3QgdGFzayA9IHRoaXMuYWRkUGFja2FnaW5nVGFzayhcInB5dGhvblwiKTtcbiAgICAgIHRoaXMucmVsZWFzZT8ucHVibGlzaGVyLnB1Ymxpc2hUb1B5UGkoe1xuICAgICAgICAuLi50aGlzLnBhY21ha0Zvckxhbmd1YWdlKFwicHl0aG9uXCIsIHRhc2spLFxuICAgICAgICAuLi5weXBpLFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMuYWRkUGFja2FnaW5nVGFyZ2V0KFwicHl0aG9uXCIsIHRhc2ssIGV4dHJhSm9iT3B0aW9ucyk7XG4gICAgfVxuXG4gICAgY29uc3QgbnVnZXQgPSBvcHRpb25zLnB1Ymxpc2hUb051Z2V0ID8/IG9wdGlvbnMuZG90bmV0O1xuICAgIGlmIChudWdldCkge1xuICAgICAgdGFyZ2V0cy5kb3RuZXQgPSB7XG4gICAgICAgIG5hbWVzcGFjZTogbnVnZXQuZG90TmV0TmFtZXNwYWNlLFxuICAgICAgICBwYWNrYWdlSWQ6IG51Z2V0LnBhY2thZ2VJZCxcbiAgICAgICAgaWNvblVybDogbnVnZXQuaWNvblVybCxcbiAgICAgIH07XG5cbiAgICAgIGNvbnN0IHRhc2sgPSB0aGlzLmFkZFBhY2thZ2luZ1Rhc2soXCJkb3RuZXRcIik7XG4gICAgICB0aGlzLnJlbGVhc2U/LnB1Ymxpc2hlci5wdWJsaXNoVG9OdWdldCh7XG4gICAgICAgIC4uLnRoaXMucGFjbWFrRm9yTGFuZ3VhZ2UoXCJkb3RuZXRcIiwgdGFzayksXG4gICAgICAgIC4uLm51Z2V0LFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMuYWRkUGFja2FnaW5nVGFyZ2V0KFwiZG90bmV0XCIsIHRhc2ssIGV4dHJhSm9iT3B0aW9ucyk7XG4gICAgfVxuXG4gICAgY29uc3QgZ29sYW5nID0gb3B0aW9ucy5wdWJsaXNoVG9HbztcbiAgICBpZiAoZ29sYW5nKSB7XG4gICAgICB0YXJnZXRzLmdvID0ge1xuICAgICAgICBtb2R1bGVOYW1lOiBnb2xhbmcubW9kdWxlTmFtZSxcbiAgICAgICAgcGFja2FnZU5hbWU6IGdvbGFuZy5wYWNrYWdlTmFtZSxcbiAgICAgIH07XG5cbiAgICAgIGNvbnN0IHRhc2sgPSB0aGlzLmFkZFBhY2thZ2luZ1Rhc2soXCJnb1wiKTtcbiAgICAgIHRoaXMucmVsZWFzZT8ucHVibGlzaGVyLnB1Ymxpc2hUb0dvKHtcbiAgICAgICAgLi4udGhpcy5wYWNtYWtGb3JMYW5ndWFnZShcImdvXCIsIHRhc2spLFxuICAgICAgICAuLi5nb2xhbmcsXG4gICAgICB9KTtcblxuICAgICAgdGhpcy5hZGRQYWNrYWdpbmdUYXJnZXQoXCJnb1wiLCB0YXNrLCBleHRyYUpvYk9wdGlvbnMpO1xuICAgIH1cblxuICAgIHRoaXMuYWRkRGV2RGVwcyhcImpzaWlcIiwgXCJqc2lpLWRpZmZcIiwgXCJqc2lpLXBhY21ha1wiKTtcblxuICAgIHRoaXMuZ2l0aWdub3JlLmV4Y2x1ZGUoXCIuanNpaVwiLCBcInRzY29uZmlnLmpzb25cIik7XG4gICAgdGhpcy5ucG1pZ25vcmU/LmluY2x1ZGUoXCIuanNpaVwiKTtcblxuICAgIGlmIChvcHRpb25zLmRvY2dlbiA/PyB0cnVlKSB7XG4gICAgICBuZXcgSnNpaURvY2dlbih0aGlzLCB7IGZpbGVQYXRoOiBvcHRpb25zLmRvY2dlbkZpbGVQYXRoIH0pO1xuICAgIH1cblxuICAgIC8vIGpzaWkgdXBkYXRlcyAubnBtaWdub3JlLCBzbyB3ZSBtYWtlIGl0IHdyaXRhYmxlXG4gICAgaWYgKHRoaXMubnBtaWdub3JlKSB7XG4gICAgICB0aGlzLm5wbWlnbm9yZS5yZWFkb25seSA9IGZhbHNlO1xuICAgIH1cblxuICAgIC8vIGh0dHBzOi8vZ2l0aHViLmNvbS9wcm9qZW4vcHJvamVuL2lzc3Vlcy8yMTY1XG4gICAgdGhpcy5wYWNrYWdlLmFkZFBhY2thZ2VSZXNvbHV0aW9ucyhcIkB0eXBlcy9wcmV0dGllckAyLjYuMFwiKTtcblxuICAgIC8vIGh0dHBzOi8vZ2l0aHViLmNvbS9wcm9qZW4vcHJvamVuL2lzc3Vlcy8yMjY0XG4gICAgdGhpcy5wYWNrYWdlLmFkZFBhY2thZ2VSZXNvbHV0aW9ucyhcIkB0eXBlcy9iYWJlbF9fdHJhdmVyc2VANy4xOC4yXCIpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSB0YXJnZXQgbGFuZ3VhZ2UgdG8gdGhlIGJ1aWxkIHdvcmtmbG93IGFuZCBjcmVhdGVzIGEgcGFja2FnZSB0YXNrLlxuICAgKiBAcGFyYW0gbGFuZ3VhZ2VcbiAgICogQHJldHVybnNcbiAgICovXG4gIHByaXZhdGUgYWRkUGFja2FnaW5nVGFyZ2V0KFxuICAgIGxhbmd1YWdlOiBKc2lpUGFjbWFrVGFyZ2V0LFxuICAgIHBhY2tUYXNrOiBUYXNrLFxuICAgIGV4dHJhSm9iT3B0aW9uczogUGFydGlhbDxKb2I+XG4gICkge1xuICAgIGlmICghdGhpcy5idWlsZFdvcmtmbG93KSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIGNvbnN0IHBhY21hayA9IHRoaXMucGFjbWFrRm9yTGFuZ3VhZ2UobGFuZ3VhZ2UsIHBhY2tUYXNrKTtcblxuICAgIHRoaXMuYnVpbGRXb3JrZmxvdy5hZGRQb3N0QnVpbGRKb2IoYHBhY2thZ2UtJHtsYW5ndWFnZX1gLCB7XG4gICAgICBydW5zT246IFtcInVidW50dS1sYXRlc3RcIl0sXG4gICAgICBwZXJtaXNzaW9uczoge30sXG4gICAgICB0b29sczoge1xuICAgICAgICBub2RlOiB7IHZlcnNpb246IHRoaXMubm9kZVZlcnNpb24gPz8gXCIxNC54XCIgfSxcbiAgICAgICAgLi4ucGFjbWFrLnB1Ymxpc2hUb29scyxcbiAgICAgIH0sXG4gICAgICBzdGVwczogcGFjbWFrLnByZVB1Ymxpc2hTdGVwcyA/PyBbXSxcbiAgICAgIC4uLmV4dHJhSm9iT3B0aW9ucyxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkUGFja2FnaW5nVGFzayhsYW5ndWFnZTogSnNpaVBhY21ha1RhcmdldCk6IFRhc2sge1xuICAgIGNvbnN0IHBhY2thZ2VUYXNrID0gdGhpcy50YXNrcy5hZGRUYXNrKGBwYWNrYWdlOiR7bGFuZ3VhZ2V9YCwge1xuICAgICAgZGVzY3JpcHRpb246IGBDcmVhdGUgJHtsYW5ndWFnZX0gbGFuZ3VhZ2UgYmluZGluZ3NgLFxuICAgIH0pO1xuICAgIHBhY2thZ2VUYXNrLmV4ZWMoYGpzaWktcGFjbWFrIC12IC0tdGFyZ2V0ICR7bGFuZ3VhZ2V9YCk7XG4gICAgdGhpcy5wYWNrYWdlQWxsVGFzay5zcGF3bihwYWNrYWdlVGFzayk7XG4gICAgcmV0dXJuIHBhY2thZ2VUYXNrO1xuICB9XG5cbiAgcHJpdmF0ZSBwYWNtYWtGb3JMYW5ndWFnZShcbiAgICB0YXJnZXQ6IEpzaWlQYWNtYWtUYXJnZXQsXG4gICAgcGFja1Rhc2s6IFRhc2tcbiAgKTogQ29tbW9uUHVibGlzaE9wdGlvbnMge1xuICAgIC8vIGF0IHRoaXMgc3RhZ2UsIGBhcnRpZmFjdHNEaXJlY3RvcnlgIGNvbnRhaW5zIHRoZSBwcmVidWlsdCByZXBvc2l0b3J5LlxuICAgIC8vIGZvciB0aGUgcHVibGlzaGluZyB0byB3b3JrIHNlYW1sZXNzZWx5LCB0aGF0IGRpcmVjdG9yeSBuZWVkcyB0byBjb250YWluIHRoZSBhY3R1YWwgYXJ0aWZhY3QuXG4gICAgLy8gc28gd2UgbW92ZSB0aGUgcmVwbywgY3JlYXRlIHRoZSBhcnRpZmFjdCwgYW5kIHB1dCBpdCBpbiB0aGUgZXhwZWN0ZWQgcGxhY2UuXG4gICAgY29uc3QgcHJlUHVibGlzaFN0ZXBzOiBBcnJheTxTdGVwPiA9IFtdO1xuXG4gICAgcHJlUHVibGlzaFN0ZXBzLnB1c2goLi4udGhpcy53b3JrZmxvd0Jvb3RzdHJhcFN0ZXBzKTtcblxuICAgIGlmICh0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIgPT09IE5vZGVQYWNrYWdlTWFuYWdlci5QTlBNKSB7XG4gICAgICBwcmVQdWJsaXNoU3RlcHMucHVzaCh7XG4gICAgICAgIG5hbWU6IFwiU2V0dXAgcG5wbVwiLFxuICAgICAgICB1c2VzOiBcInBucG0vYWN0aW9uLXNldHVwQHYyLjIuNFwiLFxuICAgICAgICB3aXRoOiB7IHZlcnNpb246IHRoaXMucGFja2FnZS5wbnBtVmVyc2lvbiB9LFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgcHJlUHVibGlzaFN0ZXBzLnB1c2goXG4gICAgICB7XG4gICAgICAgIG5hbWU6IFwiUHJlcGFyZSBSZXBvc2l0b3J5XCIsXG4gICAgICAgIHJ1bjogYG12ICR7dGhpcy5hcnRpZmFjdHNEaXJlY3Rvcnl9ICR7UkVQT19URU1QX0RJUkVDVE9SWX1gLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgbmFtZTogXCJJbnN0YWxsIERlcGVuZGVuY2llc1wiLFxuICAgICAgICBydW46IGBjZCAke1JFUE9fVEVNUF9ESVJFQ1RPUll9ICYmICR7dGhpcy5wYWNrYWdlLmluc3RhbGxDb21tYW5kfWAsXG4gICAgICB9LFxuICAgICAge1xuICAgICAgICBuYW1lOiBgQ3JlYXRlICR7dGFyZ2V0fSBhcnRpZmFjdGAsXG4gICAgICAgIHJ1bjogYGNkICR7UkVQT19URU1QX0RJUkVDVE9SWX0gJiYgbnB4IHByb2plbiAke3BhY2tUYXNrLm5hbWV9YCxcbiAgICAgIH0sXG4gICAgICB7XG4gICAgICAgIG5hbWU6IGBDb2xsZWN0ICR7dGFyZ2V0fSBBcnRpZmFjdGAsXG4gICAgICAgIHJ1bjogYG12ICR7UkVQT19URU1QX0RJUkVDVE9SWX0vJHt0aGlzLmFydGlmYWN0c0RpcmVjdG9yeX0gJHt0aGlzLmFydGlmYWN0c0RpcmVjdG9yeX1gLFxuICAgICAgfVxuICAgICk7XG4gICAgcmV0dXJuIHtcbiAgICAgIHB1Ymxpc2hUb29sczogSlNJSV9UT09MQ0hBSU5bdGFyZ2V0XSxcbiAgICAgIHByZVB1Ymxpc2hTdGVwcyxcbiAgICB9O1xuICB9XG59XG5cbmZ1bmN0aW9uIHBhcnNlQXV0aG9yQWRkcmVzcyhvcHRpb25zOiBKc2lpUHJvamVjdE9wdGlvbnMpIHtcbiAgbGV0IGF1dGhvckVtYWlsID0gb3B0aW9ucy5hdXRob3JFbWFpbDtcbiAgbGV0IGF1dGhvclVybCA9IG9wdGlvbnMuYXV0aG9yVXJsO1xuICBpZiAob3B0aW9ucy5hdXRob3JBZGRyZXNzKSB7XG4gICAgaWYgKG9wdGlvbnMuYXV0aG9yRW1haWwgJiYgb3B0aW9ucy5hdXRob3JFbWFpbCAhPT0gb3B0aW9ucy5hdXRob3JBZGRyZXNzKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIFwiYXV0aG9yRW1haWwgaXMgZGVwcmVjYXRlZCBhbmQgY2Fubm90IGJlIHVzZWQgaW4gY29uanVuY3Rpb24gd2l0aCBhdXRob3JBZGRyZXNzXCJcbiAgICAgICk7XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMuYXV0aG9yVXJsICYmIG9wdGlvbnMuYXV0aG9yVXJsICE9PSBvcHRpb25zLmF1dGhvckFkZHJlc3MpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgXCJhdXRob3JVcmwgaXMgZGVwcmVjYXRlZCBhbmQgY2Fubm90IGJlIHVzZWQgaW4gY29uanVuY3Rpb24gd2l0aCBhdXRob3JBZGRyZXNzLlwiXG4gICAgICApO1xuICAgIH1cblxuICAgIGlmIChFTUFJTF9SRUdFWC50ZXN0KG9wdGlvbnMuYXV0aG9yQWRkcmVzcykpIHtcbiAgICAgIGF1dGhvckVtYWlsID0gb3B0aW9ucy5hdXRob3JBZGRyZXNzO1xuICAgIH0gZWxzZSBpZiAoVVJMX1JFR0VYLnRlc3Qob3B0aW9ucy5hdXRob3JBZGRyZXNzKSkge1xuICAgICAgYXV0aG9yVXJsID0gb3B0aW9ucy5hdXRob3JBZGRyZXNzO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIGBhdXRob3JBZGRyZXNzIG11c3QgYmUgZWl0aGVyIGFuIGVtYWlsIGFkZHJlc3Mgb3IgYSBVUkw6ICR7b3B0aW9ucy5hdXRob3JBZGRyZXNzfWBcbiAgICAgICk7XG4gICAgfVxuICB9XG4gIHJldHVybiB7IGF1dGhvckVtYWlsLCBhdXRob3JVcmwgfTtcbn1cbiJdfQ==