"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const nag_rules_1 = require("../../nag-rules");
/**
 * Security Groups do not allow for unrestricted SSH traffic
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ec2_1.CfnSecurityGroup) {
        const ingressRules = aws_cdk_lib_1.Stack.of(node).resolve(node.securityGroupIngress);
        if (ingressRules != undefined) {
            //For each ingress rule, ensure that it does not allow unrestricted SSH traffic.
            for (const rule of ingressRules) {
                const resolvedRule = aws_cdk_lib_1.Stack.of(node).resolve(rule);
                const ipProtocol = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedRule.ipProtocol);
                const cidrIp = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedRule.cidrIp);
                const cidrIpv6 = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedRule.cidrIpv6);
                const fromPort = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedRule.fromPort);
                const toPort = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedRule.toPort);
                if ((cidrIp != undefined && cidrIp.includes('/0')) ||
                    (cidrIpv6 != undefined && cidrIpv6.includes('/0'))) {
                    if (fromPort != undefined && toPort != undefined) {
                        if ((fromPort <= 22 && toPort >= 22) ||
                            fromPort == -1 ||
                            toPort == -1 ||
                            ipProtocol == '-1') {
                            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                        }
                    }
                    else {
                        if (fromPort == 22 || ipProtocol == '-1') {
                            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                        }
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_ec2_1.CfnSecurityGroupIngress) {
        const ipProtocol = nag_rules_1.NagRules.resolveIfPrimitive(node, node.ipProtocol);
        const cidrIp = nag_rules_1.NagRules.resolveIfPrimitive(node, node.cidrIp);
        const cidrIpv6 = nag_rules_1.NagRules.resolveIfPrimitive(node, node.cidrIpv6);
        const fromPort = nag_rules_1.NagRules.resolveIfPrimitive(node, node.fromPort);
        const toPort = nag_rules_1.NagRules.resolveIfPrimitive(node, node.toPort);
        if ((cidrIp != undefined && cidrIp.includes('/0')) ||
            (cidrIpv6 != undefined && cidrIpv6.includes('/0'))) {
            //Is a port range specified?
            if (fromPort != undefined && toPort != undefined) {
                if ((fromPort <= 22 && toPort >= 22) ||
                    fromPort == -1 ||
                    toPort == -1 ||
                    ipProtocol == '-1') {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
            else {
                if (fromPort == 22 || ipProtocol == '-1') {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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