"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SizeRoundingBehavior = exports.Size = void 0;
const constructs_1 = require("constructs");
/**
 * Represents the amount of digital storage.
 *
 * The amount can be specified either as a literal value (e.g: `10`) which
 * cannot be negative, or as an unresolved number token.
 *
 * When the amount is passed as a token, unit conversion is not possible.
 */
class Size {
    constructor(amount, unit) {
        if (!constructs_1.Token.isUnresolved(amount) && amount < 0) {
            throw new Error(`Storage amounts cannot be negative. Received: ${amount}`);
        }
        this.amount = amount;
        this.unit = unit;
    }
    /**
     * Create a Storage representing an amount kibibytes.
     * 1 KiB = 1024 bytes
     */
    static kibibytes(amount) {
        return new Size(amount, StorageUnit.Kibibytes);
    }
    /**
     * Create a Storage representing an amount mebibytes.
     * 1 MiB = 1024 KiB
     */
    static mebibytes(amount) {
        return new Size(amount, StorageUnit.Mebibytes);
    }
    /**
     * Create a Storage representing an amount gibibytes.
     * 1 GiB = 1024 MiB
     */
    static gibibytes(amount) {
        return new Size(amount, StorageUnit.Gibibytes);
    }
    /**
     * Create a Storage representing an amount tebibytes.
     * 1 TiB = 1024 GiB
     */
    static tebibytes(amount) {
        return new Size(amount, StorageUnit.Tebibytes);
    }
    /**
     * Create a Storage representing an amount pebibytes.
     * 1 PiB = 1024 TiB
     */
    static pebibyte(amount) {
        return new Size(amount, StorageUnit.Pebibytes);
    }
    /**
     * Return this storage as a total number of kibibytes.
     */
    toKibibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Kibibytes, opts);
    }
    /**
     * Return this storage as a total number of mebibytes.
     */
    toMebibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Mebibytes, opts);
    }
    /**
     * Return this storage as a total number of gibibytes.
     */
    toGibibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Gibibytes, opts);
    }
    /**
     * Return this storage as a total number of tebibytes.
     */
    toTebibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Tebibytes, opts);
    }
    /**
     * Return this storage as a total number of pebibytes.
     */
    toPebibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Pebibytes, opts);
    }
}
exports.Size = Size;
/**
 * Rounding behaviour when converting between units of `Size`.
 */
var SizeRoundingBehavior;
(function (SizeRoundingBehavior) {
    /** Fail the conversion if the result is not an integer. */
    SizeRoundingBehavior[SizeRoundingBehavior["FAIL"] = 0] = "FAIL";
    /** If the result is not an integer, round it to the closest integer less than the result */
    SizeRoundingBehavior[SizeRoundingBehavior["FLOOR"] = 1] = "FLOOR";
    /** Don't round. Return even if the result is a fraction. */
    SizeRoundingBehavior[SizeRoundingBehavior["NONE"] = 2] = "NONE";
})(SizeRoundingBehavior = exports.SizeRoundingBehavior || (exports.SizeRoundingBehavior = {}));
class StorageUnit {
    constructor(label, inKibiBytes) {
        this.label = label;
        this.inKibiBytes = inKibiBytes;
        // MAX_SAFE_INTEGER is 2^53, so by representing storage in kibibytes,
        // the highest storage we can represent is 8 exbibytes.
    }
    toString() {
        return this.label;
    }
}
StorageUnit.Kibibytes = new StorageUnit('kibibytes', 1);
StorageUnit.Mebibytes = new StorageUnit('mebibytes', 1024);
StorageUnit.Gibibytes = new StorageUnit('gibibytes', 1024 * 1024);
StorageUnit.Tebibytes = new StorageUnit('tebibytes', 1024 * 1024 * 1024);
StorageUnit.Pebibytes = new StorageUnit('pebibytes', 1024 * 1024 * 1024 * 1024);
function convert(amount, fromUnit, toUnit, options = {}) {
    var _a;
    const rounding = (_a = options.rounding) !== null && _a !== void 0 ? _a : SizeRoundingBehavior.FAIL;
    if (fromUnit.inKibiBytes === toUnit.inKibiBytes) {
        return amount;
    }
    if (constructs_1.Token.isUnresolved(amount)) {
        throw new Error(`Unable to perform time unit conversion on un-resolved token ${amount}.`);
    }
    const multiplier = fromUnit.inKibiBytes / toUnit.inKibiBytes;
    const value = amount * multiplier;
    switch (rounding) {
        case SizeRoundingBehavior.NONE:
            return value;
        case SizeRoundingBehavior.FLOOR:
            return Math.floor(value);
        default:
        case SizeRoundingBehavior.FAIL:
            if (!Number.isInteger(value)) {
                throw new Error(`'${amount} ${fromUnit}' cannot be converted into a whole number of ${toUnit}.`);
            }
            return value;
    }
}
//# sourceMappingURL=data:application/json;base64,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