"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceSpecDefinition = exports.Protocol = exports.Service = exports.ServiceType = void 0;
const k8s = require("./imports/k8s");
const base_1 = require("./base");
const utils_1 = require("./utils");
/**
 * For some parts of your application (for example, frontends) you may want to expose a Service onto an
 * external IP address, that's outside of your cluster.
 * Kubernetes ServiceTypes allow you to specify what kind of Service you want.
 * The default is ClusterIP.
 */
var ServiceType;
(function (ServiceType) {
    /**
     * Exposes the Service on a cluster-internal IP.
     * Choosing this value makes the Service only reachable from within the cluster.
     * This is the default ServiceType
     */
    ServiceType["CLUSTER_IP"] = "ClusterIP";
    /**
     * Exposes the Service on each Node's IP at a static port (the NodePort).
     * A ClusterIP Service, to which the NodePort Service routes, is automatically created.
     * You'll be able to contact the NodePort Service, from outside the cluster,
     * by requesting <NodeIP>:<NodePort>.
     */
    ServiceType["NODE_PORT"] = "NodePort";
    /**
     * Exposes the Service externally using a cloud provider's load balancer.
     * NodePort and ClusterIP Services, to which the external load balancer routes,
     * are automatically created.
     */
    ServiceType["LOAD_BALANCER"] = "LoadBalancer";
    /**
     * Maps the Service to the contents of the externalName field (e.g. foo.bar.example.com), by returning a CNAME record with its value.
     * No proxying of any kind is set up.
     *
     * > Note: You need either kube-dns version 1.7 or CoreDNS version 0.0.8 or higher to use the ExternalName type.
     */
    ServiceType["EXTERNAL_NAME"] = "ExternalName";
})(ServiceType = exports.ServiceType || (exports.ServiceType = {}));
/**
 * An abstract way to expose an application running on a set of Pods as a network service.
 * With Kubernetes you don't need to modify your application to use an unfamiliar service discovery mechanism.
 * Kubernetes gives Pods their own IP addresses and a single DNS name for a set of Pods, and can load-balance across them.
 *
 * For example, consider a stateless image-processing backend which is running with 3 replicas. Those replicas are fungible—frontends do not care which backend they use.
 * While the actual Pods that compose the backend set may change, the frontend clients should not need to be aware of that,
 * nor should they need to keep track of the set of backends themselves.
 * The Service abstraction enables this decoupling.
 *
 * If you're able to use Kubernetes APIs for service discovery in your application, you can query the API server for Endpoints,
 * that get updated whenever the set of Pods in a Service changes. For non-native applications, Kubernetes offers ways to place a network port
 * or load balancer in between your application and the backend Pods.
 */
class Service extends base_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.spec = new ServiceSpecDefinition(props.spec);
        this.apiObject = new k8s.Service(this, 'Pod', {
            metadata: props.metadata,
            spec: utils_1.lazy(() => this.spec._toKube()),
        });
    }
}
exports.Service = Service;
var Protocol;
(function (Protocol) {
    Protocol["TCP"] = "TCP";
    Protocol["UDP"] = "UDP";
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * A description of a service.
 */
class ServiceSpecDefinition {
    constructor(props = {}) {
        var _a, _b, _c;
        this.clusterIP = props.clusterIP;
        this.externalIPs = (_a = props.externalIPs) !== null && _a !== void 0 ? _a : [];
        this.type = (_b = props.type) !== null && _b !== void 0 ? _b : ServiceType.CLUSTER_IP;
        this._ports = [];
        this._selector = {};
        for (const portAndOptions of (_c = props.ports) !== null && _c !== void 0 ? _c : []) {
            this.serve(portAndOptions.port, portAndOptions);
        }
    }
    /**
     * Returns the labels which are used to select pods for this service.
     */
    get selector() {
        return this._selector;
    }
    /**
     * Services defined using this spec will select pods according the provided label.
     *
     * @param label The label key.
     * @param value The label value.
     */
    addSelector(label, value) {
        this._selector[label] = value;
    }
    /**
     * Configure a port the service will bind to.
     * This method can be called multiple times.
     *
     * @param port The port definition.
     */
    serve(port, options = {}) {
        this._ports.push({ port, ...options });
    }
    /**
     * @internal
     */
    _toKube() {
        if (this._ports.length === 0) {
            throw new Error('A service must be configured with a port');
        }
        const ports = [];
        for (const port of this._ports) {
            ports.push({
                port: port.port,
                targetPort: port.targetPort,
            });
        }
        return {
            clusterIP: this.clusterIP,
            externalIPs: this.externalIPs,
            type: this.type,
            selector: this._selector,
            ports: ports,
        };
    }
}
exports.ServiceSpecDefinition = ServiceSpecDefinition;
//# sourceMappingURL=data:application/json;base64,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