"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PodSpecDefinition = exports.RestartPolicy = exports.Pod = void 0;
const k8s = require("./imports/k8s");
const base_1 = require("./base");
const utils_1 = require("./utils");
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends base_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.spec = new PodSpecDefinition(props.spec);
        this.apiObject = new k8s.Pod(this, 'Pod', {
            metadata: props.metadata,
            spec: utils_1.lazy(() => this.spec._toKube()),
        });
    }
}
exports.Pod = Pod;
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
/**
 * A description of a pod.
 */
class PodSpecDefinition {
    constructor(props = {}) {
        var _a, _b;
        this._containers = (_a = props.containers) !== null && _a !== void 0 ? _a : [];
        this._volumes = (_b = props.volumes) !== null && _b !== void 0 ? _b : [];
        this.restartPolicy = props.restartPolicy;
        this.serviceAccount = props.serviceAccount;
    }
    /**
     * List of containers belonging to the pod.
     *
     * @returns a copy - do not modify
     */
    get containers() {
        return [...this._containers];
    }
    /**
     * Adds a container to this pod.
     *
     * @param container The container to add
     */
    addContainer(container) {
        this._containers.push(container);
    }
    /**
     * Adds a volume to this pod.
     *
     * @param volume The volume to add
     */
    addVolume(volume) {
        this._volumes.push(volume);
    }
    /**
     * List of volumes that can be mounted by containers belonging to the pod.
     *
     * Returns a copy. To add volumes, use `addVolume()`.
     */
    get volumes() {
        return [...this._volumes];
    }
    /**
     * @internal
     */
    _toKube() {
        var _a;
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = [];
        const containers = [];
        for (const container of this.containers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of container.mounts) {
                volumes.push(mount.volume._toKube());
            }
            containers.push(container._toKube());
        }
        for (const volume of this._volumes) {
            volumes.push(volume._toKube());
        }
        return {
            restartPolicy: this.restartPolicy,
            serviceAccountName: (_a = this.serviceAccount) === null || _a === void 0 ? void 0 : _a.name,
            containers: containers,
            volumes: volumes,
        };
    }
}
exports.PodSpecDefinition = PodSpecDefinition;
//# sourceMappingURL=data:application/json;base64,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