"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentSpecDefinition = exports.Deployment = void 0;
const k8s = require("./imports/k8s");
const constructs_1 = require("constructs");
const service_1 = require("./service");
const base_1 = require("./base");
const utils_1 = require("./utils");
const pod_1 = require("./pod");
const cdk8s_1 = require("cdk8s");
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
* ---------
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
class Deployment extends base_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.spec = new DeploymentSpecDefinition(props.spec);
        this.apiObject = new k8s.Deployment(this, 'Pod', {
            metadata: props.metadata,
            spec: utils_1.lazy(() => this.spec._toKube(this)),
        });
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options - Options.
     */
    expose(options) {
        var _a;
        const containers = this.spec.podSpecTemplate.containers;
        if (containers.length === 0) {
            throw new Error('Cannot expose a deployment without containers');
        }
        // create a label and attach it to the deployment pods
        const selector = 'cdk8s.deployment';
        const matcher = constructs_1.Node.of(this).uniqueId;
        const service = new service_1.Service(this, 'Service', {
            spec: {
                type: (_a = options.serviceType) !== null && _a !== void 0 ? _a : service_1.ServiceType.CLUSTER_IP,
            },
        });
        service.spec.addSelector(selector, matcher);
        service.spec.serve(options.port, {
            // just a PoC, we assume the first container is the main one.
            // TODO: figure out what the correct thing to do here.
            targetPort: containers[0].port,
        });
        return service;
    }
}
exports.Deployment = Deployment;
/**
 * DeploymentSpec is the specification of the desired behavior of the Deployment.
 */
class DeploymentSpecDefinition {
    constructor(props = {}) {
        var _a;
        this.replicas = (_a = props.replicas) !== null && _a !== void 0 ? _a : 1;
        this.podSpecTemplate = new pod_1.PodSpecDefinition(props.podSpecTemplate);
        this.podMetadataTemplate = new cdk8s_1.ApiObjectMetadataDefinition(props.podMetadataTemplate);
        this._labelSelector = {};
    }
    /**
     * Configure a label selector to this deployment.
     * Pods that have the label will be selected by deployments configured with this spec.
     *
     * @param key - The label key.
     * @param value - The label value.
     */
    selectByLabel(key, value) {
        this._labelSelector[key] = value;
    }
    /**
     * The labels this deployment will match against in order to select pods.
     *
     * Returns a a copy. Use `selectByLabel()` to add labels.
     */
    get labelSelector() {
        return { ...this._labelSelector };
    }
    /**
     * @internal
     */
    _toKube(deployment) {
        // automatically select pods in this deployment
        const selector = 'cdk8s.deployment';
        const matcher = constructs_1.Node.of(deployment).uniqueId;
        this.podMetadataTemplate.addLabel(selector, matcher);
        this.selectByLabel(selector, matcher);
        return {
            replicas: this.replicas,
            template: {
                metadata: this.podMetadataTemplate.toJson(),
                spec: this.podSpecTemplate._toKube(),
            },
            selector: {
                matchLabels: this._labelSelector,
            },
        };
    }
}
exports.DeploymentSpecDefinition = DeploymentSpecDefinition;
//# sourceMappingURL=data:application/json;base64,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