"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountPropagation = exports.Container = exports.EnvValue = void 0;
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a by reading a specific key inside a secret.
     *
     * @param secret - The secret.
     * @param key - The key.
     * @param options - Additional options.
     */
    static fromSecret(secret, key, options = {}) {
        const source = {
            secretKeyRef: {
                name: secret.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        var _a, _b, _c;
        this.name = (_a = props.name) !== null && _a !== void 0 ? _a : 'main';
        this.image = props.image;
        this.port = props.port;
        this._command = props.command;
        this._env = (_b = props.env) !== null && _b !== void 0 ? _b : {};
        this.workingDir = props.workingDir;
        this.mounts = (_c = props.volumeMounts) !== null && _c !== void 0 ? _c : [];
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Add an environment value to the container. The variable value can come
     * from various dynamic sources such a secrets of config maps.
     *
     * @see EnvValue.fromXXX
     *
     * @param name - The variable name.
     * @param value - The variable value.
     */
    addEnv(name, value) {
        this._env[name] = value;
    }
    /**
     * The environment variables for this container.
     *
     * Returns a copy. To add environment variables use `addEnv()`.
     */
    get env() {
        return { ...this._env };
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param volume - The volume to mount.
     */
    mount(path, volume, options = {}) {
        this.mounts.push({ path, volume, ...options });
    }
    /**
     * @internal
     */
    _toKube() {
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        if (this.port) {
            ports.push({
                containerPort: this.port,
            });
        }
        return {
            name: this.name,
            image: this.image,
            ports,
            volumeMounts,
            command: this.command,
            workingDir: this.workingDir,
            env: renderEnv(this._env),
        };
    }
}
exports.Container = Container;
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
function renderEnv(env) {
    const result = new Array();
    for (const [name, v] of Object.entries(env)) {
        result.push({
            name,
            value: v.value,
            valueFrom: v.valueFrom,
        });
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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