"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomCloud9Ssm = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const lambda = require("aws-cdk-lib/aws-lambda");
const cloud9 = require("aws-cdk-lib/aws-cloud9");
const ssm = require("aws-cdk-lib/aws-ssm");
const iam = require("aws-cdk-lib/aws-iam");
const yaml = require('yaml');
const fs = require('fs');
/**
 * @experimental
 */
class CustomCloud9Ssm extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        let cloud9Env;
        let ssmAssociation;
        let customResource;
        // Create the Cloud9 environment using the default configuration
        if (!props.cloud9Ec2Props) {
            cloud9Env = new cloud9.CfnEnvironmentEC2(this, 'Cloud9Ec2Environment', { instanceType: "t3.large" });
        }
        // Create the Cloud9 environment using the received props
        else {
            cloud9Env = new cloud9.CfnEnvironmentEC2(this, 'Cloud9Ec2Environment', props.cloud9Ec2Props);
        }
        // Create a Role for the EC2 instance and an instance profile with it
        this.ec2Role = new iam.Role(this, 'Ec2Role', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            roleName: id + '-CustomCloud9SsmEc2Role',
            managedPolicies: [
                iam.ManagedPolicy.fromManagedPolicyArn(this, 'SsmManagedPolicy', 'arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore')
            ]
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'Ec2InstanceProfile', {
            roles: [this.ec2Role.roleName]
        });
        // Create the SSM Document using the default configuration
        if (!props.ssmDocumentProps) {
            let content = fs.readFileSync(CustomCloud9Ssm.DEFAULT_DOCUMENT_FILE_NAME, 'utf8');
            const ssmDocumentProps = {
                documentType: 'Command',
                content: yaml.parse(content),
                name: id + '-' + CustomCloud9Ssm.DEFAULT_DOCUMENT_NAME
            };
            this.document = new ssm.CfnDocument(this, 'SsmDocument', ssmDocumentProps);
            this.resizeEBSTo(CustomCloud9Ssm.DEFAULT_EBS_SIZE);
        }
        // Create the SSM Document using the received props
        else {
            if (!props.ssmDocumentProps.name) {
                throw new Error("The document name must be specified.");
            }
            this.document = new ssm.CfnDocument(this, 'SsmDocument', props.ssmDocumentProps);
        }
        // Add the name of the document as a tag to the EC2 instance to identify it as a target of the SSM Association
        aws_cdk_lib_1.Tags.of(cloud9Env).add("SSMConfiguration", this.document.name);
        // Create an SSM Association to apply the document configuration
        ssmAssociation = new ssm.CfnAssociation(this, 'SsmAssociation', {
            name: this.document.name,
            targets: [
                {
                    key: 'tag:SSMConfiguration',
                    values: [this.document.name]
                }
            ]
        });
        // Create the Lambda function that attaches the instance profile to the EC2 instance
        let code = fs.readFileSync(CustomCloud9Ssm.ATTACH_PROFILE_FILE_NAME, 'utf8');
        const lambdaFunction = new lambda.Function(this, 'ProfileAttachLambdaFunction', {
            runtime: lambda.Runtime.PYTHON_3_6,
            code: lambda.Code.fromInline(code),
            handler: 'index.handler',
            timeout: aws_cdk_lib_1.Duration.seconds(800)
        });
        // Give permissions to the function to execute some APIs
        lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeInstances',
                'ec2:AssociateIamInstanceProfile',
                'ec2:ReplaceIamInstanceProfileAssociation',
                'ec2:RebootInstances',
                'iam:ListInstanceProfiles',
                'iam:PassRole',
                'ssm:DescribeAssociationExecutions',
                'ssm:DescribeAssociationExecutionTargets'
            ],
            resources: ['*']
        }));
        // Create the Custom Resource that invokes the Lambda function
        customResource = new aws_cdk_lib_1.CustomResource(this, 'CustomResource', {
            serviceToken: lambdaFunction.functionArn,
            properties: {
                document_name: this.document.name,
                profile_arn: instanceProfile.attrArn,
                association_id: ssmAssociation.attrAssociationId
            }
        });
        // Add resource dependencies
        instanceProfile.node.addDependency(this.ec2Role);
        ssmAssociation.node.addDependency(cloud9Env);
        ssmAssociation.node.addDependency(this.document);
        customResource.node.addDependency(instanceProfile);
        customResource.node.addDependency(ssmAssociation);
    }
    /**
     * (experimental) Adds one or more steps to the content of the SSM Document.
     *
     * @param steps : YAML formatted string containing one or more steps to be added to the mainSteps section of the SSM Document.
     * @experimental
     */
    addDocumentSteps(steps) {
        // Add the mainSteps section if it doesn't exist
        if (!('mainSteps' in this.document.content)) {
            this.document.content['mainSteps'] = [];
        }
        // Add the new step
        this.document.content['mainSteps'].push(...yaml.parse(steps));
    }
    /**
     * (experimental) Adds one or more parameters to the content of the SSM Document.
     *
     * @param parameters : YAML formatted string containing one or more parameters to be added to the parameters section of the SSM Document.
     * @experimental
     */
    addDocumentParameters(parameters) {
        // Add the parameters section if it doesn't exist
        if (!('parameters' in this.document.content)) {
            this.document.content['parameters'] = {};
        }
        // Add the new parameter
        this.document.content['parameters'] = Object.assign({}, this.document.content['parameters'], yaml.parse(parameters));
    }
    /**
     * (experimental) Adds a step to the SSM Document content that resizes the EBS volume of the EC2 instance.
     *
     * Attaches the required policies to ec2Role.
     *
     * @param size : size in GiB to resize the EBS volume to.
     * @experimental
     */
    resizeEBSTo(size) {
        let steps = fs.readFileSync(CustomCloud9Ssm.RESIZE_STEP_FILE_NAME, 'utf8');
        steps = steps.replace('{{ size }}', String(size));
        // Add the resizing step
        this.addDocumentSteps(steps);
        // Grant permission to the EC2 instance to execute the statements included in the SSM Document
        this.ec2Role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeInstances',
                'ec2:ModifyVolume',
                'ec2:DescribeVolumesModifications'
            ],
            resources: ['*']
        }));
    }
}
exports.CustomCloud9Ssm = CustomCloud9Ssm;
_a = JSII_RTTI_SYMBOL_1;
CustomCloud9Ssm[_a] = { fqn: "@cdk-use-cases/custom-cloud9-ssm.CustomCloud9Ssm", version: "1.1.2" };
CustomCloud9Ssm.DEFAULT_EBS_SIZE = 100;
CustomCloud9Ssm.DEFAULT_DOCUMENT_FILE_NAME = `${__dirname}/assets/default_document.yml`;
CustomCloud9Ssm.RESIZE_STEP_FILE_NAME = `${__dirname}/assets/resize_ebs_step.yml`;
CustomCloud9Ssm.ATTACH_PROFILE_FILE_NAME = `${__dirname}/assets/profile_attach.py`;
CustomCloud9Ssm.DEFAULT_DOCUMENT_NAME = 'CustomCloudSsm-SsmDocument';
//# sourceMappingURL=data:application/json;base64,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