"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceFactory = void 0;
const cloudform_types_1 = require("cloudform-types");
const graphql_mapping_template_1 = require("graphql-mapping-template");
const graphql_transformer_common_1 = require("graphql-transformer-common");
const pluralize_1 = require("pluralize");
const graphql_transformer_core_1 = require("graphql-transformer-core");
const md5_1 = __importDefault(require("md5"));
class ResourceFactory {
    makeParams() {
        return {
            [graphql_transformer_common_1.ResourceConstants.PARAMETERS.DynamoDBModelTableReadIOPS]: new cloudform_types_1.NumberParameter({
                Description: 'The number of read IOPS the table should support.',
                Default: 5,
            }),
            [graphql_transformer_common_1.ResourceConstants.PARAMETERS.DynamoDBModelTableWriteIOPS]: new cloudform_types_1.NumberParameter({
                Description: 'The number of write IOPS the table should support.',
                Default: 5,
            }),
            [graphql_transformer_common_1.ResourceConstants.PARAMETERS.DynamoDBBillingMode]: new cloudform_types_1.StringParameter({
                Description: 'Configure @model types to create DynamoDB tables with PAY_PER_REQUEST or PROVISIONED billing modes.',
                Default: 'PAY_PER_REQUEST',
                AllowedValues: ['PAY_PER_REQUEST', 'PROVISIONED'],
            }),
            [graphql_transformer_common_1.ResourceConstants.PARAMETERS.DynamoDBEnablePointInTimeRecovery]: new cloudform_types_1.StringParameter({
                Description: 'Whether to enable Point in Time Recovery on the table',
                Default: 'false',
                AllowedValues: ['true', 'false'],
            }),
            [graphql_transformer_common_1.ResourceConstants.PARAMETERS.DynamoDBEnableServerSideEncryption]: new cloudform_types_1.StringParameter({
                Description: 'Enable server side encryption powered by KMS.',
                Default: 'true',
                AllowedValues: ['true', 'false'],
            }),
        };
    }
    initTemplate() {
        return {
            Parameters: this.makeParams(),
            Resources: {
                [graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID]: this.makeAppSyncAPI(),
            },
            Outputs: {
                [graphql_transformer_common_1.ResourceConstants.OUTPUTS.GraphQLAPIIdOutput]: this.makeAPIIDOutput(),
                [graphql_transformer_common_1.ResourceConstants.OUTPUTS.GraphQLAPIEndpointOutput]: this.makeAPIEndpointOutput(),
            },
            Conditions: {
                [graphql_transformer_common_1.ResourceConstants.CONDITIONS.ShouldUsePayPerRequestBilling]: cloudform_types_1.Fn.Equals(cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.DynamoDBBillingMode), 'PAY_PER_REQUEST'),
                [graphql_transformer_common_1.ResourceConstants.CONDITIONS.ShouldUsePointInTimeRecovery]: cloudform_types_1.Fn.Equals(cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.DynamoDBEnablePointInTimeRecovery), 'true'),
                [graphql_transformer_common_1.ResourceConstants.CONDITIONS.ShouldUseServerSideEncryption]: cloudform_types_1.Fn.Equals(cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.DynamoDBEnableServerSideEncryption), 'true'),
            },
        };
    }
    makeAppSyncAPI() {
        return new cloudform_types_1.AppSync.GraphQLApi({
            Name: cloudform_types_1.Fn.If(graphql_transformer_common_1.ResourceConstants.CONDITIONS.HasEnvironmentParameter, cloudform_types_1.Fn.Join('-', [cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.AppSyncApiName), cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.Env)]), cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.AppSyncApiName)),
            AuthenticationType: 'API_KEY',
        });
    }
    makeAppSyncSchema(schema) {
        return new cloudform_types_1.AppSync.GraphQLSchema({
            ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            Definition: schema,
        });
    }
    makeAPIIDOutput() {
        return {
            Description: 'Your GraphQL API ID.',
            Value: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            Export: {
                Name: cloudform_types_1.Fn.Join(':', [cloudform_types_1.Refs.StackName, 'GraphQLApiId']),
            },
        };
    }
    makeAPIEndpointOutput() {
        return {
            Description: 'Your GraphQL API endpoint.',
            Value: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'GraphQLUrl'),
            Export: {
                Name: cloudform_types_1.Fn.Join(':', [cloudform_types_1.Refs.StackName, 'GraphQLApiEndpoint']),
            },
        };
    }
    makeTableStreamArnOutput(resourceId) {
        return {
            Description: 'Your DynamoDB table StreamArn.',
            Value: cloudform_types_1.Fn.GetAtt(resourceId, 'StreamArn'),
            Export: {
                Name: cloudform_types_1.Fn.Join(':', [cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.AppSyncApiId), 'GetAtt', resourceId, 'StreamArn']),
            },
        };
    }
    makeDataSourceOutput(resourceId) {
        return {
            Description: 'Your model DataSource name.',
            Value: cloudform_types_1.Fn.GetAtt(resourceId, 'Name'),
            Export: {
                Name: cloudform_types_1.Fn.Join(':', [cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.AppSyncApiId), 'GetAtt', resourceId, 'Name']),
            },
        };
    }
    makeTableNameOutput(resourceId) {
        return {
            Description: 'Your DynamoDB table name.',
            Value: cloudform_types_1.Fn.Ref(resourceId),
            Export: {
                Name: cloudform_types_1.Fn.Join(':', [cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.AppSyncApiId), 'GetAtt', resourceId, 'Name']),
            },
        };
    }
    makeModelTable(typeName, hashKey = 'id', rangeKey, deletionPolicy = cloudform_types_1.DeletionPolicy.Delete, isSyncEnabled = false) {
        const keySchema = hashKey && rangeKey
            ? [
                {
                    AttributeName: hashKey,
                    KeyType: 'HASH',
                },
                {
                    AttributeName: rangeKey,
                    KeyType: 'RANGE',
                },
            ]
            : [{ AttributeName: hashKey, KeyType: 'HASH' }];
        const attributeDefinitions = hashKey && rangeKey
            ? [
                {
                    AttributeName: hashKey,
                    AttributeType: 'S',
                },
                {
                    AttributeName: rangeKey,
                    AttributeType: 'S',
                },
            ]
            : [{ AttributeName: hashKey, AttributeType: 'S' }];
        return new cloudform_types_1.DynamoDB.Table({
            TableName: this.dynamoDBTableName(typeName),
            KeySchema: keySchema,
            AttributeDefinitions: attributeDefinitions,
            StreamSpecification: {
                StreamViewType: 'NEW_AND_OLD_IMAGES',
            },
            BillingMode: cloudform_types_1.Fn.If(graphql_transformer_common_1.ResourceConstants.CONDITIONS.ShouldUsePayPerRequestBilling, 'PAY_PER_REQUEST', cloudform_types_1.Refs.NoValue),
            ProvisionedThroughput: cloudform_types_1.Fn.If(graphql_transformer_common_1.ResourceConstants.CONDITIONS.ShouldUsePayPerRequestBilling, cloudform_types_1.Refs.NoValue, {
                ReadCapacityUnits: cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.DynamoDBModelTableReadIOPS),
                WriteCapacityUnits: cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.DynamoDBModelTableWriteIOPS),
            }),
            SSESpecification: {
                SSEEnabled: cloudform_types_1.Fn.If(graphql_transformer_common_1.ResourceConstants.CONDITIONS.ShouldUseServerSideEncryption, true, false),
            },
            PointInTimeRecoverySpecification: cloudform_types_1.Fn.If(graphql_transformer_common_1.ResourceConstants.CONDITIONS.ShouldUsePointInTimeRecovery, {
                PointInTimeRecoveryEnabled: true,
            }, cloudform_types_1.Refs.NoValue),
            ...(isSyncEnabled && {
                TimeToLiveSpecification: graphql_transformer_core_1.SyncUtils.syncTTLConfig(),
            }),
        }).deletionPolicy(deletionPolicy);
    }
    dynamoDBTableName(typeName) {
        return cloudform_types_1.Fn.If(graphql_transformer_common_1.ResourceConstants.CONDITIONS.HasEnvironmentParameter, cloudform_types_1.Fn.Join('-', [
            typeName,
            cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.Env),
        ]), cloudform_types_1.Fn.Join('-', [typeName, cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId')]));
    }
    makeIAMRole(typeName, syncConfig) {
        return new cloudform_types_1.IAM.Role({
            RoleName: cloudform_types_1.Fn.If(graphql_transformer_common_1.ResourceConstants.CONDITIONS.HasEnvironmentParameter, cloudform_types_1.Fn.Join('-', [
                typeName.slice(0, 14) + md5_1.default(typeName).slice(15, 21),
                'role',
                cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
                cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.Env),
            ]), cloudform_types_1.Fn.Join('-', [
                typeName.slice(0, 24) + md5_1.default(typeName).slice(25, 31),
                'role',
                cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            ])),
            AssumeRolePolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Effect: 'Allow',
                        Principal: {
                            Service: 'appsync.amazonaws.com',
                        },
                        Action: 'sts:AssumeRole',
                    },
                ],
            },
            Policies: [
                new cloudform_types_1.IAM.Role.Policy({
                    PolicyName: 'DynamoDBAccess',
                    PolicyDocument: {
                        Version: '2012-10-17',
                        Statement: [
                            {
                                Effect: 'Allow',
                                Action: [
                                    'dynamodb:BatchGetItem',
                                    'dynamodb:BatchWriteItem',
                                    'dynamodb:PutItem',
                                    'dynamodb:DeleteItem',
                                    'dynamodb:GetItem',
                                    'dynamodb:Scan',
                                    'dynamodb:Query',
                                    'dynamodb:UpdateItem',
                                ],
                                Resource: [
                                    cloudform_types_1.Fn.Sub('arn:aws:dynamodb:${AWS::Region}:${AWS::AccountId}:table/${tablename}', {
                                        tablename: this.dynamoDBTableName(typeName),
                                    }),
                                    cloudform_types_1.Fn.Sub('arn:aws:dynamodb:${AWS::Region}:${AWS::AccountId}:table/${tablename}/*', {
                                        tablename: this.dynamoDBTableName(typeName),
                                    }),
                                    ...(syncConfig
                                        ? [
                                            cloudform_types_1.Fn.Sub('arn:aws:dynamodb:${AWS::Region}:${AWS::AccountId}:table/${tablename}', {
                                                tablename: cloudform_types_1.Fn.If(graphql_transformer_common_1.ResourceConstants.CONDITIONS.HasEnvironmentParameter, cloudform_types_1.Fn.Join('-', [
                                                    graphql_transformer_common_1.SyncResourceIDs.syncTableName,
                                                    cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
                                                    cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.Env),
                                                ]), cloudform_types_1.Fn.Join('-', [
                                                    graphql_transformer_common_1.SyncResourceIDs.syncTableName,
                                                    cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
                                                ])),
                                            }),
                                            cloudform_types_1.Fn.Sub('arn:aws:dynamodb:${AWS::Region}:${AWS::AccountId}:table/${tablename}/*', {
                                                tablename: cloudform_types_1.Fn.If(graphql_transformer_common_1.ResourceConstants.CONDITIONS.HasEnvironmentParameter, cloudform_types_1.Fn.Join('-', [
                                                    graphql_transformer_common_1.SyncResourceIDs.syncTableName,
                                                    cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
                                                    cloudform_types_1.Fn.Ref(graphql_transformer_common_1.ResourceConstants.PARAMETERS.Env),
                                                ]), cloudform_types_1.Fn.Join('-', [
                                                    graphql_transformer_common_1.SyncResourceIDs.syncTableName,
                                                    cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
                                                ])),
                                            }),
                                        ]
                                        : []),
                                ],
                            },
                        ],
                    },
                }),
                ...(syncConfig && graphql_transformer_core_1.SyncUtils.isLambdaSyncConfig(syncConfig)
                    ? [graphql_transformer_core_1.SyncUtils.createSyncLambdaIAMPolicy(syncConfig.LambdaConflictHandler)]
                    : []),
            ],
        });
    }
    makeDynamoDBDataSource(tableId, iamRoleLogicalID, typeName, isSyncEnabled = false) {
        return new cloudform_types_1.AppSync.DataSource({
            ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            Name: tableId,
            Type: 'AMAZON_DYNAMODB',
            ServiceRoleArn: cloudform_types_1.Fn.GetAtt(iamRoleLogicalID, 'Arn'),
            DynamoDBConfig: {
                AwsRegion: cloudform_types_1.Refs.Region,
                TableName: this.dynamoDBTableName(typeName),
                ...(isSyncEnabled && {
                    DeltaSyncConfig: graphql_transformer_core_1.SyncUtils.syncDataSourceConfig(),
                    Versioned: true,
                }),
            },
        }).dependsOn([iamRoleLogicalID]);
    }
    makeCreateResolver({ type, nameOverride, syncConfig, mutationTypeName = 'Mutation' }) {
        const fieldName = nameOverride ? nameOverride : graphql_transformer_common_1.graphqlName('create' + graphql_transformer_common_1.toUpper(type));
        const isSyncEnabled = syncConfig ? true : false;
        return new cloudform_types_1.AppSync.Resolver({
            ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            DataSourceName: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ModelResourceIDs.ModelTableDataSourceID(type), 'Name'),
            FieldName: fieldName,
            TypeName: mutationTypeName,
            RequestMappingTemplate: graphql_mapping_template_1.printBlock('Prepare DynamoDB PutItem Request')(graphql_mapping_template_1.compoundExpression([
                graphql_mapping_template_1.qref(`$context.args.input.put("__typename", "${type}")`),
                graphql_mapping_template_1.set(graphql_mapping_template_1.ref('condition'), graphql_mapping_template_1.obj({
                    expression: graphql_mapping_template_1.str('attribute_not_exists(#id)'),
                    expressionNames: graphql_mapping_template_1.obj({
                        '#id': graphql_mapping_template_1.str('id'),
                    }),
                })),
                graphql_mapping_template_1.iff(graphql_mapping_template_1.ref('context.args.condition'), graphql_mapping_template_1.compoundExpression([
                    graphql_mapping_template_1.set(graphql_mapping_template_1.ref('condition.expressionValues'), graphql_mapping_template_1.obj({})),
                    graphql_mapping_template_1.set(graphql_mapping_template_1.ref('conditionFilterExpressions'), graphql_mapping_template_1.raw('$util.parseJson($util.transform.toDynamoDBConditionExpression($context.args.condition))')),
                    graphql_mapping_template_1.qref(`$condition.put("expression", "($condition.expression) AND $conditionFilterExpressions.expression")`),
                    graphql_mapping_template_1.qref(`$condition.expressionNames.putAll($conditionFilterExpressions.expressionNames)`),
                    graphql_mapping_template_1.qref(`$condition.expressionValues.putAll($conditionFilterExpressions.expressionValues)`),
                ])),
                graphql_mapping_template_1.iff(graphql_mapping_template_1.and([graphql_mapping_template_1.ref('condition.expressionValues'), graphql_mapping_template_1.raw('$condition.expressionValues.size() == 0')]), graphql_mapping_template_1.set(graphql_mapping_template_1.ref('condition'), graphql_mapping_template_1.obj({
                    expression: graphql_mapping_template_1.ref('condition.expression'),
                    expressionNames: graphql_mapping_template_1.ref('condition.expressionNames'),
                }))),
                graphql_mapping_template_1.DynamoDBMappingTemplate.putItem({
                    key: graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey), graphql_mapping_template_1.raw(`$util.toJson(\$${graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey})`), graphql_mapping_template_1.obj({
                        id: graphql_mapping_template_1.raw(`$util.dynamodb.toDynamoDBJson($ctx.args.input.id)`),
                    }), true),
                    attributeValues: graphql_mapping_template_1.ref('util.dynamodb.toMapValuesJson($context.args.input)'),
                    condition: graphql_mapping_template_1.ref('util.toJson($condition)'),
                }),
            ])),
            ResponseMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.DynamoDBMappingTemplate.dynamoDBResponse(isSyncEnabled)),
            ...(syncConfig && { SyncConfig: graphql_transformer_core_1.SyncUtils.syncResolverConfig(syncConfig) }),
        });
    }
    initalizeDefaultInputForCreateMutation(input, timestamps) {
        var _a;
        const hasDefaultIdField = (_a = input.fields) === null || _a === void 0 ? void 0 : _a.find(field => field.name.value === 'id' && ['ID', 'String'].includes(graphql_transformer_common_1.getBaseType(field.type)));
        return graphql_mapping_template_1.printBlock('Set default values')(graphql_mapping_template_1.compoundExpression([
            ...(hasDefaultIdField ? [graphql_mapping_template_1.qref(`$context.args.input.put("id", $util.defaultIfNull($ctx.args.input.id, $util.autoId()))`)] : []),
            ...(timestamps && (timestamps.createdAtField || timestamps.updatedAtField)
                ? [graphql_mapping_template_1.set(graphql_mapping_template_1.ref('createdAt'), graphql_mapping_template_1.ref('util.time.nowISO8601()'))]
                : []),
            ...(timestamps && timestamps.createdAtField
                ? [
                    graphql_mapping_template_1.comment(`Automatically set the createdAt timestamp.`),
                    graphql_mapping_template_1.qref(`$context.args.input.put("${timestamps.createdAtField}", $util.defaultIfNull($ctx.args.input.${timestamps.createdAtField}, $createdAt))`),
                ]
                : []),
            ...(timestamps && timestamps.updatedAtField
                ? [
                    graphql_mapping_template_1.comment(`Automatically set the updatedAt timestamp.`),
                    graphql_mapping_template_1.qref(`$context.args.input.put("${timestamps.updatedAtField}", $util.defaultIfNull($ctx.args.input.${timestamps.updatedAtField}, $createdAt))`),
                ]
                : []),
        ]));
    }
    makeUpdateResolver({ type, nameOverride, syncConfig, mutationTypeName = 'Mutation', timestamps }) {
        const fieldName = nameOverride ? nameOverride : graphql_transformer_common_1.graphqlName(`update` + graphql_transformer_common_1.toUpper(type));
        const isSyncEnabled = syncConfig ? true : false;
        return new cloudform_types_1.AppSync.Resolver({
            ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            DataSourceName: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ModelResourceIDs.ModelTableDataSourceID(type), 'Name'),
            FieldName: fieldName,
            TypeName: mutationTypeName,
            RequestMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.compoundExpression([
                graphql_mapping_template_1.ifElse(graphql_mapping_template_1.raw(`$${graphql_transformer_common_1.ResourceConstants.SNIPPETS.AuthCondition} && $${graphql_transformer_common_1.ResourceConstants.SNIPPETS.AuthCondition}.expression != ""`), graphql_mapping_template_1.compoundExpression([
                    graphql_mapping_template_1.set(graphql_mapping_template_1.ref('condition'), graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.AuthCondition)),
                    graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey), graphql_mapping_template_1.forEach(graphql_mapping_template_1.ref('entry'), graphql_mapping_template_1.ref(`${graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey}.entrySet()`), [
                        graphql_mapping_template_1.qref('$condition.put("expression", "$condition.expression AND attribute_exists(#keyCondition$velocityCount)")'),
                        graphql_mapping_template_1.qref('$condition.expressionNames.put("#keyCondition$velocityCount", "$entry.key")'),
                    ]), graphql_mapping_template_1.compoundExpression([
                        graphql_mapping_template_1.qref('$condition.put("expression", "$condition.expression AND attribute_exists(#id)")'),
                        graphql_mapping_template_1.qref('$condition.expressionNames.put("#id", "id")'),
                    ])),
                ]), graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey), graphql_mapping_template_1.compoundExpression([
                    graphql_mapping_template_1.set(graphql_mapping_template_1.ref('condition'), graphql_mapping_template_1.obj({
                        expression: graphql_mapping_template_1.str(''),
                        expressionNames: graphql_mapping_template_1.obj({}),
                        expressionValues: graphql_mapping_template_1.obj({}),
                    })),
                    graphql_mapping_template_1.forEach(graphql_mapping_template_1.ref('entry'), graphql_mapping_template_1.ref(`${graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey}.entrySet()`), [
                        graphql_mapping_template_1.ifElse(graphql_mapping_template_1.raw('$velocityCount == 1'), graphql_mapping_template_1.qref('$condition.put("expression", "attribute_exists(#keyCondition$velocityCount)")'), graphql_mapping_template_1.qref('$condition.put(\
"expression", "$condition.expression AND attribute_exists(#keyCondition$velocityCount)")')),
                        graphql_mapping_template_1.qref('$condition.expressionNames.put("#keyCondition$velocityCount", "$entry.key")'),
                    ]),
                ]), graphql_mapping_template_1.set(graphql_mapping_template_1.ref('condition'), graphql_mapping_template_1.obj({
                    expression: graphql_mapping_template_1.str('attribute_exists(#id)'),
                    expressionNames: graphql_mapping_template_1.obj({
                        '#id': graphql_mapping_template_1.str('id'),
                    }),
                    expressionValues: graphql_mapping_template_1.obj({}),
                })))),
                ...(timestamps && timestamps.updatedAtField
                    ? [
                        graphql_mapping_template_1.comment(`Automatically set the updatedAt timestamp.`),
                        graphql_mapping_template_1.qref(`$context.args.input.put("${timestamps.updatedAtField}", $util.defaultIfNull($ctx.args.input.${timestamps.updatedAtField}, $util.time.nowISO8601()))`),
                    ]
                    : []),
                graphql_mapping_template_1.qref(`$context.args.input.put("__typename", "${type}")`),
                graphql_mapping_template_1.comment('Update condition if type is @versioned'),
                graphql_mapping_template_1.iff(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.VersionedCondition), graphql_mapping_template_1.compoundExpression([
                    graphql_mapping_template_1.qref(`$condition.put("expression", "($condition.expression) AND $${graphql_transformer_common_1.ResourceConstants.SNIPPETS.VersionedCondition}.expression")`),
                    graphql_mapping_template_1.qref(`$condition.expressionNames.putAll($${graphql_transformer_common_1.ResourceConstants.SNIPPETS.VersionedCondition}.expressionNames)`),
                    graphql_mapping_template_1.qref(`$condition.expressionValues.putAll($${graphql_transformer_common_1.ResourceConstants.SNIPPETS.VersionedCondition}.expressionValues)`),
                ])),
                graphql_mapping_template_1.iff(graphql_mapping_template_1.ref('context.args.condition'), graphql_mapping_template_1.compoundExpression([
                    graphql_mapping_template_1.set(graphql_mapping_template_1.ref('conditionFilterExpressions'), graphql_mapping_template_1.raw('$util.parseJson($util.transform.toDynamoDBConditionExpression($context.args.condition))')),
                    graphql_mapping_template_1.qref(`$condition.put("expression", "($condition.expression) AND $conditionFilterExpressions.expression")`),
                    graphql_mapping_template_1.qref(`$condition.expressionNames.putAll($conditionFilterExpressions.expressionNames)`),
                    graphql_mapping_template_1.qref(`$condition.expressionValues.putAll($conditionFilterExpressions.expressionValues)`),
                ])),
                graphql_mapping_template_1.iff(graphql_mapping_template_1.and([graphql_mapping_template_1.ref('condition.expressionValues'), graphql_mapping_template_1.raw('$condition.expressionValues.size() == 0')]), graphql_mapping_template_1.set(graphql_mapping_template_1.ref('condition'), graphql_mapping_template_1.obj({
                    expression: graphql_mapping_template_1.ref('condition.expression'),
                    expressionNames: graphql_mapping_template_1.ref('condition.expressionNames'),
                }))),
                graphql_mapping_template_1.DynamoDBMappingTemplate.updateItem({
                    key: graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey), graphql_mapping_template_1.raw(`$util.toJson(\$${graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey})`), graphql_mapping_template_1.obj({
                        id: graphql_mapping_template_1.obj({ S: graphql_mapping_template_1.ref('util.toJson($context.args.input.id)') }),
                    }), true),
                    condition: graphql_mapping_template_1.ref('util.toJson($condition)'),
                    objectKeyVariable: graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey,
                    nameOverrideMap: graphql_transformer_common_1.ResourceConstants.SNIPPETS.DynamoDBNameOverrideMap,
                    isSyncEnabled,
                }),
            ])),
            ResponseMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.DynamoDBMappingTemplate.dynamoDBResponse(isSyncEnabled)),
            ...(syncConfig && { SyncConfig: graphql_transformer_core_1.SyncUtils.syncResolverConfig(syncConfig) }),
        });
    }
    makeGetResolver(type, nameOverride, isSyncEnabled = false, queryTypeName = 'Query') {
        const fieldName = nameOverride ? nameOverride : graphql_transformer_common_1.graphqlName('get' + graphql_transformer_common_1.toUpper(type));
        return new cloudform_types_1.AppSync.Resolver({
            ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            DataSourceName: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ModelResourceIDs.ModelTableDataSourceID(type), 'Name'),
            FieldName: fieldName,
            TypeName: queryTypeName,
            RequestMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.DynamoDBMappingTemplate.getItem({
                key: graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey), graphql_mapping_template_1.raw(`$util.toJson(\$${graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey})`), graphql_mapping_template_1.obj({
                    id: graphql_mapping_template_1.ref('util.dynamodb.toDynamoDBJson($ctx.args.id)'),
                }), true),
                isSyncEnabled,
            })),
            ResponseMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.DynamoDBMappingTemplate.dynamoDBResponse(isSyncEnabled)),
        });
    }
    makeSyncResolver(type, queryTypeName = 'Query') {
        const fieldName = graphql_transformer_common_1.graphqlName('sync' + graphql_transformer_common_1.toUpper(pluralize_1.plural(type)));
        return new cloudform_types_1.AppSync.Resolver({
            ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            DataSourceName: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ModelResourceIDs.ModelTableDataSourceID(type), 'Name'),
            FieldName: fieldName,
            TypeName: queryTypeName,
            RequestMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.DynamoDBMappingTemplate.syncItem({
                filter: graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref('context.args.filter'), graphql_mapping_template_1.ref('util.transform.toDynamoDBFilterExpression($ctx.args.filter)'), graphql_mapping_template_1.nul()),
                limit: graphql_mapping_template_1.ref(`util.defaultIfNull($ctx.args.limit, ${graphql_transformer_common_1.ResourceConstants.DEFAULT_SYNC_QUERY_PAGE_LIMIT})`),
                lastSync: graphql_mapping_template_1.ref('util.toJson($util.defaultIfNull($ctx.args.lastSync, null))'),
                nextToken: graphql_mapping_template_1.ref('util.toJson($util.defaultIfNull($ctx.args.nextToken, null))'),
            })),
            ResponseMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.DynamoDBMappingTemplate.dynamoDBResponse(true)),
        });
    }
    makeQueryResolver(type, nameOverride, isSyncEnabled = false, queryTypeName = 'Query') {
        const fieldName = nameOverride ? nameOverride : graphql_transformer_common_1.graphqlName(`query${graphql_transformer_common_1.toUpper(type)}`);
        return new cloudform_types_1.AppSync.Resolver({
            ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            DataSourceName: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ModelResourceIDs.ModelTableDataSourceID(type), 'Name'),
            FieldName: fieldName,
            TypeName: queryTypeName,
            RequestMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.compoundExpression([
                graphql_mapping_template_1.set(graphql_mapping_template_1.ref('limit'), graphql_mapping_template_1.ref(`util.defaultIfNull($context.args.limit, ${graphql_transformer_common_1.ResourceConstants.DEFAULT_PAGE_LIMIT})`)),
                graphql_mapping_template_1.DynamoDBMappingTemplate.query({
                    query: graphql_mapping_template_1.obj({
                        expression: graphql_mapping_template_1.str('#typename = :typename'),
                        expressionNames: graphql_mapping_template_1.obj({
                            '#typename': graphql_mapping_template_1.str('__typename'),
                        }),
                        expressionValues: graphql_mapping_template_1.obj({
                            ':typename': graphql_mapping_template_1.obj({
                                S: graphql_mapping_template_1.str(type),
                            }),
                        }),
                    }),
                    scanIndexForward: graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref('context.args.sortDirection'), graphql_mapping_template_1.ifElse(graphql_mapping_template_1.equals(graphql_mapping_template_1.ref('context.args.sortDirection'), graphql_mapping_template_1.str('ASC')), graphql_mapping_template_1.bool(true), graphql_mapping_template_1.bool(false)), graphql_mapping_template_1.bool(true)),
                    filter: graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref('context.args.filter'), graphql_mapping_template_1.ref('util.transform.toDynamoDBFilterExpression($ctx.args.filter)'), graphql_mapping_template_1.nul()),
                    limit: graphql_mapping_template_1.ref('limit'),
                    nextToken: graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref('context.args.nextToken'), graphql_mapping_template_1.ref('util.toJson($context.args.nextToken)'), graphql_mapping_template_1.nul()),
                }),
            ])),
            ResponseMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.DynamoDBMappingTemplate.dynamoDBResponse(isSyncEnabled, graphql_mapping_template_1.compoundExpression([graphql_mapping_template_1.iff(graphql_mapping_template_1.raw('!$result'), graphql_mapping_template_1.set(graphql_mapping_template_1.ref('result'), graphql_mapping_template_1.ref('ctx.result'))), graphql_mapping_template_1.raw('$util.toJson($result)')]))),
        });
    }
    makeListResolver(type, nameOverride, isSyncEnabled = false, queryTypeName = 'Query') {
        const fieldName = nameOverride ? nameOverride : graphql_transformer_common_1.graphqlName('list' + graphql_transformer_common_1.plurality(graphql_transformer_common_1.toUpper(type)));
        const requestVariable = 'ListRequest';
        return new cloudform_types_1.AppSync.Resolver({
            ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            DataSourceName: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ModelResourceIDs.ModelTableDataSourceID(type), 'Name'),
            FieldName: fieldName,
            TypeName: queryTypeName,
            RequestMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.compoundExpression([
                graphql_mapping_template_1.set(graphql_mapping_template_1.ref('limit'), graphql_mapping_template_1.ref(`util.defaultIfNull($context.args.limit, ${graphql_transformer_common_1.ResourceConstants.DEFAULT_PAGE_LIMIT})`)),
                graphql_mapping_template_1.set(graphql_mapping_template_1.ref(requestVariable), graphql_mapping_template_1.obj({
                    version: graphql_mapping_template_1.str(graphql_mapping_template_1.RESOLVER_VERSION_ID),
                    limit: graphql_mapping_template_1.ref('limit'),
                })),
                graphql_mapping_template_1.iff(graphql_mapping_template_1.ref('context.args.nextToken'), graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${requestVariable}.nextToken`), graphql_mapping_template_1.ref('context.args.nextToken'))),
                graphql_mapping_template_1.iff(graphql_mapping_template_1.ref('context.args.filter'), graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${requestVariable}.filter`), graphql_mapping_template_1.ref('util.parseJson("$util.transform.toDynamoDBFilterExpression($ctx.args.filter)")'))),
                graphql_mapping_template_1.ifElse(graphql_mapping_template_1.raw(`!$util.isNull($${graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelQueryExpression})
                        && !$util.isNullOrEmpty($${graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelQueryExpression}.expression)`), graphql_mapping_template_1.compoundExpression([
                    graphql_mapping_template_1.qref(`$${requestVariable}.put("operation", "Query")`),
                    graphql_mapping_template_1.qref(`$${requestVariable}.put("query", $${graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelQueryExpression})`),
                    graphql_mapping_template_1.ifElse(graphql_mapping_template_1.raw(`!$util.isNull($ctx.args.sortDirection) && $ctx.args.sortDirection == "DESC"`), graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${requestVariable}.scanIndexForward`), graphql_mapping_template_1.bool(false)), graphql_mapping_template_1.set(graphql_mapping_template_1.ref(`${requestVariable}.scanIndexForward`), graphql_mapping_template_1.bool(true))),
                ]), graphql_mapping_template_1.qref(`$${requestVariable}.put("operation", "Scan")`)),
                graphql_mapping_template_1.raw(`$util.toJson($${requestVariable})`),
            ])),
            ResponseMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.DynamoDBMappingTemplate.dynamoDBResponse(isSyncEnabled)),
        });
    }
    makeDeleteResolver({ type, nameOverride, syncConfig, mutationTypeName = 'Mutation' }) {
        const fieldName = nameOverride ? nameOverride : graphql_transformer_common_1.graphqlName('delete' + graphql_transformer_common_1.toUpper(type));
        const isSyncEnabled = syncConfig ? true : false;
        return new cloudform_types_1.AppSync.Resolver({
            ApiId: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, 'ApiId'),
            DataSourceName: cloudform_types_1.Fn.GetAtt(graphql_transformer_common_1.ModelResourceIDs.ModelTableDataSourceID(type), 'Name'),
            FieldName: fieldName,
            TypeName: mutationTypeName,
            RequestMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.compoundExpression([
                graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.AuthCondition), graphql_mapping_template_1.compoundExpression([
                    graphql_mapping_template_1.set(graphql_mapping_template_1.ref('condition'), graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.AuthCondition)),
                    graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey), graphql_mapping_template_1.forEach(graphql_mapping_template_1.ref('entry'), graphql_mapping_template_1.ref(`${graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey}.entrySet()`), [
                        graphql_mapping_template_1.qref('$condition.put("expression", "$condition.expression AND attribute_exists(#keyCondition$velocityCount)")'),
                        graphql_mapping_template_1.qref('$condition.expressionNames.put("#keyCondition$velocityCount", "$entry.key")'),
                    ]), graphql_mapping_template_1.compoundExpression([
                        graphql_mapping_template_1.qref('$condition.put("expression", "$condition.expression AND attribute_exists(#id)")'),
                        graphql_mapping_template_1.qref('$condition.expressionNames.put("#id", "id")'),
                    ])),
                ]), graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey), graphql_mapping_template_1.compoundExpression([
                    graphql_mapping_template_1.set(graphql_mapping_template_1.ref('condition'), graphql_mapping_template_1.obj({
                        expression: graphql_mapping_template_1.str(''),
                        expressionNames: graphql_mapping_template_1.obj({}),
                    })),
                    graphql_mapping_template_1.forEach(graphql_mapping_template_1.ref('entry'), graphql_mapping_template_1.ref(`${graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey}.entrySet()`), [
                        graphql_mapping_template_1.ifElse(graphql_mapping_template_1.raw('$velocityCount == 1'), graphql_mapping_template_1.qref('$condition.put("expression", "attribute_exists(#keyCondition$velocityCount)")'), graphql_mapping_template_1.qref('$condition.put(\
"expression", "$condition.expression AND attribute_exists(#keyCondition$velocityCount)")')),
                        graphql_mapping_template_1.qref('$condition.expressionNames.put("#keyCondition$velocityCount", "$entry.key")'),
                    ]),
                ]), graphql_mapping_template_1.set(graphql_mapping_template_1.ref('condition'), graphql_mapping_template_1.obj({
                    expression: graphql_mapping_template_1.str('attribute_exists(#id)'),
                    expressionNames: graphql_mapping_template_1.obj({
                        '#id': graphql_mapping_template_1.str('id'),
                    }),
                })))),
                graphql_mapping_template_1.iff(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.VersionedCondition), graphql_mapping_template_1.compoundExpression([
                    graphql_mapping_template_1.qref(`$condition.put("expression", "($condition.expression) AND $${graphql_transformer_common_1.ResourceConstants.SNIPPETS.VersionedCondition}.expression")`),
                    graphql_mapping_template_1.qref(`$condition.expressionNames.putAll($${graphql_transformer_common_1.ResourceConstants.SNIPPETS.VersionedCondition}.expressionNames)`),
                    graphql_mapping_template_1.set(graphql_mapping_template_1.ref('expressionValues'), graphql_mapping_template_1.raw('$util.defaultIfNull($condition.expressionValues, {})')),
                    graphql_mapping_template_1.qref(`$expressionValues.putAll($${graphql_transformer_common_1.ResourceConstants.SNIPPETS.VersionedCondition}.expressionValues)`),
                    graphql_mapping_template_1.set(graphql_mapping_template_1.ref('condition.expressionValues'), graphql_mapping_template_1.ref('expressionValues')),
                ])),
                graphql_mapping_template_1.iff(graphql_mapping_template_1.ref('context.args.condition'), graphql_mapping_template_1.compoundExpression([
                    graphql_mapping_template_1.set(graphql_mapping_template_1.ref('conditionFilterExpressions'), graphql_mapping_template_1.raw('$util.parseJson($util.transform.toDynamoDBConditionExpression($context.args.condition))')),
                    graphql_mapping_template_1.qref(`$condition.put("expression", "($condition.expression) AND $conditionFilterExpressions.expression")`),
                    graphql_mapping_template_1.qref(`$condition.expressionNames.putAll($conditionFilterExpressions.expressionNames)`),
                    graphql_mapping_template_1.set(graphql_mapping_template_1.ref('conditionExpressionValues'), graphql_mapping_template_1.raw('$util.defaultIfNull($condition.expressionValues, {})')),
                    graphql_mapping_template_1.qref(`$conditionExpressionValues.putAll($conditionFilterExpressions.expressionValues)`),
                    graphql_mapping_template_1.set(graphql_mapping_template_1.ref('condition.expressionValues'), graphql_mapping_template_1.ref('conditionExpressionValues')),
                    graphql_mapping_template_1.qref(`$condition.expressionValues.putAll($conditionFilterExpressions.expressionValues)`),
                ])),
                graphql_mapping_template_1.iff(graphql_mapping_template_1.and([graphql_mapping_template_1.ref('condition.expressionValues'), graphql_mapping_template_1.raw('$condition.expressionValues.size() == 0')]), graphql_mapping_template_1.set(graphql_mapping_template_1.ref('condition'), graphql_mapping_template_1.obj({
                    expression: graphql_mapping_template_1.ref('condition.expression'),
                    expressionNames: graphql_mapping_template_1.ref('condition.expressionNames'),
                }))),
                graphql_mapping_template_1.DynamoDBMappingTemplate.deleteItem({
                    key: graphql_mapping_template_1.ifElse(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey), graphql_mapping_template_1.raw(`$util.toJson(\$${graphql_transformer_common_1.ResourceConstants.SNIPPETS.ModelObjectKey})`), graphql_mapping_template_1.obj({
                        id: graphql_mapping_template_1.ref('util.dynamodb.toDynamoDBJson($ctx.args.input.id)'),
                    }), true),
                    condition: graphql_mapping_template_1.ref('util.toJson($condition)'),
                    isSyncEnabled,
                }),
            ])),
            ResponseMappingTemplate: graphql_mapping_template_1.print(graphql_mapping_template_1.DynamoDBMappingTemplate.dynamoDBResponse(isSyncEnabled)),
            ...(syncConfig && { SyncConfig: graphql_transformer_core_1.SyncUtils.syncResolverConfig(syncConfig) }),
        });
    }
}
exports.ResourceFactory = ResourceFactory;
//# sourceMappingURL=resources.js.map